# -*- coding: UTF-8 -*-
import os
import re
import traceback
from common import cmdManager
from common import checkDevice
from common.constant import CheckedResult
from common.contextUtil import getLang, getLogger, getSshObj, getSftpObj, getPackMgr, getTimeStamp,getExtraDataDict
from common.cmdRetManager import getCliRet
from common.compressionManager import getCollectTmpDir, deleteRemoteFile, deleteRemoteFiles
from common.cTV1R1 import cHandleCliRet
from common.sysInfoManager import isDoubleCtrlsModel


def execute(dataDict):
    '''
    @summary: the entrance of main method, this check item is used to collect system log
    @param dataDict: the dictionary of data which provided by tool framework
    @return: (pass status, CLI information, error message) as (boolean, string, string)
    '''
    flag = CheckedResult.PASS
    cliRet = ''
    errMsg = ''
    INTERVAL_TIME = 30 #心跳间隔
    COLLECT_CMD_TIMEOUT = 25 * 60 #一键收集日志命令超时时间（25分钟）
    deCompressDestDir = ''
    isDoubleCtrlLostOne = False #
    
    fileHandle = getPackMgr(dataDict)
    log = getLogger(dataDict)
    lang = getLang(dataDict)
    
    #检查是否存在T4卡不允许信息收集
    (flg, cliRet, errMsg) = checkDevice.checkLimit4InfModule(dataDict)
    if not flg:
        flag = CheckedResult.NOCHECK
        getExtraDataDict(dataDict)['ret4Datacollect'] = (flag, cliRet, errMsg, deCompressDestDir, isDoubleCtrlLostOne)
        return (flag, cliRet, errMsg)

    #删除远端日志文件
    deleteRemoteFiles(dataDict, "/OSM/export_import/", "^Datacollect_(.*?)\.tgz$")

    #本地保存一键解压日志的文件夹名称
    localResultDirName = 'Datacollect'

    #获取数据解压临时文件夹
    dataCollectTmpDir = getCollectTmpDir(dataDict)
    log.info('collect temp file dir is:' + dataCollectTmpDir)
    
    #步骤1：需要执行一键收集日志（设置超时时间25分钟）
    cliRet = cmdManager.execCmdWithKeepAlive(dataDict, 'datacollect', INTERVAL_TIME, COLLECT_CMD_TIMEOUT)
    if re.search('The system is busy', cliRet, re.IGNORECASE):
        flag = CheckedResult.NOTPASS
        if lang == 'zh':
            errMsg = u'未收集到日志，请等待5分钟后重试，如果重试仍然失败请联系技术支持工程师处理。'
        else:
            errMsg = 'No log is collected. Please wait for 5 minutes and try again. If the retry still fails, contact technical support engineers.'
        getExtraDataDict(dataDict)['ret4Datacollect'] = (flag, cliRet, errMsg, deCompressDestDir, isDoubleCtrlLostOne)
        return (flag, cliRet, errMsg)
    
    #命令执行失败
    if not re.search('Path :', cliRet, re.IGNORECASE):
        flag = CheckedResult.NOTPASS
        if lang == 'zh':
            errMsg = u'收集系统日志失败。'
        else:
            errMsg = 'Failed to collect system log.'
        getExtraDataDict(dataDict)['ret4Datacollect'] = (flag, cliRet, errMsg, deCompressDestDir, isDoubleCtrlLostOne)
        return (flag, cliRet, errMsg)
        
    #步骤2：获取远程文件路径
    remotePath = ''
    remoteFileName = ''
    lineList = cliRet.splitlines()
    
    #获取远端阵列上的临时文件全路径
    for line in lineList:
        if line.find('Path :') != -1:
            lineSub = line[line.index('Path :'):]
            remotePath = lineSub.replace('Path :', '')
            break
    if remotePath == '':
        flag = CheckedResult.NOTPASS
        if lang == 'zh':
            errMsg = u'获取日志收集文件路径失败。'
        else:
            errMsg = 'Failed to obtain the path for system log files.'
        getExtraDataDict(dataDict)['ret4Datacollect'] = (flag, cliRet, errMsg, deCompressDestDir, isDoubleCtrlLostOne)
        return (flag, cliRet, errMsg)
    
    #步骤3：通过SFTP下载文件
    #获取远端文件名
    remoteFileName = ''
    remoteFileName = remotePath.split('/')[-1]
    #通过SFTP下载文件
    sftp = getSftpObj(dataDict)
    remoteFileDir = remotePath.replace(remoteFileName, '')
    
    #创建临时文件夹
    if not os.path.exists(dataCollectTmpDir):
        os.makedirs(dataCollectTmpDir)
    
    #指定文件名称为Datacollect（避免文件名过长）
    localFileName = localResultDirName + '.tgz'
    localFile = dataCollectTmpDir + os.sep + localFileName
    
    #通过sftp下载文件
    try:
        sftp.getFile(remotePath, localFile, None)
    except:
        #打印错误信息
        log.error('catch except when download log file over!')
        log.error('except trace back:' + unicode(traceback.format_exc()))

        flag = CheckedResult.WARN
        if lang == 'zh':
            errMsg = u'通过sftp下载文件失败。'
        else:
            errMsg = 'Downloading the file by sftp failed.'
        getExtraDataDict(dataDict)['ret4Datacollect'] = (flag, cliRet, errMsg, deCompressDestDir, isDoubleCtrlLostOne)
        return (flag, cliRet, errMsg)
    finally:
        #删除设备上临时文件
        deleteRemoteFile(dataDict, remotePath)
    
    #步骤4：解压本地日志文件
    deCompressCmds = ['OceanspaceS5000T', '']
    deCompressDestDir = fileHandle.deCompressPackage(localFile, deCompressCmds)
    log.info('Decompress datacollect file directory is: ' + deCompressDestDir)
    
    #获取结果返回
    if not (deCompressDestDir and os.path.exists(deCompressDestDir) and os.path.isdir(deCompressDestDir)):
        flag = CheckedResult.NOTPASS
        if lang == 'zh':
            errMsg = u'解压日志包失败。'
        else:
            errMsg = 'Failed to decompress the system log package.'
        getExtraDataDict(dataDict)['ret4Datacollect'] = (flag, cliRet, errMsg, deCompressDestDir, isDoubleCtrlLostOne)
        return (flag, cliRet, errMsg)
    
    #判断是否为部分收集成功
    if re.search('Part of', cliRet, re.IGNORECASE):
        flag = CheckedResult.NOTPASS
        isDoubleCtrls = isDoubleCtrlsModel(dataDict)#判断单双控
        ctrlNum = getCtrlNumFromSysLog(deCompressDestDir)#从系统日志文件夹中获取收集的控制器个数
        #双控只收集到1个控制器的信息
        if isDoubleCtrls and ctrlNum != 2:
            isDoubleCtrlLostOne = True
            if lang == 'zh':
                errMsg = u'只收集到单个控制器日志。'
            else:
                errMsg = 'Only the logs of a single controller are collected.'
        else:
            if lang == 'zh':
                errMsg = u'控制器部分信息收集失败。'
            else:
                errMsg = 'Failed to collect some information about controllers.'
    
    #存入解压目标文件夹，供后续涉及信息收集的检查项使用
    getExtraDataDict(dataDict)['ret4Datacollect'] = (flag, cliRet, errMsg, deCompressDestDir, isDoubleCtrlLostOne)
    
    #存入主被控和控制器ID(A、B控)映射关系
    getExtraDataDict(dataDict)['mainOrSlaveAndCtrlIdMap'] = getMainOrSlaveAndCtrlIdMap(dataDict)
    
    return (flag, cliRet, errMsg)


def getCtrlNumFromSysLog(deCompressDestDir):
    '''
    @summary: 从系统日志文件夹中获取收集的控制器个数
    @param deCompressDestDir: 日志收集结果文件夹
    @return: 收集的控制器个数
    '''
    ctrlNum = 0
    for temp in os.listdir(deCompressDestDir):
        ipMatch = "^[\d]{1,3}\.[\d]{1,3}\.[\d]{1,3}\."
        if os.path.isdir(os.path.join(deCompressDestDir, temp)):
            if re.match(ipMatch, temp):
                ctrlNum += 1
    return ctrlNum


def getMainOrSlaveAndCtrlIdMap(dataDict):
    '''
    @summary: 从系统日志文件夹中获取收集的控制器个数
    @param dataDict: 上下文环境数据字典
    @return: 收集的控制器个数
    '''
    cliRet = getCliRet(dataDict, 'showcontroller')
    formatFunction = cHandleCliRet(cliRet)
    ctrlDictList = formatFunction.handle()
    tmpDict = {}
    for ctrlDict in ctrlDictList:
        ctrlId = ctrlDict.get('Controller Id')
        mainOrSlave = ctrlDict.get('Primary/Secondary Status')
        tmpDict[mainOrSlave] = ctrlId
    return tmpDict
