# -*- coding: UTF-8 -*-

import os
import re
from common.sysInfoManager import getCurHotPatchVersion,getCurRealDeviceType
from common.constant import CheckedResult
from common.cmdRetManager import getCliRet, checkCliInfoValid
from common.contextUtil import getLang, getLogger, getCurSysVer, getCliRet4UpgradePkgInfo


#需要处理的设备类型
G_8619ChipNeedCheckDeviceType = \
(
   "S3900-M200", "S3900-M300",
   "S5900-M100", "S5900-M200",
   "S6900-M100"
)

#系统版本+需安装热补丁+需卸载热补丁映射关系：
G_8619ChipVersionListMap = \
{   #系统软件版本:           (#需要安装热补丁,         #安装前需要卸载的热补丁)
    "V100R002C00SPC008": ("V100R002C00SPH801", ""             ),
    "V100R002C00SPC009": ("V100R002C00SPH901", ""             ),
    "V100R002C00SPC011": ("V100R002C00SPHb05", "V100R002C00SPHb04"  ),
    "V100R002C00SPC012": ("V100R002C00SPHc05", "V100R002C00SPHc04"  ),
    "V100R002C00SPC013": ("V100R002C00SPHd02", "V100R002C00SPHd01"  ),
    "V100R002C00SPC015": ("V100R002C00SPHf03", "V100R002C00SPHf02"  ),
    "V100R002C00SPCg00": ("V100R002C00SPHg03", "V100R002C00SPHg02"  ),
    "V100R002C00SPCh00": ("V100R002C00SPHh03", "V100R002C00SPHh01"  )
}

#需要安装热补丁
INSTALL_HOTPATCH_INDEX = 0
#安装前需要卸载的热补丁
UNINSTALL_HOTPATCH_INDEX = 1

# **************************************************************************** #
# 函数名称: check8619ChipHotpatch
# 功能说明: 检查8619芯片补丁安装情况
# 输入参数: DeviceType, CurrentVer, curHotPatchVer
# 输出参数: 无
# **************************************************************************** # 
def check8619ChipHotpatch(DeviceType, CurrentVer, curHotPatchVer, lang):

    errMsg = ''
    global G_8619ChipNeedCheckDeviceType, G_8619ChipVersionListMap

    #不需要处理的型号，直接通过
    if DeviceType not in G_8619ChipNeedCheckDeviceType:
       return (True, errMsg)
   
    #不需要检查的版本，直接通过
    if not G_8619ChipVersionListMap.has_key(CurrentVer):
        return (True, errMsg)

    #检查是否需要卸载、安装热补丁动作
    versionTuple = G_8619ChipVersionListMap.get(CurrentVer)
    needInstallHotPatch = versionTuple[INSTALL_HOTPATCH_INDEX]
    needRollbackHotPatch = versionTuple[UNINSTALL_HOTPATCH_INDEX]
    #已安装指定热补丁
    if curHotPatchVer >= needInstallHotPatch:
        return (True, errMsg)
    #需要卸载指定的补丁
    elif needRollbackHotPatch and curHotPatchVer == needRollbackHotPatch:
        if "zh" == lang:
            errMsg += u"请在升级前先卸载热补丁%s，然后安装%s或者更高版本的热补丁。" % (curHotPatchVer, needInstallHotPatch)
        else:
            errMsg += "Please rollback the hotpatch %s firstly,then install the hotpatch %s \
or a later version before performing the upgrade." % (curHotPatchVer, needInstallHotPatch)
    #不需要卸载，直接安装指定补丁
    else:
        if "zh" == lang:
            errMsg += u"请在升级前安装%s或者更高版本的热补丁。" % needInstallHotPatch
        else:
            errMsg += "Please install the hotpatch %s or a later version before performing the upgrade." % needInstallHotPatch  

    #检查结果返回
    return (False, errMsg)

# **************************************************************************** #
# 函数名称: processCheckItems
# 功能说明: 处理所有检查原子的结果
# 输入参数: checkItemResultList
# 输出参数: 无
# **************************************************************************** # 
def processCheckItems(checkItemResultList):
    
    checkFlag = True
    allErrMsg = ""
    
    for index in len(checkItemResultList):
        checkResult = checkItemResultList[index]
        flag = checkResult.get("Flag")
        errMsg = checkResult.get("errMsg")        
        if not flag:
            checkFlag = False
            allErrMsg += errMsg + "\n\n"
    
    return (checkFlag, allErrMsg)


def execute(dataDict):
    '''
    @summary: the entrance of main method, this check item is used to check hot patch before upgrade
    @param dataDict: the dictionary of data which provided by tool framework
    @return: (pass status, CLI information, error message) as (boolean, string, string)
    '''
    
    lang = getLang(dataDict)
    log = getLogger(dataDict)

    cliRet = ''
    errMsg = ''

    #步骤1：获取设备类型(此次需要获取设备的实际型号，不能使用commonT的方法)
    sysInfo, DeviceType = getCurRealDeviceType(dataDict)
    cliRet += sysInfo + "\n"
    if not DeviceType:
        if "zh" == lang:
            errMsg += u"\n未取到有效的设备型号。"
        else:
            errMsg += "\nFailed to get system device type."
        return (CheckedResult.PASS, cliRet, errMsg)
    
    #步骤2：获取当前系统版本号
    CurrentVer = getCurSysVer(dataDict)
    cliRet += getCliRet4UpgradePkgInfo(dataDict) + os.linesep

    #步骤3：获取当前已安装热补丁版本
    iRet = getCurHotPatchVersion(dataDict)
    processFlag = iRet[0]
    curHotPatchVer = iRet[1]
    cliRet += iRet[2] + "\n"
    if not processFlag:
        if "zh" == lang:
            errMsg += u"\n未获取有效的热补丁版本信息。"
        else:
            errMsg += "\nFailed to get hotpatch information."
        return (CheckedResult.NOTPASS, cliRet, errMsg)    
    
    #检查8619芯片热补丁
    processFlag, errMsg = check8619ChipHotpatch(DeviceType, CurrentVer, curHotPatchVer, lang)
    if not processFlag:
        return (CheckedResult.NOTPASS, cliRet, errMsg)    

    return (CheckedResult.PASS, cliRet, errMsg)
