# -*- coding: UTF-8 -*-

import os
import re
from common.constant import CheckedResult
from common.cmdRetManager import getCliRet, checkCliInfoValid
from common.contextUtil import getLang, getLogger, getSshObj, getRet4Datacollect,getMainOrSlaveAndCtrlIdMap


#设备系统时间
G_SYSTEM_TIME = ""

#内存耗尽检查映射关系
G_Moudle_ResUseUp_CheckMap = \
{
  "MSG": ["\[1500003e80003\]", "\[8001500003e80003\]"],
  "OS":  ["\[1500003e500a0\]", "\[8001500003e500a0\]"],
  "Socket": ["nf_conntrack: table full, dropping packet."]
}

# **************************************************************************** #
# 函数名称: saveCurrentSysTime
# 功能说明: 获取当前系统时间并保存
# 输入参数: ssh,lang,log
# 输出参数: (Flag, cliRet, errMsg)
# **************************************************************************** # 
def saveCurrentSysTime(ssh, lang, log):
    
    errMsg = ""
    #获取当前系统时间
    cliRet = ssh.execCmd("showsys")
    lineList = cliRet.splitlines()
    if len(lineList) < 7:
        if "zh" == lang:
            errMsg += u"\n未获取到有效的系统信息。"
        else:
            errMsg += "\nFailed to get system information."

        return (False, cliRet, errMsg)
    #获取系统时间    
    timeStr = ""
    for line in lineList:
        line = line.strip()
        if line.startswith("Time"):
            field = line.split("|")
            timeStr = field[-1].strip().replace("-", "").replace(":", "").replace(" ", "")
            break

    #判断系统时间(必须是14位的数字)
    if not timeStr.isdigit() or len(timeStr) != 14:
        if "zh" == lang:
            errMsg += u"\n未获取到有效的系统信息。"
        else:
            errMsg += "\nFailed to get system information."

        return (False, cliRet, errMsg) 
    
    #保存系统时间
    global G_SYSTEM_TIME  
    G_SYSTEM_TIME = timeStr
    
    return (True, cliRet, errMsg) 

# **************************************************************************** #
# 函数名称: getFailModuleInMessage
# 功能说明: 通过解析文件，查看存在资源耗尽的模块
# 输入参数: messageFile
# 输出参数: checkFailModuleList
# **************************************************************************** # 
def getFailModuleInMessage(messageFile):
    
    global G_Moudle_ResUseUp_CheckMap
    #资源耗尽的模块列表
    checkFailModuleList = []
    checkFailLineList = []
    
    #解析message文件
    fileHandle = open(messageFile)
    for line in fileHandle:
        #搜索关键字
        for module in G_Moudle_ResUseUp_CheckMap.keys():
            
            searchKeyList = G_Moudle_ResUseUp_CheckMap.get(module)
            #搜索到任意一个关键字都记录
            for searchKey in searchKeyList:
                if re.search(searchKey, line, re.IGNORECASE):
                    #保存检查失败的模块
                    if module not in checkFailModuleList:
                        checkFailModuleList.append(module)                
                    
                    #回文中最多显示20条日志
                    if len(checkFailLineList) < 20:
                        checkFailLineList.append(line)
    
    fileHandle.close()
    return checkFailLineList, checkFailModuleList

# **************************************************************************** #
# 函数名称: getNewMessageFileName
# 功能说明: 通过控制器路径名称，获取最新的message文件
# 输入参数: deCompressDestDir, fileName
# 输出参数: 文件绝对路径
# **************************************************************************** # 
def getNewMessageFileName(deCompressDestDir, fileName):
    
    global G_SYSTEM_TIME
    newMesFile = ""

    #文件夹名称匹配
    messageDir = os.sep.join([deCompressDestDir, fileName, 'msg_other', 'msg_other', 'Messages'])
    if not os.path.exists(messageDir) or not os.path.isdir(messageDir):
        return ""

    #获取日志文件列表
    messageFileList = []
    for temp in os.listdir(messageDir):
        fileTemp = messageDir + os.sep + temp
        if os.path.isfile(fileTemp) and temp.startswith("messages_"):
            messageFileList.append(temp)
    if not messageFileList:
        return ""
    
    #获取与系统时间相比最新的日志
    minTimeDistance = None
    #文件名从大到小排列
    messageFileList.sort(reverse=True)
    for temp in messageFileList:
        
        tempTime = temp.strip().replace("messages_", "")
        #时间戳左边对齐，右边补0
        adjustLen = max((len(G_SYSTEM_TIME), len(tempTime)))
        G_SYSTEM_TIME = G_SYSTEM_TIME.ljust(adjustLen, "0")
        tempTime = tempTime.ljust(adjustLen, "0")
        
        #获取时间差绝对值最小的文件（从大到小依次查询）
        if not (tempTime.isdigit() and G_SYSTEM_TIME.isdigit()):
            continue
        distanceTmp = abs(int(tempTime) - int(G_SYSTEM_TIME))
        if minTimeDistance == None:
            minTimeDistance = distanceTmp
            newMesFile = temp
        elif distanceTmp < minTimeDistance:
            minTimeDistance = distanceTmp
            newMesFile = temp
    
    if not newMesFile:
        return ""
    #返回文件绝对路径
    return os.path.join(messageDir, newMesFile)

# **************************************************************************** #
# 函数名称: checkResourceByParseMessage
# 功能说明: 解析最新的Message文件，查看是否存在内部资源不足
# 输入参数: deCompressDestDir, fileName, context
# 输出参数: flag, checkStr, errMsg
# **************************************************************************** # 
def checkResourceByParseMessage(deCompressDestDir, fileName, context):
    
    flag = CheckedResult.PASS
    errMsg = ""
    checkStr = ""
    lang = getLang(context)
    log = getLogger(context)
    
    #获取控制器ID信息
    ctrlId = ''
    if -1 != fileName.find('_MAIN'):#主控
        ctrlId = getMainOrSlaveAndCtrlIdMap(context).get('Primary')
    else:
        ctrlId = getMainOrSlaveAndCtrlIdMap(context).get('Secondary')

    #通过控制器Ip获取最新的Message文件
    messageFilePath = getNewMessageFileName(deCompressDestDir, fileName)

    log.info("Getting new message file path of controller( ID: "  + ctrlId + ") is: " + messageFilePath)
    #判断文件是否存在
    if not (messageFilePath and os.path.exists(messageFilePath)):
        flag = CheckedResult.WARN
        if lang == "zh":
            errMsg = u"控制器%s：无法找到messages_文件，可能原因是日志收集失败或日志解压失败。" % ctrlId
        else:
            errMsg = "Controller %s: Failed to find the messages_ files. The possible cause is log collection or decompression failure." % ctrlId
        return (flag, checkStr, errMsg)

    #资源耗尽的模块列表
    checkFailLineList, checkFailModuleList = getFailModuleInMessage(messageFilePath)
    if not checkFailModuleList:
        return (flag, checkStr, errMsg)

    #整合错误提示信息(checkFailLineList中的回文本身存在换行)
    checkStr += "\nCheck fail lines in file:" + messageFilePath + ":\n" + ("").join(checkFailLineList)
    flag = CheckedResult.NOTPASS
    if lang == "zh":
        errMsg = u"控制器" + ctrlId + u"如下模块存在资源耗尽的情况：" + (",").join(checkFailModuleList)
    else:
        errMsg = "The modules of controller " + ctrlId + " exist the exhaustion of resources: " + (",").join(checkFailModuleList)
    return (flag, checkStr, errMsg)


def execute(dataDict):
    '''
    @summary: the entrance of main method, this check item is used to check internal resource
    @param dataDict: the dictionary of data which provided by tool framework
    @return: (pass status, CLI information, error message) as (boolean, string, string)
    '''
    
    ssh = getSshObj(dataDict)
    log = getLogger(dataDict)
    lang = getLang(dataDict)

    flag = CheckedResult.PASS
    cliRet = ''
    errMsg = ''
    
    #记录系统时间
    iRet = saveCurrentSysTime(ssh, lang, log)
    processFlag = iRet[0]
    cliRet = iRet[1]
    errMsg = iRet[2]
    if not processFlag:
        return (CheckedResult.NOTPASS, cliRet, errMsg) 

    #获取一键收集日志解压文件夹
    flag4Datacollect, dataCollectRet, errMsg4Datacollect, deCompressDestDir, isDoubleCtrlLostOne = getRet4Datacollect(dataDict)
    cliRet += "\n" + dataCollectRet
    #解压路径为空或者不存在说明一定收集不成功(包括解压失败)；双控只收集到一个控制器的日志说明部分收集成功
    if not (deCompressDestDir and os.path.exists(deCompressDestDir)) or isDoubleCtrlLostOne:
        return (CheckedResult.NOCHECK, cliRet, errMsg4Datacollect)

    #查看所有控制器的最新的Message文件，查看是否存在资源不足问题
    #文件夹名称匹配
    for fileName in os.listdir(deCompressDestDir):
        if not fileName.startswith("DebugLog"):
            #解析最新的Message文件
            iRet = checkResourceByParseMessage(deCompressDestDir, fileName, dataDict)
            #判断回文信息
            checkFlag = iRet[0]
            cliRet += "\n" + iRet[1]
            if checkFlag != CheckedResult.PASS:
                flag = checkFlag
                errMsg += "\n" + iRet[2]
    
    #检查结果返回
    return (flag, cliRet, errMsg.strip()) 
