# -*- coding: UTF-8 -*-
import re
import os
from common.constant import CheckedResult
from common.cmdRetManager import getCliRet, checkCliInfoValid
from common.contextUtil import getLang, getLogger, getRet4Datacollect,getMainOrSlaveAndCtrlIdMap
from common.cTV1R1 import cHandleCliRet


# **************************************************************************** #
# 函数名称: parseRaidMmlInfo
# 功能说明: 解析MML信息
# 输入参数: deCompressDestDir, fileName
# 输出参数: ctrlMask
# **************************************************************************** # 
def parseRaidMmlInfo(mmlLineList, log):
    
    lunInfoList = []
    if len(mmlLineList) < 4:
        return lunInfoList
    
    #获取key的列表
    keyLine = mmlLineList[2]
    fieldList = keyLine.split("|")
    #去掉key中的空格
    keyList = [key.strip() for key in fieldList][1:-1]
    
    #解析每一行的LUN信息
    for line in mmlLineList[3:]:

        lunInfoDict = {}
        fieldList = line.split("|")
        valueList = [value.strip() for value in fieldList][1:-1]
        
        if len(valueList) != len(keyList):
            log.error("[parseRaidMmlInfo] line information is invalid:" + line)
            continue
        #保存LUN的信息
        for index in range(len(valueList)):
            lunInfoDict[keyList[index]] = valueList[index]
        lunInfoList.append(lunInfoDict)

    #返回信息
    return lunInfoList


def getRaidMmlFileName(deCompressDestDir, fileName):
    '''
    @summary: 通过控制器路径名称，获取一键收集日志中的指定raid_mmlinfo文件路径
    @param deCompressDestDir: 已解压的日志目录
    @param fileName: IP开头的文件夹名称
    @return: raid_mmlinfo文件的绝对路径
    '''
    #文件夹名称匹配
    mmlFilePath = os.sep.join([deCompressDestDir, fileName, 'msg_other', 'msg_other', 'Other', 'raid_mmlinfo'])
    
    #返回信息
    return mmlFilePath


# **************************************************************************** #
# 函数名称: getMmlFormatLunList
# 功能说明: 解析raid的mml文件，获取所有格式化LUN的信息
# 输入参数: deCompressDestDir, fileName, context
# 输出参数: flag
# **************************************************************************** # 
def getMmlFormatLunList(deCompressDestDir, fileName, context):
    
    flag = CheckedResult.PASS
    isFileExist = True
    errMsg = ""
    checkStr = ""
    lang = getLang(context)
    log = getLogger(context)
    
    allMmlFormatLunList = []
    
    #获取控制器ID信息
    ctrlId = ''
    if -1 != fileName.find('_MAIN'):#主控
        ctrlId = getMainOrSlaveAndCtrlIdMap(context).get('Primary')
    else:
        ctrlId = getMainOrSlaveAndCtrlIdMap(context).get('Secondary')

    #通过控制器Ip查找mml文件位置
    mmlFilePath = getRaidMmlFileName(deCompressDestDir, fileName)
    log.info("Getting mml file path of controller( ID: "  + ctrlId + ") is: "+ mmlFilePath)
    
    #判断文件是否存在
    if not os.path.exists(mmlFilePath):
        flag = CheckedResult.WARN
        isFileExist = False
        if lang == "zh":
            errMsg = u"控制器%s：无法找到文件raid_mmlinfo，可能原因是日志收集失败或日志解压失败。" % ctrlId
        else:
            errMsg = "Controller %s: Failed to find the raid_mmlinfo file. The possible cause is log collection or decompression failure." % ctrlId
        return (flag, checkStr, [], errMsg, isFileExist)

    #解析mml信息文件
    file = open(mmlFilePath)
    mmlInfo = file.read()
    lineList = mmlInfo.splitlines()
    file.close()

    #信息有效性判断
    if not re.search("LUN status", mmlInfo, re.IGNORECASE):
        flag = CheckedResult.WARN
        if lang == "zh":
            errMsg = u"控制器" + ctrlId + u"：获取LUN信息失败。"
        else:
            errMsg = "Controller " + ctrlId + ": Getting LUN information failed."
        return (flag, checkStr, [], errMsg, isFileExist)
    
    #获取有效的命令信息
    lunInfoLineList = []
    startInfoFlag = False
    for line in lineList:
        #文件有效内容记录
        if re.search("LUN status", line, re.IGNORECASE):
            startInfoFlag = True
        
        if startInfoFlag:
            if not line.strip():
                break
            #将字段返回
            lunInfoLineList.append(line)
    
    #界面显示原始信息
    checkStr = "\n[Controller " + ctrlId + "]: raid_mmlinfo file content:\n"
    checkStr += "\n".join(lunInfoLineList)
    #解析原始信息
    lunInfoList = parseRaidMmlInfo(lunInfoLineList, log)
    for lunInfo in lunInfoList:
        status = lunInfo.get("status")
        #4:formatting
        if status == "4":
            allMmlFormatLunList.append(lunInfo)

    #解析成功返回
    return (flag, checkStr, allMmlFormatLunList, errMsg, isFileExist)


def execute(dataDict):
    '''
    @summary: the entrance of main method, this check item is used to check LUN Status (Offline)
    @param dataDict: the dictionary of data which provided by tool framework
    @return: (pass status, CLI information, error message) as (boolean, string, string)
    '''
    
    flag = CheckedResult.PASS
    allCliRet = ""
    errMsg = ""
    
    lang = getLang(dataDict)
    log = getLogger(dataDict)
    
    #步骤1：查询LUN状态
    lunInfo = getCliRet(dataDict, "showlun", False)

    allCliRet += lunInfo + "\n"
    #判断cli信息是否有效
    if not checkCliInfoValid(lunInfo):
        flag = CheckedResult.NOTPASS
        if lang == "zh":
            errMsg += u"\nCli信息无效。"
        else:
            errMsg += "\nCli information invalid."           
        return (flag, allCliRet, errMsg)

    #步骤2：查询Thin Pool
    poolInfo = getCliRet(dataDict, "showpool", False)
    
    allCliRet += poolInfo + "\n"
    #判断cli信息是否有效
    if not checkCliInfoValid(poolInfo):
        flag = CheckedResult.NOTPASS
        if lang == "zh":
            errMsg += u"\nCli信息无效。"
        else:
            errMsg += "\nCli information invalid."           
        return (flag, allCliRet, errMsg)

    #判断是否存在Thin Pool
    existThinPool = False
    if len(poolInfo.splitlines()) > 7:
        existThinPool = True
    
    #使用通用接口解析LUN信息
    formatFunction = cHandleCliRet(lunInfo)
    lunInfoList = formatFunction.handle()

    cliLunIDList = []
    for lunInfoTmp in lunInfoList:

        #获取Lun信息
        lunId = lunInfoTmp.get('Id')
        lunStatus = lunInfoTmp.get('Status')
        
        #记录对外展示的LUN的id
        cliLunIDList.append(lunId)
        if lunStatus == "Normal":
            continue
        elif lunStatus  == "Formatting" and existThinPool:
            #存在ThinPool，格式化的LUN需要报进度，需要通过mml信息查询，此处暂时不处理
            continue
        else:
            flag = CheckedResult.NOTPASS
            if "zh" == lang:
                errMsg += u"\nLUN（ID：" + lunId + u"）的状态存在异常（" + lunStatus + u"）。"
            else:
                errMsg += "\nThe status of LUN(ID:" + lunId +") is abnormal(" + lunStatus + ")."
    
    #不存在ThinPool的情况，直接返回
    if not existThinPool:
        return (flag, allCliRet, errMsg)

    #步骤3：通过一键收集日志获取raid的mml信息
    #获取一键收集日志解压文件夹
    flag4Datacollect, dataCollectRet, errMsgTmp, deCompressDestDir, isDoubleCtrlLostOne = getRet4Datacollect(dataDict)
    allCliRet += "\n" + dataCollectRet
    #解压路径为空或者不存在说明一定收集不成功(包括解压失败)
    if not (deCompressDestDir and os.path.exists(deCompressDestDir)):
        errMsg += "\n" + errMsgTmp
        return (CheckedResult.NOCHECK, allCliRet, errMsg)
    
    #查看制器的最新的raid的mml文件，查看是否存在格式化中的LUN
    allMmlFormatLunList = []
    isFileExistent = False
    for fileName in os.listdir(deCompressDestDir):
        if not fileName.startswith("DebugLog"):
            #解析raid的mml信息
            iRet = getMmlFormatLunList(deCompressDestDir, fileName, dataDict)
            if False == iRet[4]:#所需文件raid_mmlinfo不存在
                continue
            
            #已经获取到MML文件并处理
            isFileExistent = True
            #获取结果
            checkFlag = iRet[0]
            allCliRet += "\n" + iRet[1]
            allMmlFormatLunList = iRet[2]
            
            if checkFlag != CheckedResult.PASS:
                flag = checkFlag
                errMsg += "\n" + iRet[3]
            break
    #所有控制器中均未获取到MML文件
    if not isFileExistent:
        flag = CheckedResult.NOTPASS
        if "zh" == lang:
            errMsg += u"\n无法找到文件raid_mmlinfo，可能原因是日志收集失败或日志解压失败。"
        else:
            errMsg += "\nFailed to find the raid_mmlinfo file. The possible cause is log collection or decompression failure."
        #检查结果返回
        return (flag, allCliRet, errMsg)

    #处理所有的MML查看到的格式化LUN信息
    for mmlLunInfo in allMmlFormatLunList:
        lunId = mmlLunInfo.get("LUN ID")
        lunSize = mmlLunInfo.get("size(M)")
        progress = mmlLunInfo.get("progress")
        
        flag = CheckedResult.NOTPASS
        #构造错误提示信息
        if lunId in cliLunIDList:        
            if "zh" == lang:
                errMsg += u"\nLUN（ID：" + lunId + u"）正在格式化（LUN大小：" + lunSize + u"M，格式化进度："  + progress + u"%）。"
            else:
                errMsg += "\nThe LUN(ID:" + lunId +") is in formatting(LUN size:" + lunSize + "M, format process:"  + progress + "%)." 
        else:
            if "zh" == lang:
                errMsg += u"\nThin Pool的资源LUN（ID：" + lunId + u"）正在格式化（LUN大小：" + lunSize + u"M，格式化进度："  + progress + u"%）。"
            else:
                errMsg += "\nThe Thin Pool's resource LUN(ID:" + lunId +") is in formatting(LUN size:" + lunSize + "M, format process:"  + progress + "%)."             
    
    #检查结果返回
    return (flag, allCliRet, errMsg.strip())

