# -*- coding: UTF-8 -*-

import os
import re
from common.sysInfoManager import getCurDeviceType
from common.constant import CheckedResult, DeviceType
from common.cmdRetManager import getCliRet, checkCliInfoValid
from common.contextUtil import getLang, getLogger, getSshObj, getExtraDataDict, getRet4Datacollect,getMainOrSlaveAndCtrlIdMap, getCurSysVer
from common.cTV1R1 import cHandleCliRet
from common.modelManager import changeCli2Mml
from common.utils import reverseDict


# 保险箱盘不需要检查 added 20140821 Begin
#设备类型的全局信息
G_DEVICE_TYPE = DeviceType.UNKNOWN
# 保险箱盘不需要检查 added 20140821 End


def getCofferDiskLocationList():
    '''
    @summary: 通过设备类型获取保险箱盘location的列表（内部位置）
    @return: 保险箱盘location的列表（内部位置）
    '''

    global G_DEVICE_TYPE
    
    #通过设备类型区分
    if G_DEVICE_TYPE == DeviceType.S6900:
        return ["(1,0)", "(1,1)", "(1,2)", "(1,3)"]
    elif G_DEVICE_TYPE in [DeviceType.S2900, DeviceType.S3900]:
        return ["(0,0)", "(0,1)", "(0,2)", "(0,3)"]
    else:
        return []


def getIoStatFileAbsPath(deCompressDestDir, fileName):
    '''
    @summary: 通过控制器路径名称，获取一键收集日志中的指定iostat_1_10文件路径
    @param deCompressDestDir: 已解压 的日志目录
    @param fileName: IP开头的文件夹名称
    @return: iostat_1_10文件的绝对路径
    '''
    #文件夹名称匹配
    ioStatFilePath = os.sep.join([deCompressDestDir, fileName, 'msg_other', 'msg_other', 'Other', 'iostat_1_10'])
    return ioStatFilePath


def getEmpMmlFileAbsPath(deCompressDestDir, fileName):
    '''
    @summary: 通过控制器路径名称，获取一键收集日志中的指定emp_mmlinfo文件路径
    @param deCompressDestDir: 已解压 的日志目录
    @param fileName: IP开头的文件夹名称
    @return: emp_mmlinfo文件的绝对路径
    '''
    #文件夹名称匹配
    empMmlFilePath = os.sep.join([deCompressDestDir, fileName, 'msg_other', 'msg_other', 'Other', 'emp_mmlinfo'])
    return empMmlFilePath


def getFileContentByAbsPath(fileAbsPath, dataDict):
    '''
    @summary: get the content of file
    @param fileAbsPath: the absolute path of file
    @param dataDict: the dictionary of data which provided by tool framework
    @return: the content of file, as type string
    '''
    #解析mml信息文件
    file = open(fileAbsPath)
    fileContent = file.read()
    file.close()
    return fileContent


def getDiskNameAndInnerLocMap(empMmlFileContent, dataDict):
    '''
    @summary: get disk name list and inner location map relationship
    @param empMmlFileContent: emp_mmlinfo content
    @param dataDict: the dictionary of data which provided by tool framework
    @return: disk name and location map relationship, key:diskName, val:location
    '''
    log = getLogger(dataDict)
    lang = getLang(dataDict)
    diskNameAndLocMap = {}
    
    infoStart = False
    infoEnd = False
    location = ""
    diskName = ""
    
    lineList = empMmlFileContent.splitlines()
    for line in lineList:
        #获取diskinfo命令执行区间
        if "MML>dev cmd analyze ," in line:
            if "diskinfo" in line:
                infoStart = True
            else:
                infoEnd = True
        if not infoStart:
            continue
        if infoEnd:
            break
        
        #获取磁盘位置与盘符对应关系
        if line.find("wwn") >= 0:
            #查询磁盘的location
            field = line.split()
            if len(field) > 2:
                location = field[1]
            #继续查询下一行
            continue
        if location != "" and line.find("Name") >= 0:
            #查询盘符信息
            field = line.split(",")
            for temp in field:
                if temp.startswith("Name"):
                    #盘符可能为空，也需要记录
                    diskName = temp.replace("Name ", "")
                    break
            if location and diskName:
                diskNameAndLocMap[diskName] = location
            #初始化变量
            location = ""
            diskName = ""
    return diskNameAndLocMap


def getFramerecordContent(empMmlFileContent, context):
    '''
    @summary: get content of the 'framerecord' information in file emp_mmlinfo
    @param empMmlFileContent: the content of file emp_mmlinfo
    @param context: the dictionary of data which provided by tool framework
    @return: content of the 'framerecord' information in file emp_mmlinfo
    @note: in fact, it is the same as getting result of sending command 'dev framerecord'
    '''
    logger = getLogger(context)
    pattern = 'MML>([^MML>]+?framerecord.+?MML>)'
    ret4Search = re.search(pattern, empMmlFileContent, re.DOTALL)
    if not ret4Search:
        return ''
    framerecordInfo = ret4Search.group(1)
    logger.info('The framerecord information in file emp_mmlinfo is:' + framerecordInfo)
    return framerecordInfo


def getEnclosureIdMap(empMmlFileContent, context):
    '''
    @summary: get outer and inner enclosure ID map relationship
    @param empMmlFileContent: the content of file emp_mmlinfo
    @param context: the dictionary of data which provided by tool framework
    @return: enclosure ID map relationship like {outer:inner}, as dictionary
    '''
    enclosureIdDict = {}
    ssh = getSshObj(context)
    logger = getLogger(context)
    devFrameRecordContent = getFramerecordContent(empMmlFileContent, context)
    for line in devFrameRecordContent.splitlines():
        if 'user, inner, mac' in line:
            pattern = '\[(\d+), (\d+), .+?\]'
            ret4Search = re.search(pattern, line)
            if ret4Search:
                userId = ret4Search.group(1)
                innerId =ret4Search.group(2)
                enclosureIdDict[userId] = innerId
    logger.info('enclosureIdDict=' + unicode(enclosureIdDict))
    return enclosureIdDict


def translateOuter2Inner(enclosureIdMap, outerDiskLocaList):
    '''
    @summary: translate outer enclosure id to inner enclosure id list
    @param enclosureIdMap: a dictionary which map enclosure ID relationship like {outer:inner} 
    @param outerDiskLocaList: the list of outer disk location
    @return: inner enclosure id and slot map list
    '''
    innerDiskLocaList = []
    for outerDiskLoca in outerDiskLocaList:
        outerEnclosureId = outerDiskLoca[1:-1].split(',')[0]
        slotId = outerDiskLoca[1:-1].split(',')[1]#槽位号内部和外部是一致的，所以不需要转换
        innerEnclosureId = enclosureIdMap.get(outerEnclosureId)
        innerDiskLocaList.append('(%s,%s)' % (innerEnclosureId, slotId))
    return innerDiskLocaList


def translateInner2Outer(enclosureIdMap, innerDiskLocaList):
    '''
    @summary: translate inner enclosure id list to outer enclosure id list
    @param enclosureIdMap: a dictionary which map enclosure ID relationship like {inner:outer} 
    @param innerDiskLocaList: the list of inner disk location
    @return: outer enclosure id and slot map list
    '''
    outerDiskLocaList = []
    for innerDiskLoca in innerDiskLocaList:
        innerEnclosureId = innerDiskLoca[1:-1].split(',')[0]
        slotId = innerDiskLoca[1:-1].split(',')[1]#槽位号内部和外部是一致的，所以不需要转换
        outerEnclosureId = enclosureIdMap.get(innerEnclosureId)
        outerDiskLocaList.append('(%s,%s)' % (outerEnclosureId, slotId))
    return outerDiskLocaList


def beautifyDict(tmpDict):
    '''
    @summary: beautify dictionary
    @param tmpDict: a dictionary which need to beautify, e.g:{'sda': '(0,0)', 'sdd': '(0,1)'}
    @return: a beautified string, like '[sda:(0,0), sdd:(0,1)]'
    '''
    tmpDictStr = unicode(tmpDict)
    tmpDictStr = tmpDictStr.replace('{', '[').replace('}', ']').replace('\'', '').replace(': ', ':')
    return tmpDictStr


def checkCofferDiskPerRaid(empMmlFileAbsPath, dataDict, notPassDiskDict):
    '''
    @summary: check whether every member disk of RAID group is coffer disk
    @param empMmlFileAbsPath: the absolute path of file emp_mmlinfo
    @param dataDict: the dictionary of data which provided by tool framework
    @param notPassDiskDict: the dictionary of no pass disk, like {disk_name: outer_location_list}
    @return: error message
    '''
    errMsg = ''
    log = getLogger(dataDict)
    lang = getLang(dataDict)
    
    #获取该设备所有RAID组信息回文
    cliRet = getCliRet(dataDict, "showrg")
    cliRetHandler = cHandleCliRet(cliRet)
    raidGroupDictList = cliRetHandler.handle()
    log.info('raidGroupDictList=' + unicode(raidGroupDictList))
    
    #获取框的外部ID与内部ID对应关系
    enclosureIdMap = getExtraDataDict(dataDict).get('enclosureIdMap')
    if not enclosureIdMap:
        if lang == 'zh':
            errMsg += u'获取框的外部ID与内部ID对应关系失败。'
        else:
            errMsg += 'Failed to obtain the relationship between the external ID and internal ID of the enclosure.'
        return errMsg
    
    #获取规范的保险箱盘位置列表（内部框号槽位号）
    cofferDiskLocationList = getCofferDiskLocationList()
    
    #获取所有硬盘名称和外部位置映射字典
    allDiskNameAndOuterLocMap = getExtraDataDict(dataDict).get('diskNameAndOuterLocMap')
    log.info('allDiskNameAndOuterLocMap=' + unicode(allDiskNameAndOuterLocMap))
    
    #检查每个RAID组成员盘是否都为保险箱盘
    for raidGroupDict in raidGroupDictList:
        rgId = raidGroupDict.get('Id')
        rgDiskLocaListStr = raidGroupDict.get('Disk List')#例如：1,0;1,1;1,2;
        tmpRgDiskLocaList = rgDiskLocaListStr.split(';')
        log.info('rgId=%s;\t rgDiskLocaListStr=%s' % (rgId, rgDiskLocaListStr))
        
        #获取当前RAID组内的所有硬盘位置列表（外部框号槽位号）
        outerRgDiskLocaList = []
        for tmpOuterRgDiskLoca in tmpRgDiskLocaList:
            if not tmpOuterRgDiskLoca:
                continue
            outerRgDiskLocaList.append('(%s)' % tmpOuterRgDiskLoca)
        log.info('outerRgDiskLocaList=' + unicode(outerRgDiskLocaList))#['(1,0)', '(1,1)', '(1,2)']
        
        #将外部框号槽位号装换为内部框号槽位号
        innerRgDiskLocaList = translateOuter2Inner(enclosureIdMap, outerRgDiskLocaList)
        
        isCofferRaid = True #是否为保险箱RAID组（即该RAID组成员盘全为保险箱盘）
        for innerRgDiskLoca in innerRgDiskLocaList:
            if innerRgDiskLoca not in cofferDiskLocationList:
                isCofferRaid = False
                break
        #若当前RAID组的成员盘不全为保险箱盘，则检查下一个RAID组
        if not isCofferRaid:
            continue
        
        curRgNotPassDiskDict = {}#当前RAID组检查不通过硬盘字典，格式如{盘符:外部位置}
        
        #获取当前RAID组硬盘名称和外部位置映射字典
        curRgDiskNameAndOuterLocMap = {}
        allDiskOuterLocAndNameMap = reverseDict(allDiskNameAndOuterLocMap)
        for outerRgDiskLoca in outerRgDiskLocaList:
            diskName = allDiskOuterLocAndNameMap.get(outerRgDiskLoca)
            curRgDiskNameAndOuterLocMap[diskName] = outerRgDiskLoca
        
        #当前RAID组成员盘全部为保险箱盘，查看有哪些盘是不通过的盘
        for rgDiskName in curRgDiskNameAndOuterLocMap.keys():
            if rgDiskName in notPassDiskDict.keys():
                curRgNotPassDiskDict[rgDiskName] = curRgDiskNameAndOuterLocMap.get(rgDiskName)
        if not curRgNotPassDiskDict:
            continue
        tuple2BReplaced_zh = (rgId, beautifyDict(curRgDiskNameAndOuterLocMap), beautifyDict(curRgNotPassDiskDict))
        tuple2BReplaced_en = (beautifyDict(curRgDiskNameAndOuterLocMap), rgId, beautifyDict(curRgNotPassDiskDict))
        if lang == 'zh':
            #RAID组0的所有成员盘[sda:(0,0), sdd:(0,1), sdb:(0,2), sdc:(0,3)]都是保险箱盘，其中IO利用率超过了阈值20%的有[sdb:(0,2), sdc:(0,3)]。
            errMsg += u'RAID组%s的所有成员盘%s都是保险箱盘，其中IO利用率超过了阈值20%%的有%s。\n' % tuple2BReplaced_zh
        else:
            errMsg += 'All member disks %s of RAID group %s are coffer disks. Among them, %s have their I/O usage exceeding 20%% of the threshold.\n' % tuple2BReplaced_en
        
    return errMsg


def getDiskNameAndOuterLocMap(empMmlFileAbsPath, dataDict):
    '''
    @summary: get disk name and outer location map relationship
    @param empMmlFileAbsPath: the absolute path of file emp_mmlinfo
    @param dataDict: the dictionary of data which provided by tool framework
    @return: disk map relationship like {disk_name:disk_outer_location}
    '''
    diskNameAndOuterLocMap = {}
    log = getLogger(dataDict)
    
    #1.获取盘符和内部位置（框号，槽位号）的映射关系
    empMmlFileContent = getFileContentByAbsPath(empMmlFileAbsPath, dataDict)
    diskNameAndInnerLocMap = getDiskNameAndInnerLocMap(empMmlFileContent, dataDict)
    log.info('diskNameAndInnerLocMap=' + unicode(diskNameAndInnerLocMap))
    
    #2.获取框的外部ID与内部ID对应关系
    enclosureIdMap = getEnclosureIdMap(empMmlFileContent, dataDict)
    getExtraDataDict(dataDict)['enclosureIdMap'] = enclosureIdMap
    enclosureIdMap = reverseDict(enclosureIdMap)
    log.info('enclosureIdMap=' + unicode(enclosureIdMap))
    
    #3.通过步骤1和2获取盘符外部位置（框号，槽位号）的映射关系
    for diskName in diskNameAndInnerLocMap.keys():
        InnerLoc = diskNameAndInnerLocMap.get(diskName)
        diskNameAndOuterLocMap[diskName] = translateInner2Outer(enclosureIdMap, [InnerLoc])[0]
    
    return diskNameAndOuterLocMap


def getDiskUsageListDict(ioStatContent, dataDict):
    '''
    @summary: get the dictionary of every disk usage list
    @param ioStatContent: the content of file iostat_1_10
    @param dataDict: the dictionary of data which provided by tool framework
    @return: get the dictionary of every disk usage list, like {diskName:[usage2,usage3]}
    '''
    log = getLogger(dataDict)
    diskUsageListDict = {}
    
    counter = 0
    lineList = ioStatContent.splitlines()
    for line in lineList:
        #从第二份数据开始判断
        if re.search("avg-cpu", line, re.IGNORECASE):
            counter += 1
        #从第二份数据开始判断
        if counter < 2:
            continue

        if re.search("sd", line, re.IGNORECASE):
            field = line.split()
            #保存盘符信息
            deviceName = field[0]
            
            if not diskUsageListDict.has_key(deviceName):
                diskUsageListDict[deviceName] = []

            #获取并保存利用率数值
            if len(field) == 14:
                diskUsage = int(field[13].split(".")[0])
                diskUsageListDict[deviceName].append(diskUsage)
            else:
                log.error("Disk useage info error: " + line)
                
    return diskUsageListDict


def checkDiskUseageByFile(deCompressDestDir, fileName, dataDict):
    '''
    @summary: 解析iostat_1_10文件，查看硬盘利用率
    @param deCompressDestDir: 一键收集日志解压路径
    @param fileName: IP打头的文件夹名
    @param dataDict: 数据字典（工具框架提供）
    @return: (通过标识, 检查的信息, 错误消息) as (integer, string, string)
    '''
    flag = CheckedResult.PASS
    errMsg = ""
    checkStr = ""
    log = getLogger(dataDict)
    lang = getLang(dataDict)
    notPassDiskDict = {}#不通过的盘名称和外部位置映射字典

    #获取控制器ID信息
    ctrlId = ''
    if -1 != fileName.find('_MAIN'):#主控
        ctrlId = getMainOrSlaveAndCtrlIdMap(dataDict).get('Primary')
    else:
        ctrlId = getMainOrSlaveAndCtrlIdMap(dataDict).get('Secondary')
    
    #通过控制器Ip获取ioStat文件绝对路径
    ioStatFilePath = getIoStatFileAbsPath(deCompressDestDir, fileName)
    log.info("Getting mml file path of controller( ID: %s) is: %s" % (ctrlId, ioStatFilePath))
    
    #判断文件是否存在
    if not os.path.exists(ioStatFilePath):
        flag = CheckedResult.WARN
        if lang == "zh":
            errMsg = u"控制器%s：无法找到文件iostat_1_10，可能原因是日志收集失败或日志解压失败。" % ctrlId
        else:
            errMsg = "Controller %s: Failed to find the iostat_1_10 file. The possible cause is log collection or decompression failure." % ctrlId
        return (flag, checkStr, errMsg)
    
    #通过控制器Ip获取emp_mmlinfo文件绝对路径
    empMmlFileAbsPath = getEmpMmlFileAbsPath(deCompressDestDir, fileName)
    log.info("Getting mml file path of controller( ID: %s) is: %s" % (ctrlId, empMmlFileAbsPath))
    
    #判断文件是否存在
    if not os.path.exists(empMmlFileAbsPath):
        flag = CheckedResult.WARN
        if lang == "zh":
            errMsg = u"控制器%s：无法找到文件emp_mmlinfo，可能原因是日志收集失败或日志解压失败。" % ctrlId
        else:
            errMsg = "Controller %s: Failed to find the emp_mmlinfo file. The possible cause is log collection or decompression failure." % ctrlId
        return (flag, '', errMsg)
    
    #解析mml信息文件
    ioStatInfo = getFileContentByAbsPath(ioStatFilePath, dataDict)
    checkStr += "[Controller " + ctrlId + "'s iostat information]:\n" + ioStatInfo
    
    #获取硬盘利用率列表字典
    checkDiskUsageListDict = getDiskUsageListDict(ioStatInfo, dataDict)
    
    #获取盘符和外部位置（框号，槽位号）的映射关系
    diskNameAndOuterLocMap = getDiskNameAndOuterLocMap(empMmlFileAbsPath, dataDict)
    getExtraDataDict(dataDict)['diskNameAndOuterLocMap'] = diskNameAndOuterLocMap
    log.info('diskNameAndOuterLocMap=' + unicode(diskNameAndOuterLocMap))
    
    #计算平均值判断
    for diskName in checkDiskUsageListDict:
        useageValueList = checkDiskUsageListDict.get(diskName)
        curSysVer = getCurSysVer(dataDict)
        threshold = 40
        if curSysVer > 'V100R005C30SPC100':
            threshold = 20
        if useageValueList:
            averageValue = sum(useageValueList)/len(useageValueList)
            if averageValue > threshold:
                diskOuterLoc = diskNameAndOuterLocMap.get(diskName)
                notPassDiskDict[diskName] = diskOuterLoc
                flag = CheckedResult.NOTPASS
                tuple2BReplaced = (diskName, diskOuterLoc, unicode(averageValue), unicode(threshold))
                if lang == "zh":
                    errMsg += u'硬盘[Name:%s, Location:%s]的平均利用率为%s%%，超过了阈值%s%%。\n' % tuple2BReplaced
                else:
                    errMsg += 'The average usage of [Name:%s, Location:%s] is %s%%, larger than the threshold(%s%%).\n' % tuple2BReplaced
        else:
            continue
    
    #查询每个RAID组的成员盘是否都为保险箱盘：
    errMsg += checkCofferDiskPerRaid(empMmlFileAbsPath, dataDict, notPassDiskDict)
    
    #附上控制器ID
    if errMsg:
        if lang == "zh":
            errMsg = u"控制器" + ctrlId + u"检查详细结果：\n" + errMsg
        else:
            errMsg = u"Controller " + ctrlId + u" 's check result:\n" + errMsg
    
    return (flag, checkStr, errMsg)


def execute(dataDict):
    '''
    @summary: the entrance of main method, this check item is used to check Service Pressure
    @param dataDict: the dictionary of data which provided by tool framework
    @return: (pass status, CLI information, error message) as (boolean, string, string)
    '''
    
    log = getLogger(dataDict)
    lang = getLang(dataDict)

    flag = CheckedResult.PASS
    cliRet = ''
    errMsg = ''

    # 保险箱盘不需要检查 added 20140821 Begin
    #获取设备类型
    global G_DEVICE_TYPE
    iRet = getCurDeviceType(dataDict)
    cliRet = iRet[0]
    G_DEVICE_TYPE = iRet[1]
    if G_DEVICE_TYPE == DeviceType.UNKNOWN:
        flag = CheckedResult.NOTPASS
        if lang == "zh":
            errMsg = u"获取设备类型失败。"
        else:
            errMsg = "Getting device type failed."
        return (flag, cliRet, errMsg) 
    # 保险箱盘不需要检查 added 20140821 End

    #获取一键收集日志解压文件夹
    flag4Datacollect, dataCollectRet, errMsg4Datacollect, deCompressDestDir, isDoubleCtrlLostOne = getRet4Datacollect(dataDict)
    cliRet += "\n" + dataCollectRet
    #解压路径为空或者不存在说明一定收集不成功(包括解压失败)；双控只收集到一个控制器的日志说明部分收集成功
    if not (deCompressDestDir and os.path.exists(deCompressDestDir)) or isDoubleCtrlLostOne:
        return (CheckedResult.NOCHECK, cliRet, errMsg4Datacollect)
    
    #查看所有控制器的iostat_1_10文件，确认io利用率
    #文件夹名称匹配
    for fileName in os.listdir(deCompressDestDir):
        if not fileName.startswith("DebugLog"):
            #解析iostat_1_10文件，查看硬盘利用率
            iRet = checkDiskUseageByFile(deCompressDestDir, fileName, dataDict)
            #判断回文信息
            checkFlag = iRet[0]
            cliRet += "\n" + iRet[1]
            if checkFlag != CheckedResult.PASS:
                flag = checkFlag
                errMsg += "\n" + iRet[2]
    
    #检查结果返回
    return (flag, cliRet, errMsg.strip()) 
