# -*- coding: UTF-8 -*-

import re
from time import sleep
import string
from common.constant import CheckedResult
from common.cmdRetManager import getCliRet, checkCliInfoValid
from common.contextUtil import getLang, getLogger, getSshObj
from common.modelManager import changeCli2Mml
from common.utils import reverseDict

# **************************************************************************** #
# 函数名称: getPeerHeartBeatIP
# 功能说明: 解析数据，获取远端控制器的心跳IP
# 输入参数: 
# 输出参数:
# 返 回 值: 
# **************************************************************************** # 
def getPeerHeartBeatIP(heartBeatIPMsg):
    rowList = string.split(heartBeatIPMsg, '\n')
    rowCount = len(rowList)
    for iter in range(rowCount):
        peerHeartBeatIP = ''
        if bool(re.search('inet addr:', rowList[iter])):
            columnList = string.split(rowList[iter])
            peerHeartBeatIP = string.split(columnList[1], ':')[1]
        if '127.127.127.10' == peerHeartBeatIP:
            return '127.127.127.11'
        elif '127.127.127.11' == peerHeartBeatIP:
            return '127.127.127.10'
    return ''

# **************************************************************************** #
# 函数名称: LinkCtrlByHeartBeat
# 功能说明: 通过心跳来切换命令发送的控制器
# 输入参数: dataDict
# 输出参数: NA
# 返 回 值: True,False
# **************************************************************************** # 
def LinkCtrlByHeartBeat(ssh):
    #执行此操作前确保切换到Storage命令模式
    heartBeatIPMsgOld = ssh.execCmd('ifconfig bond0')
    peerHeartBeatIPOld = getPeerHeartBeatIP(heartBeatIPMsgOld)
    ssh.execCmd('ssh ibc_os_hs@' + peerHeartBeatIPOld)
    heartBeatIPMsgNew = ssh.execCmd('ifconfig bond0')
    peerHeartBeatIPNew = getPeerHeartBeatIP(heartBeatIPMsgNew)
    if bool(re.search('127.127.127', peerHeartBeatIPOld)) \
    and bool(re.search('127.127.127', peerHeartBeatIPNew)) \
    and (peerHeartBeatIPOld != peerHeartBeatIPNew):
        return True
    else:
        return False


def getDiskNameAndInnerLocMap(empMmlFileContent, dataDict):
    '''
    @summary: get disk name list and inner location map relationship
    @param empMmlFileContent: emp_mmlinfo content
    @param dataDict: the dictionary of data which provided by tool framework
    @return: disk name and location map relationship, key:diskName, val:location
    '''
    log = getLogger(dataDict)
    lang = getLang(dataDict)
    diskNameAndLocMap = {}
    
    infoStart = False
    infoEnd = False
    location = ""
    diskName = ""
    
    lineList = empMmlFileContent.splitlines()
    for line in lineList:
        #获取diskinfo命令执行区间
        if "dev cmd analyze ," in line:
            if "diskinfo" in line:
                infoStart = True
            else:
                infoEnd = True
        if not infoStart:
            continue
        if infoEnd:
            break
        
        #获取磁盘位置与盘符对应关系
        if line.find("wwn") >= 0:
            #查询磁盘的location
            field = line.split()
            if len(field) > 2:
                location = field[1]
            #继续查询下一行
            continue
        if location != "" and line.find("Name") >= 0:
            #查询盘符信息
            field = line.split(",")
            for temp in field:
                if temp.startswith("Name"):
                    #盘符可能为空，也需要记录
                    diskName = temp.replace("Name ", "")
                    break
            if location and diskName:
                diskNameAndLocMap[diskName] = location
            #初始化变量
            location = ""
            diskName = ""
    return diskNameAndLocMap


def translateInner2Outer(enclosureIdMap, innerDiskLocaList):
    '''
    @summary: translate inner enclosure id list to outer enclosure id list
    @param enclosureIdMap: a dictionary which map enclosure ID relationship like {inner:outer} 
    @param innerDiskLocaList: the list of inner disk location
    @return: outer enclosure id and slot map list
    '''
    outerDiskLocaList = []
    for innerDiskLoca in innerDiskLocaList:
        innerEnclosureId = innerDiskLoca[1:-1].split(',')[0]
        slotId = innerDiskLoca[1:-1].split(',')[1]#槽位号内部和外部是一致的，所以不需要转换
        outerEnclosureId = enclosureIdMap.get(innerEnclosureId)
        outerDiskLocaList.append('(%s,%s)' % (outerEnclosureId, slotId))
    return outerDiskLocaList


def getEnclosureIdMap(context):
    '''
    @summary: get outer and inner enclosure ID map relationship
    @param context: the dictionary of data which provided by tool framework
    @return: enclosure ID map relationship like {outer:inner}, as dictionary
    '''
    enclosureIdDict = {}
    ssh = getSshObj(context)
    logger = getLogger(context)
    
    temp = ssh.execCmd('dev framerecord')
    for line in temp.splitlines():
        if 'user, inner, mac' in line:
            pattern = '\[(\d+), (\d+), .+?\]'
            ret4Search = re.search(pattern, line)
            if ret4Search:
                userId = ret4Search.group(1)
                innerId =ret4Search.group(2)
                enclosureIdDict[userId] = innerId
    logger.info('enclosureIdDict=' + unicode(enclosureIdDict))
    return enclosureIdDict


def getDiskNameAndOuterLocMap(devDiskinfo, dataDict, enclosureIdMap):
    '''
    @summary: get disk name and outer location map relationship
    @param devDiskinfo: the executed result of mml command 'dev diskinfo'
    @param dataDict: the dictionary of data which provided by tool framework
    @param enclosureIdMap: the outer and inner of enclosure ID map relationship
    @return: disk map relationship like {disk_name:disk_outer_location}
    '''
    log = getLogger(dataDict)
    diskNameAndOuterLocMap = {}
    
    #1.获取盘符和内部位置（框号，槽位号）的映射关系
    diskNameAndInnerLocMap = getDiskNameAndInnerLocMap(devDiskinfo, dataDict)
    
    #2.通过步骤1和enclosureIdMap获取盘符外部位置（框号，槽位号）的映射关系
    for diskName in diskNameAndInnerLocMap.keys():
        InnerLoc = diskNameAndInnerLocMap.get(diskName)
        diskNameAndOuterLocMap[diskName] = translateInner2Outer(enclosureIdMap, [InnerLoc])[0]
    
    return diskNameAndOuterLocMap
   
    
def checkCurCtrlPressure(dataDict):
    '''
    @summary: check current controller pressure
    @param dataDict: the dictionary of data which provided by tool framework
    @return: (pass status, command return information, error message) as (int, string, string)
    '''
    ssh = getSshObj(dataDict)
    lang = getLang(dataDict)
    log = getLogger(dataDict)
    #进入此函数执行前，需保证ssh在mml命令模式下
    allRet = ''
    checkFlag = CheckedResult.PASS
    errMsg = ''
    
    """查询IO利用率"""
    mmlRet = ssh.execCmd("ioperf ioutilization 5")
    #命令执行结果正确才添加到回显信息中并判断
    if re.search("IO utilization:", mmlRet, re.IGNORECASE):
        allRet += mmlRet + '\n'
        
        lineList = mmlRet.splitlines()
        for list in lineList:
            if re.search("IO utilization:", list, re.IGNORECASE):
                field = list.split()
                ioUsage = int(field[2].replace("[", "").replace("]", ""))
                if ioUsage > 60:
                    checkFlag = CheckedResult.NOTPASS
                    log.error("Check io usage error:" + list)
                    if lang == "zh":
                        errMsg += u"系统当前IO利用率为" + unicode(ioUsage) + u"%，超过了阈值60%。\n"
                    else:
                        errMsg += "The IO Usage is " + unicode(ioUsage) + "%, larger than the threshold(60%).\n"
                break
            else:
                continue
    
    """查询硬盘利用率"""
    #获取框的外部ID与内部ID对应关系
    enclosureIdMap = getEnclosureIdMap(dataDict)
    enclosureIdMap = reverseDict(enclosureIdMap)
    log.info('enclosureIdMap=' + unicode(enclosureIdMap))
    
    #通过内部框ID获取硬盘信息（包括盘符和内部框号槽位号）
    innerEnclosureIdList = enclosureIdMap.keys()
    devDiskinfo = ''
    for innerEnclosureId in innerEnclosureIdList:
        devDiskinfo += '\n' + ssh.execCmd('dev diskinfo ' + innerEnclosureId)
    
    #获取盘符和外部位置（框号，槽位号）的映射关系
    diskNameAndOuterLocMap = getDiskNameAndOuterLocMap(devDiskinfo, dataDict, enclosureIdMap)
    log.info('diskNameAndOuterLocMap=' + unicode(diskNameAndOuterLocMap))
    
    #统计当前5秒内的4次硬盘利用率数据
    #要去掉第一份数据，因为第一份数据是自系统启动以来的利用率，只保留后三份数据
    ssh.execCmd("exit")
    mmlRet = ssh.execCmd("iostat -x 5 4")
    allRet += mmlRet + '\n'
    lineList = mmlRet.splitlines()
    dateValid = False
    counter = 0
    
    #检查不通过的硬盘盘符列表
    checkFailDiskList = []
    
    for list in lineList:
        #只查询第二份数据
        if re.search("avg-cpu", list, re.IGNORECASE):
            counter += 1

        if counter >= 2:
            dateValid = True
    
        if dateValid != True:
            continue
        
        if re.search("sd", list, re.IGNORECASE):
            field = list.split()
            device = field[0]
            if len(field) == 14:
                diskUsage = int(field[13].split(".")[0])
                if diskUsage > 40:
                    log.error("Check disk usage error:" + list)
                    checkFlag = CheckedResult.NOTPASS

                    #将硬盘记录到检查不通过列表中
                    #每块硬盘最多只打印一次
                    if device not in checkFailDiskList:
                        checkFailDiskList.append(device)
                        diskOuterLoc = diskNameAndOuterLocMap.get(device)
                        tuple2BReplaced = (unicode(counter), device, diskOuterLoc, unicode(diskUsage))
                        if lang == "zh":
                            errMsg += u"第%s次数据：硬盘[Name:%s, Location:%s]的利用率为%s%%，超过了阈值40%%。\n" % tuple2BReplaced
                        else:
                            errMsg += "The %stime's data: the disk [Name:%s, Location:%s] usage is %s%%, larger than the threshold(40%%).\n" % tuple2BReplaced
        else:
            continue
    return (checkFlag, allRet, errMsg)


"""业务压力检查"""
def execute(dataDict):
    '''
    @summary: the entrance of main method, this check item is used to check service pressure
    @param dataDict: the dictionary of data which provided by tool framework
    @return: (pass status, CLI information, error message) as (boolean, string, string)
    '''
    allRet = ''
    checkFlag = CheckedResult.PASS
    errMsg = ''
    checkRetTmp1 = None
    checkRetTmp2 = None
    
    ssh = getSshObj(dataDict)
    log = getLogger(dataDict)
    lang = getLang(dataDict)
    iRet = changeCli2Mml(dataDict)
    if True == iRet:
        #查询当前控制器的业务压力
        checkRetTmp1 = checkCurCtrlPressure(dataDict)
        #心跳到对端，查询对端控制器的业务压力
        ret = LinkCtrlByHeartBeat(ssh)
        if True == ret:
            ssh.execCmd("mml")
            checkRetTmp2 = checkCurCtrlPressure(dataDict)
            #退出心跳
            ssh.execCmd("exit")
        else:
            pass
            
        """退回到cli模式"""
        ssh.execCmd("exit")
        ssh.execCmd("exit")
        
        """确认总体检查结果"""
        if checkRetTmp1 != None:
            if lang == 'zh':
                allRet += u"本端控制器命令结果：\n" + checkRetTmp1[1]
            else:
                allRet += u"Local Controller command results:\n" + checkRetTmp1[1]

            if checkRetTmp1[0] == CheckedResult.PASS:
                pass
            else:
                checkFlag = CheckedResult.NOTPASS
                if lang == 'zh':
                    errMsg += u"本端控制器检查结果：\n" + checkRetTmp1[2]
                else:
                    errMsg += u"Local Controller check results:\n" + checkRetTmp1[2]
                
        if checkRetTmp2 != None:
            if lang == 'zh':
                allRet += u"对端控制器命令结果：\n" + checkRetTmp2[1]
            else:
                allRet += u"Peer Controller command results:\n" + checkRetTmp2[1]
                
            if checkRetTmp2[0] == CheckedResult.PASS:
                pass
            else:
                checkFlag = CheckedResult.NOTPASS
                if lang == 'zh':
                    errMsg += u"对端控制器检查结果：\n" + checkRetTmp2[2]
                else:
                    errMsg += u"Peer Controller check results:\n" + checkRetTmp2[2]     
        
    else:
        checkFlag = CheckedResult.WARN
        if lang == "zh":
            errMsg = u"进入MML模式失败，业务压力检查失败。失败的原因可能为：\n" \
                    + u"（1）添加设备时未输入developer密码。\n（2）添加设备时输入的developer密码无效。"
        else:
            errMsg = "Login to MML model failed, can not execute service pressure check.The reason of failure may be:\n" \
                    + "(1) Did not enter a developer password when adding the device.\n(2) The developer password entered is incorrect."

    return (checkFlag, allRet, errMsg)

