# -*- coding: UTF-8 -*-

import re
from time import sleep
from common.constant import CheckedResult
from common.cmdRetManager import getCliRet, checkCliInfoValid
from common.contextUtil import getLang, getLogger, getSshObj
from common.modelManager import *


# **************************************************************************** #
# 函数名称: getTGTIoInfo
# 功能说明: 获取TGT的io数据
# 输入参数: tgtIoInfo, lang
# 输出参数: NA
# 返 回 值: Start和not back的Io数据
# **************************************************************************** # 
def getTGTIoInfo(showIoInfo, lang):

    startIoData = ""
    notbackIoData = ""

    readIoTgtStart = ""
    readIoTgtNotback = ""
    writeIoTgtStart = ""
    writeIoTgtNotback = ""

    tgtIoInfo = ()
    
    writeIoDataFlag = False
        
    #解析正确返回的命令回文
    lineList = showIoInfo.splitlines()
    for line in lineList:
        #查看是否为Write IOS数据
        if re.search("Write IOs:", line, re.IGNORECASE):
            writeIoDataFlag = True
            continue
        elif line.startswith("TGT"):
            #初始化临时变量
            startIoData = ""
            notbackIoData = ""

            field = line.split()
            for temp in field:
                #获取数据
                if temp.startswith("Start:"):
                    startIoData = temp.replace("Start:", "").replace(",", "")
                elif temp.startswith("back:"): 
                    notbackIoData = temp.replace("back:", "")
                else:
                    continue

            #获取数据成功
            if startIoData != "" and notbackIoData != "":
                #判断是读IO还是写IO
                if not writeIoDataFlag:
                    readIoTgtStart = startIoData
                    readIoTgtNotback = notbackIoData
                else:
                    writeIoTgtStart = startIoData
                    writeIoTgtNotback = notbackIoData
            
            #读写的TGT的IO数据获取成功后退出
            if readIoTgtStart != "" and writeIoTgtStart != "":
                break
        else:
            pass
    
    #获取数据返回
    tgtIoInfo = (readIoTgtStart, readIoTgtNotback, writeIoTgtStart, writeIoTgtNotback) 
    return tgtIoInfo

# **************************************************************************** #
# 函数名称: getCurTimeTGTStartIoInfo
# 功能说明: 获取当前时间两端控制器的io数据
# 输入参数: tgtIoInfo
# 输出参数: NA
# 返 回 值: Start和not back的Io数据
# **************************************************************************** # 
def getCurTimeTGTStartIoInfo(ioInfoACtrl, ioInfoBCtrl, lang, log):
    
    flag = True
    errMsg = ""
    
    #A、B控制器的IO数据
    tgtIoInfoA = ()
    tgtIoInfoB = () 
    #返回信息列表
    resultIoInfoList = ()
    
    #判断developer回文是否正确返回
    if not re.search("Read IOs:", ioInfoACtrl, re.IGNORECASE):
        flag = False
        if lang == "zh":
            errMsg = u"Cli命令执行异常。"
        else:
            errMsg = "The command execute error."

        return (flag, errMsg, ())

    #判断developer回文是否正确返回
    if not re.search("Read IOs:", ioInfoBCtrl, re.IGNORECASE):
        flag = False
        if lang == "zh":
            errMsg = u"Cli命令执行异常。"
        else:
            errMsg = "The command execute error."

        return (flag, errMsg, ())
    
    #解析A控制器的IO数据
    tgtIoInfoA = getTGTIoInfo(ioInfoACtrl, lang)
    log.info("A Controller IO information:" + unicode(tgtIoInfoA))

    #查询A控制器数据的有效性
    if len(tgtIoInfoA) != 4:
        if lang == "zh":
            errMsg = u"获取A控制器的TGT的io数据失败。"
        else:
            errMsg = "Getting TGT io data of controller A failed."
        return (False, errMsg, ())

    for temp in tgtIoInfoA:
        if temp == "":
            if lang == "zh":
                errMsg = u"获取A控制器的TGT的io数据失败。"
            else:
                errMsg = "Getting TGT io data of controller A failed."
            return (False, errMsg, ())

    #解析B控制器的IO数据
    tgtIoInfoB = getTGTIoInfo(ioInfoBCtrl, lang)
    log.info("B Controller IO information:" + unicode(tgtIoInfoB))

    #查询B控制器数据的有效性
    if len(tgtIoInfoB) != 4:
        if lang == "zh":
            errMsg = u"获取B控制器的TGT的io数据失败。"
        else:
            errMsg = "Getting TGT io data of controller B failed."
        return (False, errMsg, ())

    for temp in tgtIoInfoB:
        if temp == "":
            if lang == "zh":
                errMsg = u"获取B控制器的TGT的io数据失败。"
            else:
                errMsg = "Getting TGT io data of controller b failed."
            return (False, errMsg, ())
 
    #判断A控制器的Notback数据是否均为0
    if tgtIoInfoA[1] != "0":
        flag = False
        if lang == "zh":
            errMsg += u"A控制器TGT模块的未返回的读IO个数不为0（" + unicode(tgtIoInfoA[1]) + u"）。\n"
        else:
            errMsg += "The TGT module's unreturned read IO number of controller A is not 0(" + unicode(tgtIoInfoA[1]) + ").\n"

    if tgtIoInfoA[3] != "0":
        flag = False
        if lang == "zh":
            errMsg += u"A控制器TGT模块的未返回的写IO个数不为0（" + unicode(tgtIoInfoA[3]) + u"）。\n"
        else:
            errMsg += "The TGT module's unreturned write IO number of controller A is not 0(" + unicode(tgtIoInfoA[3]) + ").\n"

    #判断B控制器的Notback数据是否均为0
    if tgtIoInfoB[1] != "0":
        flag = False
        if lang == "zh":
            errMsg += u"B控制器TGT模块的未返回的读IO个数不为0（" + unicode(tgtIoInfoB[1]) + u"）。\n"
        else:
            errMsg += "The TGT module's unreturned read IO number of controller B is not 0(" + unicode(tgtIoInfoB[1]) + ").\n"

    if tgtIoInfoB[3] != "0":
        flag = False
        if lang == "zh":
            errMsg += u"B控制器TGT模块的未返回的写IO个数不为0（" + unicode(tgtIoInfoB[3]) + u"）。\n"
        else:
            errMsg += "The TGT module's unreturned write IO number of controller B is not 0(" + unicode(tgtIoInfoB[3]) + ").\n"
    
    #如果已经检查失败，返回
    if not flag:
        return (False, errMsg, ())
    
    #Notback值已经检查，只需要返回Start的值
    resultIoInfoList = (tgtIoInfoA[0], tgtIoInfoA[2], tgtIoInfoB[0], tgtIoInfoB[2])
    return (True, "", resultIoInfoList)
    
# **************************************************************************** #
# 函数名称: checkCtrlServiceStop
# 功能说明: 查询两端控制器的业务是否完全停止
# 输入参数: ssh,lang
# 输出参数: NA
# 返 回 值: True,False
# **************************************************************************** # 
def checkCtrlServiceStop(ssh, lang, log):

    #进入此函数执行前，需保证ssh在developer命令模式下
    allRet = ''
    checkFlag = CheckedResult.PASS
    errMsg = ''

    #两端控制器查询数据
    allTgtioInfo = ()
    allTgtioInfoTmp = ()

    #查询IO信息
    ioInfoACtrl = ssh.execCmd("showio -c a")
    ioInfoBCtrl = ssh.execCmd("showio -c b")
    if lang == "zh":
        allRet = u"【第一次获取的io数据】\n"
    else:
        allRet = "[The io information for the first time]\n"
    allRet += ioInfoACtrl + '\n' + ioInfoBCtrl + '\n'

    #获取一次TGT的io信息
    iRet0 = getCurTimeTGTStartIoInfo(ioInfoACtrl, ioInfoBCtrl, lang, log)
    if not iRet0[0]:
        checkFlag = CheckedResult.NOTPASS
        errMsg = iRet0[1]
        return (checkFlag, allRet, errMsg)
    
    #获取两端控制器tgt的io信息
    allTgtioInfo = iRet0[2]
    log.info("First time two Controllers IO information:" + unicode(allTgtioInfo))
    
    #睡眠1分钟后再发送ddio命令
    sleep(60)
    ioInfoACtrl = ssh.execCmd("showio -c a")
    ioInfoBCtrl = ssh.execCmd("showio -c b")
    if lang == "zh":
        allRet += u"\n\n【第二次获取的io数据】\n"
    else:
        allRet += "\n\n[The io information for the second time]\n"
    allRet += ioInfoACtrl + '\n' + ioInfoBCtrl + '\n'

    #再次获取TGT的io信息
    iRet1 = getCurTimeTGTStartIoInfo(ioInfoACtrl, ioInfoBCtrl, lang, log)
    if not iRet1[0]:
        checkFlag = CheckedResult.NOTPASS
        errMsg = iRet1[1]
        return (checkFlag, allRet, errMsg)
    
    #获取两端控制器tgt的io信息
    allTgtioInfoTmp = iRet1[2]
    log.info("Second time two Controllers IO information:" + unicode(allTgtioInfoTmp))
    
    #判断两次获取的数据是否相同
    if allTgtioInfo != allTgtioInfoTmp:
        checkFlag = CheckedResult.NOTPASS
        if lang == "zh":
            errMsg = u"两次查询的IO数据不相同，存在未停止的业务。"
        else:
            errMsg = "The datas of two times are not same, the service is not stopped."
        return (checkFlag, allRet, errMsg)
    
    #检查通过，返回
    return (checkFlag, allRet, errMsg)

# **************************************************************************** #
# 函数名称: checkSingleCtrlServiceStop
# 功能说明: 查询单控的业务是否完全停止
# 输入参数: ssh,lang
# 输出参数: NA
# 返 回 值: True,False
# **************************************************************************** # 
def checkSingleCtrlServiceStop(ssh, lang, log):

    #进入此函数执行前，需保证ssh在developer命令模式下
    allRet = ''
    checkFlag = CheckedResult.PASS
    errMsg = ''

    ctrlInfo = ssh.execCmd("showctrlip")
    allRet += ctrlInfo + "\n"
    lineList = ctrlInfo.splitlines()
    if len(lineList) < 7:
        checkFlag = CheckedResult.NOTPASS
        if lang == "zh":
            errMsg = u"获取控制器信息失败"
        else:
            errMsg = "Failed to get controller information"
        return (checkFlag, allRet, errMsg)
    
    #获取单控控制器ID，并转化为小写
    controllerId = lineList[6].split()[0].strip().lower()
    #控制器查询数据
    allTgtioInfo = ()
    allTgtioInfoTmp = ()

    #查询IO信息
    ioInfoCtrl = ssh.execCmd("showio -c " + controllerId)
    if lang == "zh":
        allRet += u"【第一次获取的io数据】\n"
    else:
        allRet += "[The io information for the first time]\n"
    allRet += ioInfoCtrl + '\n'

    #获取一次TGT的io信息
    iRet0 = getCurTimeSingleCtrlTGTIoInfo(ioInfoCtrl, lang, log)
    if not iRet0[0]:
        checkFlag = CheckedResult.NOTPASS
        errMsg = iRet0[1]
        return (checkFlag, allRet, errMsg)
    
    #获取两端控制器tgt的io信息
    allTgtioInfo = iRet0[2]
    log.info("First time Controller IO information:" + unicode(allTgtioInfo))
    
    #睡眠1分钟后再发送ddio命令
    sleep(60)
    ioInfoCtrl = ssh.execCmd("showio -c " + controllerId)
    if lang == "zh":
        allRet += u"\n\n【第二次获取的io数据】\n"
    else:
        allRet += "\n\n[The io information for the second time]\n"
    allRet += ioInfoCtrl + '\n'

    #再次获取TGT的io信息
    iRet1 = getCurTimeSingleCtrlTGTIoInfo(ioInfoCtrl, lang, log)
    if not iRet1[0]:
        checkFlag = CheckedResult.NOTPASS
        errMsg = iRet1[1]
        return (checkFlag, allRet, errMsg)
    
    #获取两端控制器tgt的io信息
    allTgtioInfoTmp = iRet1[2]
    log.info("Second time Controller IO information:" + unicode(allTgtioInfoTmp))
    
    #判断两次获取的数据是否相同
    if allTgtioInfo != allTgtioInfoTmp:
        checkFlag = CheckedResult.NOTPASS
        if lang == "zh":
            errMsg = u"两次查询的IO数据不相同，存在未停止的业务。"
        else:
            errMsg = "The datas of two times are not same, the service is not stopped."
        return (checkFlag, allRet, errMsg)

    #检查通过，返回
    return (checkFlag, allRet, errMsg)

# **************************************************************************** #
# 函数名称: getCurTimeSingleCtrlTGTIoInfo
# 功能说明: 获取当前时间单端控制器的io数据
# 输入参数: tgtIoInfo
# 输出参数: NA
# 返 回 值: Start和not back的Io数据
# **************************************************************************** # 
def getCurTimeSingleCtrlTGTIoInfo(ioInfoCtrl, lang, log):
    
    flag = True
    errMsg = ""
    
    #单控制器的IO数据
    tgtIoInfo = ()
    #返回信息列表
    resultIoInfoList = ()
    
    #判断developer回文是否正确返回
    if not re.search("Read IOs:", ioInfoCtrl, re.IGNORECASE):
        flag = False
        if lang == "zh":
            errMsg = u"Cli命令执行异常。"
        else:
            errMsg = "The command execute error."

        return (flag, errMsg, ())
    
    #解析控制器的IO数据
    tgtIoInfo = getTGTIoInfo(ioInfoCtrl, lang)
    log.info("Controller IO information:" + unicode(tgtIoInfo))

    #查询控制器数据的有效性
    if len(tgtIoInfo) != 4:
        if lang == "zh":
            errMsg = u"获取控制器的TGT的io数据失败。"
        else:
            errMsg = "Getting TGT io data of controller failed."
        return (False, errMsg, ())

    for temp in tgtIoInfo:
        if temp == "":
            if lang == "zh":
                errMsg = u"获取控制器的TGT的io数据失败。"
            else:
                errMsg = "Getting TGT io data of controller failed."
            return (False, errMsg, ())

    #判断控制器的Notback数据是否均为0
    if tgtIoInfo[1] != "0":
        flag = False
        if lang == "zh":
            errMsg += u"控制器TGT模块的未返回的读IO个数不为0（" + unicode(tgtIoInfo[1]) + u"）。\n"
        else:
            errMsg += "The TGT module's unreturned read IO number of controller is not 0(" + unicode(tgtIoInfo[1]) + ").\n"

    if tgtIoInfo[3] != "0":
        flag = False
        if lang == "zh":
            errMsg += u"控制器TGT模块的未返回的写IO个数不为0（" + unicode(tgtIoInfo[3]) + u"）。\n"
        else:
            errMsg += "The TGT module's unreturned write IO number of controller is not 0(" + unicode(tgtIoInfo[3]) + ").\n"
    
    #如果已经检查失败，返回
    if not flag:
        return (False, errMsg, ())
    
    #Notback值已经检查，只需要返回Start的值
    resultIoInfoList = (tgtIoInfo[0], tgtIoInfo[2])
    return (True, "", resultIoInfoList)


def execute(dataDict):
    '''
    @summary: the entrance of main method, this check item is used to check Service Stopped
    @param dataDict: the dictionary of data which provided by tool framework
    @return: (pass status, CLI information, error message) as (boolean, string, string)
    '''

    allRet = ''
    checkFlag = CheckedResult.PASS
    errMsg = ''
    checkRetTmp = None
    developerFlag = False
    
    ssh = getSshObj(dataDict)
    log = getLogger(dataDict)
    lang = getLang(dataDict)

    #查询单双控模式
    isDoubleCtrlFlag = True
    sysInfo = ssh.execCmd("showsys")
    allRet += sysInfo + "\n"
    if not re.search("System Information", sysInfo, re.IGNORECASE):
        checkFlag = CheckedResult.NOTPASS
        if lang == "zh":
            errMsg = u"Cli命令执行异常。"
        else:
            errMsg = "The command execute error."    
        return (checkFlag, allRet, errMsg)
    #系统为单控
    if re.search("Single Controller", sysInfo, re.IGNORECASE):
        isDoubleCtrlFlag = False

    #安全红线后的设备切换到developer模式
    if not changeCli2Developer(dataDict):
        checkFlag = CheckedResult.WARN
        if lang == "zh":
            errMsg = u"进入developer模式失败，查询IO数据失败。失败的原因可能为：\n" \
                    + u"（1）添加设备时未输入developer密码。\n（2）添加设备时输入的developer密码无效。"
        else:
            errMsg = "Login to developer model failed, can not get I/O information.The reason of failure may be:\n" \
                    + "(1) Did not enter a developer password when adding the device.\n(2) The developer password entered is incorrect."
        return (checkFlag, allRet, errMsg)

    #单双控调用不同的处理方法
    if isDoubleCtrlFlag:
        checkRetTmp = checkCtrlServiceStop(ssh, lang, log)
    else:
        checkRetTmp = checkSingleCtrlServiceStop(ssh, lang, log)
    
    #获取返回值
    checkFlag = checkRetTmp[0]
    allRet += checkRetTmp[1] + "\n"
    errMsg = checkRetTmp[2]
    #退回到cli模式
    ssh.execCmd("exit")
        
    return (checkFlag, allRet, errMsg)
