# -*- coding: UTF-8 -*-
from frame.cli.cliFactory import CliConnection
from frame.tlv.tlvFactory import TlvConnection
from frame.tlv.tlvFactory import TlvConnectionSVP
from frame.common import jsonUtil
import os
import traceback
from cbb.frame.util.common import wrapAllExceptionLogged

SCRIPT_DEF_CONTEXT = "SCRIPT_DEF_CONTEXT"
RESOURCE_DEF_CONTEXT = "RESOURCE_DEF_CONTEXT"

CLI_CONNECTION = "CLI_CONNECTION"
TLV_CONNECTION = "TLV_CONNECTION"
SVP_CONNECTION = "SVP_CONNECTION"

ERROR_CODE_DEFINE = {
    "16797697":
        {
            "errMsg_zh":u"通信异常或系统繁忙。",
            "errMsg_en":"The communication is abnormal or the system is busy.",
            "suggestion_zh":u"",
            "suggestion_en":"",
        },
    "1073949185":
        {
            "errMsg_zh":u"与设备通信异常。",
            "errMsg_en":"Communicating with the device failed.",
            "suggestion_zh":u"请检查网络连接或设备状态是否正常。如果有任何疑问，请联系技术支持工程师协助处理。",
            "suggestion_en":"Please check that the network connection or the system is normal. If you have any problems, please contact technical support engineers for help.",
        },
    "TLV_CAN_NOT_CONNECT_CODE_SVP":
        {
            "errMsg_zh":u"与SVP通信异常。",
            "errMsg_en":"Communicating with the SVP failed.",
            "suggestion_zh":u"请检查SVP状态是否正常。如果有任何疑问，请联系技术支持工程师协助处理。",
            "suggestion_en":"Please check that the status of SVP is normal. If you have any problems, please contact technical support engineers for help.",
        },
    "1077936891":
        {
            "errMsg_zh":u"命令执行失败。",
            "errMsg_en":"Command execute failed.",
            "suggestion_zh":u"",
            "suggestion_en":"",
        },
    "1077949083":
        {
            "errMsg_zh":u"引擎无法监控。",
            "errMsg_en":"The engine can not be monitored.",
            "suggestion_zh":u"请查看系统中告警并按照告警修复建议进行处理。",
            "suggestion_en":"Please rectify the faults by referring to recommended actions.",
        },
    "1077949058":
        {
            "errMsg_zh":u"用户权限不足。",
            "errMsg_en":"The user does not have the permission.",
            "suggestion_zh":u"请联系管理员。",
            "suggestion_en": "Contact the administrator.",
        },
}
        
def getLang(context):
    '''
    @summary: 获取语言
    @param context: 上下文对象
    '''
    return context.get("lang")

def getLogger(context):
    '''
    @summary: 获取日志打印对象
    @param context: 上下文对象
    '''
    return context.get("logger")

def getSSH(context):
    '''
    @summary: 获取工具与阵列的SSH连接对象
    '''
    return context.get('ssh')

def getSFTP(context):
    '''
    @summary: 获取工具与阵列的SFTP连接对象
    '''
    return context.get('sftp')

def getDevType(context):
    '''
    @summary: 获取设备类型
    '''
    return getDevObj(context).getDeviceModel()

def getDevSN(context):
    '''
    @summary: 获取设备序列号Serial Number
    '''
    return getDevObj(context).getDeviceSerialNumber()

def getCurVersion(context):
    '''
    @summary: 获取系统当前版本
    @param context: 上下文对象
    '''
    return getDevObj(context).getProductVersion()

def getTargetVersion(context):
    '''
    @summary: 获取目标版本号
    @param context: 上下文对象
    '''
    return context.get("dev").getDestVersion()

def getScriptDefDict(context):
    '''
    @summary: 获取上下文自定义字典
    @param context: 上下文对象
    '''
    if isinstance(context, dict):
        if not context.has_key(SCRIPT_DEF_CONTEXT):
            context.setdefault(SCRIPT_DEF_CONTEXT, {})
    else:
        if not context.containsKey(SCRIPT_DEF_CONTEXT):
            context.put(SCRIPT_DEF_CONTEXT, {})
    return context.get(SCRIPT_DEF_CONTEXT)

def setItem(context, key, value):
    '''
    @summary: 往自定义字典中添加项
    @param context: 上下文对象
    @param key: 添加项的键
    @param value: 添加项的值
    '''
    scriptDefDict = getScriptDefDict(context)
    scriptDefDict[key] = value
    
    if isinstance(context, dict):
        context[SCRIPT_DEF_CONTEXT] = scriptDefDict.copy()
    else:
        context.put(SCRIPT_DEF_CONTEXT, scriptDefDict.copy())
    return

def removeItem(context, key):
    '''
    @summary: 删除自定义字典中的项
    @param context: 上下文对象
    @param key: 删除项的键
    '''
    scriptDefDict = getScriptDefDict(context)
    if scriptDefDict.has_key(key):
        scriptDefDict.pop(key)
    return

def getItem(context, key):
    '''
    @summary: 获取自定义字典中的项
    @param context: 上下文对象
    @param key: 项的键
    '''
    scriptDefDict = getScriptDefDict(context)
    return scriptDefDict.get(key, None)

def getConnectorFactoryObj(context):
    '''
    @summary: 获取上下文对象中创建连接的对象
    @param context: 上下文对象
    '''
    return context.get("sshFactory")#connectorFactory

def getDevObj(context):
    '''
    @summary: 获取上下文对象中dev对象
    @param context: 上下文对象
    '''
    return context.get("dev")

def getDevLoginUserName(context):
    '''
    @summary: 获取上下文对象中dev对象中的用户名
    @param context: 上下文对象
    '''
    dev = getDevObj(context)
    return dev.getLoginUser().getUserName()

def getDevLoginUserPwd(context):
    '''
    @summary: 获取上下文对象中dev对象中的的登录密码
    @param context: 上下文对象
    '''
    dev = getDevObj(context)
    return dev.getLoginUser().getPassword()

def getDevPort(context):
    '''
    @summary: 获取上下文对象中dev对象中的的登录端口
    @param context: 上下文对象
    '''
    dev = getDevObj(context)
    return dev.getPort()

def getTmpDir(context):
    '''
    @summary: 获取子工具运行根目录下的临时目录
    '''
    return context.get('tmpDir')

def getCurCtrlIp(context):
    '''
    @summary: 获取设备当前控制器IP
    @param context: 上下文对象
    '''
    return getDevObj(context).getIp()

def getDeveloperPwd(context):
    '''
    @summary: 获取developer密码
    @param context: 上下文对象
    '''
    return getDevObj(context).getDeveloperPwd()

def getTlvObj(context):
    '''
    @summary: 获取上下文对象中tlv对象
    @param context: 上下文对象
    '''
    return context.get("tlv")

def getUpgradeModel(context):
    '''
    @summary: 获取升级模式（在线或者离线），其值为"ONLINE"或"OFFLINE"
    @param context: 上下文对象
    '''
    return unicode(getDevObj(context).getUpgradeMode())

def initCliConnection(context):
    '''
    @summary: 初始化cli连接
    @param context: 上下文对象
    '''
    connectorFactoryObj = getConnectorFactoryObj(context)
    devObj = getDevObj(context)
    
    ip = devObj.get("ip")
    sshPort = int(devObj.get("sshPort"))
    
    conn = CliConnection(connectorFactoryObj, ip, sshPort)
    conn.create(devObj.get("user"), devObj.get("pawd"))
    
    setItem(context, CLI_CONNECTION, conn)
    return

def getCli(context):
    '''
    @summary: 获取cli连接
    @param context: 上下文对象
    '''
    conn = getItem(context, CLI_CONNECTION)
    if conn is None:
        initCliConnection(context)
        conn = getItem(context, CLI_CONNECTION)
        
    devObj = getDevObj(context)
    cli = conn.getCli(devObj.get("user"), devObj.get("pawd"))
    return cli


@wrapAllExceptionLogged(logger=None)
def destroyCliConnection(context):
    '''
    @summary: 销毁cli连接
    @param context: 上下文对象
    '''

    conn = getItem(context, CLI_CONNECTION)
    if conn is not None:
        conn.close()
        removeItem(context, CLI_CONNECTION)


def initTlvConnection(context):
    '''
    @summary: 初始化tlv连接
    @param context: 上下文对象
    '''
    connectorFactoryObj = getConnectorFactoryObj(context)
    devObj = getDevObj(context)
    
    ip = devObj.get("ip")
    tlvPort = devObj.get("tlvPort")
    
    conn = TlvConnection(connectorFactoryObj, ip, tlvPort, innerIps = getItem(context, "innerIps"))
    conn.create(devObj.get("user"), devObj.get("pawd"))
    
    setItem(context, TLV_CONNECTION, conn)
    return

def getTlv(context):
    '''
    @summary: 获取tlv连接
    @param context: 上下文对象
    '''
    conn = getItem(context, TLV_CONNECTION)
    if conn is None:
        initTlvConnection(context)
        conn = getItem(context, TLV_CONNECTION)
    
    devObj = getDevObj(context)
    tlv = conn.getTlv(devObj.get("user"), devObj.get("pawd"))
    return tlv

def getLoginUserName(context):
    '''
    @summary: 获取登录用户名
    '''
    return getDevObj(context).getLoginUser().getUserName()

def destroyTlvConnection(context):
    '''
    @summary: 销毁tlv连接
    @param context: 上下文对象
    '''
    try:
        conn = getItem(context, TLV_CONNECTION)
        if conn is not None:
            conn.close()
            removeItem(context, TLV_CONNECTION)
        return True
    except:
        return False

def initTlvConnectionSVP(context):
    '''
    @summary: 初始化SVP的tlv连接
    @param context: 上下文对象
    '''
    connectorFactoryObj = getConnectorFactoryObj(context)
    devObj = getDevObj(context)
    
    conn = TlvConnectionSVP(connectorFactoryObj, deviceType=devObj.get("type"), sn=devObj.get("sn"))
    conn.create()
    
    setItem(context, SVP_CONNECTION, conn)
    return

def getTlvSVP(context):
    '''
    @summary: 获取SVP的tlv连接
    @param context: 上下文对象
    '''
    conn = getItem(context, SVP_CONNECTION)
    if conn is None:
        initTlvConnectionSVP(context)
        conn = getItem(context, SVP_CONNECTION)
    
    tlv = conn.getTlv()
    return tlv

def destroyTlvConnectionSVP(context):
    '''
    @summary: 销毁SVP的tlv连接
    @param context: 上下文对象
    '''
    try:
        conn = getItem(context, SVP_CONNECTION)
        if conn is not None:
            conn.close()
            removeItem(context, SVP_CONNECTION)
        return True
    except:
        return False

def setResult(context, flag, errMsg="", suggestion=""):
    '''
    @summary: 往上下文对象中传递处理结果
    @param context: 上下文对象
    @param flag: 标识
    @param errMsg: 错误消息
    @param suggestion: 修复建议
    '''
    context["succ"] = flag
    context["errMsg"] = errMsg
    context["suggestion"] = suggestion
    return

def handleSuccess(context):
    '''
    @summary: 处理成功时，返回成功的结果给上下文对象
    @param context: 上下文对象
    '''
    setResult(context, True)
    return

def handleFailure(context, resultDict={"flag":False, "errMsg":"", "suggestion":""}):
    '''
    @summary: 处理失败时，返回失败的结果给上下文对象
    @param context: 上下文对象
    @param resultDict: 结果字典
    '''
    errMsg = resultDict.get("errMsg","")
    suggestion = resultDict.get("suggestion","")
    
    lang = getLang(context)
    if len(errMsg) == 0:
        if lang == "zh":
            errMsg = u"系统处理消息失败。"
        else:
            errMsg = "Failed to process the message."
    if len(suggestion) == 0:
        if lang == "zh":
            suggestion = u"请检查系统状态或稍后重试。如果有任何疑问，请联系技术支持工程师协助处理。"
        else:
            suggestion = "Please check the system status or try again later. If you have any problems, please contact technical support engineers for help."
    
    setResult(context, False, errMsg, suggestion)
    return

def handleException(context, exception):
    '''
    @summary: 处理异常时，返回失败的结果给上下文对象
    @param context: 上下文对象
    @param exception: 异常信息
    '''
    lang = getLang(context)
    removeItem(context, TLV_CONNECTION)
    removeItem(context, CLI_CONNECTION)
    expArgs = exception.args
    if expArgs is not None and len(expArgs) == 2:
        errCode = unicode(expArgs[0]).strip()
        keys = ERROR_CODE_DEFINE.keys()
        if errCode in keys:
            errInfo = ERROR_CODE_DEFINE.get(errCode, {})
            errMsg = errInfo.get("errMsg_%s" % lang, "")
            suggestion = errInfo.get("suggestion_%s" % lang, "")
            setResult(context, False, errMsg, suggestion)
            return

    handleFailure(context)
    return

def handleDetectException(context, exception):
    '''
    @summary: 处理异常时，返回失败的结果给上下文对象
    @param context: 上下文对象
    @param exception: 异常信息
    '''    
    handleException(context, exception)
    infoList = [context["errMsg"], context["suggestion"]]
    setCableDetectFailure(context, infoList)
    return

def getImportRootDir(context):
    '''
    @summary: 获取脚本的根目录
    @param context: 上下文对象
    '''
    return os.path.dirname(context.get("importRootDir"))

def getResourceDict(context):
    '''
    @summary: 将property资源文件转换为键值对字典
    @param context: 上下文对象
    '''
    scriptDefDict = getScriptDefDict(context)
    if not scriptDefDict.has_key(RESOURCE_DEF_CONTEXT):
        resourceFilePath = getResourecePath(context)
        resourceFile = open(resourceFilePath)
        contents = resourceFile.read().splitlines()
        resourceDict = {}
        for line in contents:
            fields = line.split("=")
            if len(fields) < 2:
                continue
            key = fields[0].strip()
            value = "=".join(fields[1:]).strip()
            resourceDict.setdefault(key, value)
        scriptDefDict.setdefault(RESOURCE_DEF_CONTEXT, resourceDict)
        
    return scriptDefDict.get(RESOURCE_DEF_CONTEXT)

def getResource(context, key):
    '''
    @summary: 根据property资源文件中的键，获取对应的值
    @param context: 上下文对象
    @param key: property资源文件中的键
    '''
    resourceDict = getResourceDict(context)
    return resourceDict.get(key, "--")

def getResourecePath(context):
    '''
    @summary: 返回property文件所在的路径
    @param context: 上下文对象
    '''
    lang = getLang(context)
    return os.path.join(getImportRootDir(context), "res", "language", "common_%s.properties" % lang)

def setCableDetectItem(context, checkItems):
    '''
    @summary: 设置上下文中检测项(限线缆检测)
    @param context: 上下文对象
    '''
    context["retData"] = jsonUtil.dictList2JsonArray(checkItems)
    return

def setCableDetectFailure(context, infoList):
    '''
    @summary: 返回上下文中检测项的检测结果(限线缆检测)
    @param context: 上下文对象
    '''
    dictList = []
    cnt = 0
    for info in infoList:
        dictList.append({"id":unicode(cnt),"title":info})
        cnt += 1
    context["retData"] = jsonUtil.dictList2JsonArray(dictList)
    handleFailure(context)
    return 

def createCliConnection(context, ip):
    '''
    @summary: 根据ip创建cli连接，支持跳转
    @param context: 上下文对象
    @param ip:创建cli连接的ip
    @return: 创建成功返回cli连接，否则为None 
    '''
    logger = getLogger(context)
    dev = getDevObj(context)
    
    #先保存原有ip，SN号，使用新ip建立连接后进行恢复
    originalIp = dev.getIp()    
    originalSN = dev.getDeviceSerialNumber()
    
    try:
        sn = originalSN + originalIp + ip    #生成临时SN用于建立cli连接，一个SN号建立一个连接（框架限制）
        
        dev.setIp(ip)
        dev.setDeviceSerialNumber(sn)
        connectorFactoryObj = getConnectorFactoryObj(context)
        sshPort = int(getDevPort(context))
        conn = CliConnection(connectorFactoryObj, ip, sshPort)
        cliConn = conn.createForwardConnetor(dev)
        return cliConn
    except:
        logger.info(unicode(traceback.format_exc()))
        return None
    finally:
        #恢复为原有ip，SN号
        dev.setIp(originalIp)
        dev.setDeviceSerialNumber(originalSN)

def getExtraDataDict(context):
    '''
    @summary: 获取额外数据字典，主要用于存放脚本执行时的临时数据
    @param context: 上下文对象
    '''
    return context.get('extraMap')

def getCtrlIpAndIdMap(context):
    '''
    @summary: 获取控制器的IP和ID(即Name)的映射关系
    @param context: 上下文对象
    '''
    return getExtraDataDict(context).get('ctrlIpAndIdMap')


def getCollectTmpDir(context):
    '''
    @summary: 获取当前设备存放文件临时目录
    '''
    tmpDir = getTmpDir(context)
    tmpFileName = getItem(context, DEFINED_CONTEXT.TMP_FILE_NAME)
    if not tmpFileName:
        return ""
    TmpDir = os.path.join(tmpDir, tmpFileName)
    return TmpDir           
 

