# -*- coding: UTF-8 -*-
import traceback
import re
from common import util
from common.constant import CheckedResult
from common.cmdRetManager import getCliRet
from common.contextUtil import getLang, getLogger,getSshObj


# modified 2013-12-16 begin
def getBBUNumDict(dataDict):
    '''
    @summary: 获取BBU标配个数
    @param dataDict: 数据字典上下文环境（框架提供）
    @return: 型号个数匹配字典{型号:个数} 
    '''
    cli = getSshObj(dataDict)
    ctrlNum = 0
    bbuNumDict = {}
    cliRet = cli.execCmd("show controller general")
    lines = cliRet.splitlines()

    for field in lines:
        field = field.replace(" ", "")
        if field.startswith("Controller:"):
            ctrlNum = ctrlNum + 1
            continue
    # 控制器BBU标准配置规格    
    if ctrlNum <= 2:
        bbuNumDict = {"S2600T":2, "S5500T":2, "S5600T":4, "S5800T":4, "S6800T":4}
    elif ctrlNum <= 4 and ctrlNum > 2:
        bbuNumDict = {"S5500T":4, "S5600T":8, "S5800T":8, "S6800T":8}
    return bbuNumDict
# modified 2013-12-16 end

# 需要优化的检查项  modified 20131125 Begin
def checkBbuNum(dataDict, cliRet4Bbu):
    '''
    @summary: 检测BBU个数是否达标
    @param dataDict: 数据字典上下文环境（框架提供）
    @param cliRet4Bbu: BBU状态回文
    @return: （通过标识flag，错误信息errMsg，回显结果cliRet） as (integer, string, string)
    '''
    cli = getSshObj(dataDict)
    flag = CheckedResult.PASS
    errMsg = ''
    log = getLogger(dataDict)
    # modified 2013-12-16 begin
    bbuNumDict = {}
    # modified 2013-12-16 end
    cliRet4SysInfo = cli.execCmd("show system general")
    cliRet4SysInfoList = cliRet4SysInfo.splitlines()
    lang = getLang(dataDict)
    pdtModel = ""
    #能查询到系统信息
    if re.search("System Name", cliRet4SysInfo, re.IGNORECASE):
        #获取产品型号
        for line in cliRet4SysInfoList:
            if re.search("Product Model", line, re.IGNORECASE):
                pdtModel = (line.split(":")[1]).strip()
                break
    else:
        for line in cliRet4SysInfoList:
            #系统没有运行在admin模式
            if line.startswith("-bash") or line.startswith("upgrade"):
                flag = CheckedResult.NOTPASS
                log.info("Sequence[bbuStatus] not in CLI mode, out!")
                if "zh" == lang:
                    errMsg = u"\n系统没有运行在admin模式。"
                else:
                    errMsg = "\nSystem is not in admin Mode."
                return (flag, cliRet4SysInfo, errMsg)
            #其它出现“Error:”字眼的信息视为不通过
            if re.search("Error:", line, re.IGNORECASE):
                flag = CheckedResult.NOTPASS
                log.info("There is an error in Sequence[bbuStatus]!")
                if "zh" == lang:
                    errMsg = u"\n获取信息出错。"
                else:
                    errMsg = "\nThere is an error in getting information."
                return (flag, cliRet4SysInfo, errMsg)
            
    #无法获取产品型号
    if "" == pdtModel:
        flag = CheckedResult.NOTPASS
        log.info("Sequence[bbuStatus] cannot get product model.")
        if "zh" == lang:
            errMsg += u"\n无法获取产品型号。"
        else:
            errMsg += "\nCannot get product model."
    else:
        #根据产品型号获取标配BBU个数
        standardBbuNum = 0
        # modified 2013-12-16 begin
        bbuNumDict = getBBUNumDict(dataDict)
        # modified 2013-12-16 end
        if bbuNumDict.has_key(pdtModel):
            standardBbuNum = bbuNumDict[pdtModel]
        
        #未查到标配BBU个数
        if 0 == standardBbuNum:
            flag = CheckedResult.NOTPASS
            log.info("Sequence[bbuStatus] The standard configuration number of BBU cannot be found.")
            if "zh" == lang:
                errMsg += u"\n未查到BBU标配个数。"
            else:
                errMsg += "\nThe standard configuration number of BBU cannot be found."
            return (flag, cliRet4SysInfo, errMsg)
        
        #获取真实BBU个数
        cliRet4BbuList = cliRet4Bbu.splitlines()
        count4Bbu = 0
        isCounterSwitchOn = False
        for line in cliRet4BbuList:
            #如果找到列分隔符"---"，就打开计数开关，并在下一行开始计数
            if re.search("---", line, re.IGNORECASE):
                isCounterSwitchOn = True
                continue
            #如果找到“:/>”，就关闭计数开关，并退出循环
            if re.search(":/>", line, re.IGNORECASE):
                isCounterSwitchOn = False
                break
            #如果是空行，则不计数
            if "" == line.strip():
                continue
            #如果计数开关打开，则开始计数
            if isCounterSwitchOn:
                count4Bbu += 1
        
        #比较真实BBU个数和标配BBU个数是否一致
        if count4Bbu < standardBbuNum:
            flag = CheckedResult.NOTPASS
            log.info("Sequence[bbuStatus] The number of BBU is not enough.")
            if "zh" == lang:
                errMsg += u"\nBBU个数不足。"
            else:
                errMsg += "\nThe number of BBU is not enough."
    return (flag, cliRet4SysInfo, errMsg)


def execute(dataDict):
    '''
    @summary: the entrance of main method, this check item is used to check BBU Status
    @param dataDict: the dictionary of data which provided by tool framework
    @return: (pass status, CLI information, error message) as (boolean, string, string)
    '''
    cliRet4Bbu = ""
    cliRet4SysInfo = ""
    try:
        flag = CheckedResult.PASS
        i = 0
        lang = getLang(dataDict)
        log = getLogger(dataDict)
        errMsg = ""
        cliRet4Bbu = getCliRet(dataDict, "show bbu general")
        # BBU不在位检查通过 modified 20131122 Begin
        #不能查到BBU状态信息
        if not (re.search("ID", cliRet4Bbu, re.IGNORECASE) and \
                re.search("Health", cliRet4Bbu, re.IGNORECASE) and \
                re.search("Running", cliRet4Bbu, re.IGNORECASE)):
            cliRet4BbuList = cliRet4Bbu.splitlines()
            for line in cliRet4BbuList:
                #系统不在admin模式
                if line.startswith("-bash") or line.startswith("upgrade"):
                    flag = CheckedResult.NOTPASS
                    log.info("Sequence[bbuStatus] not in CLI mode, out!")
                    if "zh" == lang:
                        errMsg = u"\n系统没有运行在admin模式。"
                    else:
                        errMsg = "\nSystem is not in admin Mode."
                    return (flag, cliRet4Bbu, errMsg)
                #其它出现“Error:”字眼的信息视为不通过
                if re.search("Error:", line, re.IGNORECASE):
                    flag = CheckedResult.NOTPASS
                    log.info("There is an error in Sequence[bbuStatus]!")
                    if "zh" == lang:
                        errMsg = u"\n获取信息出错。"
                    else:
                        errMsg = "\nThere is an error in getting information."
                    return (flag, cliRet4Bbu, errMsg)
                #BBU不在位
                if line.startswith("Command executed successfully."):
                    flag = CheckedResult.NOTPASS
                    log.info("Sequence[bbuStatus] BBU does not exist!")
                    if "zh" == lang:
                        errMsg = u"\nBBU不在位。"
                    else:
                        errMsg = "\nBBU does not exist."
                    return (flag, cliRet4Bbu, errMsg)
        else:
            #检测BBU个数是否达标
            retTuple = checkBbuNum(dataDict, cliRet4Bbu)
            flag = retTuple[0]
            cliRet4SysInfo = retTuple[1]
            errMsg = retTuple[2]
        
            #检测BBU状态是否正常
            BbuList = util.formatStr(cliRet4Bbu)
            for i in range(len(BbuList)):
                bbuId = BbuList[i].get("ID")
                bbuHealthStatus = BbuList[i].get("Health Status")
                bbuRunningStatus = BbuList[i].get("Running Status")
                if "Normal" != bbuHealthStatus:
                    flag = CheckedResult.NOTPASS
                    tuple2BReplaced = (bbuId, bbuHealthStatus)
                    log.info("Sequence[bbuStatus] health status of BBU(%s) is abnormal(%s)" % tuple2BReplaced)
                    if "zh" == lang:
                        errMsg += u"\nBBU[%s]健康状态异常（%s）。" % tuple2BReplaced
                    else:
                        errMsg += "\nThe health status of BBU[%s] is abnormal(%s)." % tuple2BReplaced
                        
                if bbuRunningStatus.lower() not in  ["online", "charging", "discharging"]:
                    flag = CheckedResult.NOTPASS
                    tuple2BReplaced = (bbuId, bbuRunningStatus)
                    log.info("Sequence[bbuStatus] running status of BBU(%s) is abnormal(%s)" % tuple2BReplaced)
                    if "zh" == lang:
                        errMsg += u"\nBBU[%s]运行状态异常（%s）。" % tuple2BReplaced
                    else:
                        errMsg += "\nThe running status of BBU[%s] is abnormal(%s)." % tuple2BReplaced
            
        return (flag, cliRet4Bbu + cliRet4SysInfo, errMsg)
        # BBU不在位检查通过 modified 20131122 End
        # 需要优化的检查项  modified 20131125 End
    except Exception:
        log.error("Sequence[bbuStatus] catch except of trace back:" + unicode(traceback.format_exc()))
        flag = CheckedResult.NOTPASS
        if "zh" == lang:
            errMsg = u"\n查询结果异常。"
        else:
            errMsg = "\nQuery result is abnormal."
        return (flag, cliRet4Bbu + cliRet4SysInfo, errMsg)
