# -*- coding: UTF-8 -*-
import traceback
import re
from common import util
from common.constant import CheckedResult
from common.cmdRetManager import getCliRet
from common.contextUtil import getLang, getLogger


# 需要优化的检查项  modified 20131125 Begin
def getEnclosureList(cliRet4Enclosure, dataDict):
    '''
    @summary: 获取框（或者PCIE交换机）列表
    @param cliRet4Enclosure: 框（或者PCIE交换机）的cli回文
    @param dataDict: 数据字典上下文环境（框架提供）
    @return: （框（或者PCIE交换机）列表，通过标识flag，错误信息errMsg） as (list, integer, string)
    '''
    lang = getLang(dataDict)
    log = getLogger(dataDict)
    flag = CheckedResult.PASS
    errMsg = ""
    enclosureList = []
    #能够查询到框（或者PCIE交换机）信息
    if re.search("ID", cliRet4Enclosure, re.IGNORECASE) and re.search("Health", cliRet4Enclosure, re.IGNORECASE):
        cliRet4EnclosureList = cliRet4Enclosure.splitlines()
        isSwitchOn = False
        for line in cliRet4EnclosureList:
            #找到"---"后打开开关，并在下一行开始
            if re.search("---", line, re.IGNORECASE):
                isSwitchOn = True
                continue
            #找到":/>"则退出
            if re.search(":/>", line, re.IGNORECASE):
                break
            #如果是空行，则继续
            if "" == line.strip():
                continue
            #开关打开，表示有效行（即在"---"和":/>"之间的非空白行）
            if isSwitchOn:
                enclosureList.append(line.strip().split()[0])
        return (enclosureList, flag, errMsg)
    else:
        cliRet4EnclosureList = cliRet4Enclosure.splitlines()
        for line in cliRet4EnclosureList:
            #系统没有运行在admin模式
            if line.startswith("-bash") or line.startswith("upgrade"):
                flag = CheckedResult.NOTPASS
                log.info("Sequence[fanStatus] not in CLI mode, out!")
                if "zh" == lang:
                    errMsg = u"\n系统没有运行在admin模式。"
                else:
                    errMsg = "\nSystem is not in admin Mode."
                return (enclosureList, flag, errMsg)
            #其它出现“Error:”字眼的信息视为不通过
            if re.search("Error:", line, re.IGNORECASE):
                flag = CheckedResult.NOTPASS
                log.info("There is an error in Sequence[fanStatus]!")
                if "zh" == lang:
                    errMsg = u"\n获取信息出错。"
                else:
                    errMsg = "\nThere is an error in getting information."
                log.info(enclosureList, flag, errMsg)
        #可能产生的其它错误信息
        flag = CheckedResult.NOTPASS
        log.info("There is an error in Sequence[fanStatus]!")
        if "zh" == lang:
            errMsg = u"\n获取信息出错。"
        else:
            errMsg = "\nThere is an error in getting information."
        return (enclosureList, flag, errMsg)


def checkFanNumAndStatus(cliRet4Fan, enclosureList, dataDict):
    '''
    @summary: 检查风扇个数和状态是否达标
    @param cliRet4Fan: 风扇cli回文
    @param enclosureList: 框（或者PCIE交换机）列表
    @param dataDict: 数据字典上下文环境（框架提供）
    @return: （通过标识flag，错误信息errMsg） as (integer, string)
    '''
    lang = getLang(dataDict)
    log = getLogger(dataDict)
    flag = CheckedResult.PASS
    errMsg = ""
    enclosureId = ""
    enclosureIdDict = {}
    
    #初始化框（或者PCIE交换机）ID字典
    for enclosure in enclosureList:
        enclosureIdDict[enclosure] = 0
    
    #能够查询到数据
    if re.search("ID", cliRet4Fan, re.IGNORECASE) and re.search("Health", cliRet4Fan, re.IGNORECASE):
        
        fanList = util.formatList(cliRet4Fan)
        for fanDetail in  fanList:
            fanID = fanDetail.get("ID")
            enclosureId = fanID.split(".")[0]
            #记录框中风扇的数量
            enclosureIdDict[enclosureId] = enclosureIdDict.get(enclosureId, 0) + 1
            
            healthStatus = fanDetail.get("Health Status")
            runningStatus = fanDetail.get("Running Status")
            
            if healthStatus != "Normal" or runningStatus != "Running":
                flag = CheckedResult.NOTPASS
                tuple2BReplaced = (fanID, healthStatus, runningStatus)
                if "zh" == lang:  
                    errMsg += u"\n风扇[%s]状态异常（Health Status：%s，Running Status：%s）。" % tuple2BReplaced
                else:
                    errMsg += "\nThe status of fan[%s] is abnormal(Health Status: %s, Running Status: %s)." % tuple2BReplaced
                log.info("\nSequence[fanStatus] inspect not pass.")
            
        #判断风扇个数是否达标：每个框（或PCIE）的风扇个数大于等于2为达标
        for enclosure in enclosureList:
            if int(enclosureIdDict.get(enclosure)) < 2:
                flag = CheckedResult.NOTPASS
                log.info("Sequence[fanStatus] the number of fan is not enough")
                if "zh" == lang:
                    errMsg += u"\n框[%s]的风扇数量不足。" % enclosure
                else:
                    errMsg += "\nThe number of fan in enclosure[%s] is not enough." % enclosure
        return (flag, errMsg)
    
    #回文结果异常分支
    else:
        cliRet4FanList = cliRet4Fan.splitlines()
        for line in cliRet4FanList:
            #系统没有运行在admin模式
            if line.startswith("-bash") or line.startswith("upgrade"):
                flag = CheckedResult.NOTPASS
                log.info("Sequence[fanStatus] not in CLI mode, out!")
                if "zh" == lang:
                    errMsg = u"\n系统没有运行在admin模式。"
                else:
                    errMsg = "\nSystem is not in admin Mode."
                return (flag, errMsg)
            #其它出现“Error:”字眼的信息视为不通过
            if re.search("Error:", line, re.IGNORECASE):
                flag = CheckedResult.NOTPASS
                log.info("There is an error in Sequence[fanStatus]!")
                if "zh" == lang:
                    errMsg = u"\n获取信息出错。"
                else:
                    errMsg = "\nThere is an error in getting information."
                return (flag, errMsg)
            #风扇不在位
            if line.startswith("Command executed successfully."):
                flag = CheckedResult.NOTPASS
                log.info("Sequence[fanStatus] fan is not on the place!")
                if "zh" == lang:
                    errMsg = u"\n风扇不在位。"
                else:
                    errMsg = "\nFan does not exist."
                return (flag, errMsg)
        #可能产生的其它错误信息
        flag = CheckedResult.NOTPASS
        log.info("There is an error in Sequence[fanStatus]!")
        if "zh" == lang:
            errMsg = u"\n获取信息出错。"
        else:
            errMsg = "\nThere is an error in getting information."
        return (flag, errMsg)

def execute(dataDict):
    '''
    @summary: the entrance of main method, this check item is used to check Fan Status
    @param dataDict: the dictionary of data which provided by tool framework
    @return: (pass status, CLI information, error message) as (boolean, string, string)
    '''
    flag = CheckedResult.PASS
    cliRet = ""
    errMsg = ""
    lang = "en" #语言默认值设为英文
    try:
        lang = getLang(dataDict)
        log = getLogger(dataDict)
        
        #获取框（或者PCIE交换机）回文
        cliRet4Enclosure = getCliRet(dataDict, "show enclosure")
        
        #获取框ID信息
        enclosureInfoTuple = getEnclosureList(cliRet4Enclosure, dataDict)
        enclosureList = enclosureInfoTuple[0]
        flag4Enclosure = enclosureInfoTuple[1]
        errMsg4Enclosure = enclosureInfoTuple[2]
        #无法查询到框（或者PCIE交换机）信息，则直接返回
        if 0 == len(enclosureList):
            return (flag4Enclosure, cliRet4Enclosure, errMsg4Enclosure)
        
        #获取风扇状态回文
        cliRet4Fan = getCliRet(dataDict, "show fan")
        
        #检查风扇个数和状态是否达标
        fanNumAndStatusCheckRetTuple = checkFanNumAndStatus(cliRet4Fan, enclosureList, dataDict)
        flag4FanNumAndStatusCheck = fanNumAndStatusCheckRetTuple[0]
        errMsg4FanNumAndStatusCheck = fanNumAndStatusCheckRetTuple[1]
        
        #返回结果
        flag = flag4Enclosure and flag4FanNumAndStatusCheck
        cliRet = cliRet4Enclosure + cliRet4Fan
        errMsg = errMsg4Enclosure + errMsg4FanNumAndStatusCheck
        return (flag, cliRet, errMsg)
        
        # 需要优化的检查项  modified 20131125 End
    except Exception:
        log.error("Sequence[fanStatus] catch except of trace back:" + unicode(traceback.format_exc()))
        flag = CheckedResult.NOTPASS
        if "zh" == lang:
            errMsg = u"\n查询结果异常。"
        else:
            errMsg = "\nQuery result is abnormal."
        return (flag, cliRet, errMsg)
