# -*- coding: UTF-8 -*-

from frameone.util import contextUtil
from frameone.cli import cliUtil
from frameone.util import baseUtil
from frameone.base.constants import CheckStatus
from frameone.base.exception import CliCmdException, ErrorCodeSet, ErrorCode
import traceback

'''
=表示精确匹配
@表示以XX开头
'''
RULE_DICT = {
    "condition1" :{
        "model" : ["=HUC101818CS4200", "=HUC101812CS4200"],
        "version" : (None, "AAG4")
    },
    "condition2" :{
        "model" : ["=HUS156060VLS600"],
        "version" : (None, "A8G0")
    },
    "condition3" :{
            "model" : ["@HSSD-D5"],
            "version" : (None, "2142")
        },
    "condition4" :{
            "model" : ["@HSSD-D3J"],
            "version" : (None, "9203")
        },
    "condition5" :{
            "model" : ["@HSSD-D322XAM", "@HSSD-D322ZAM", "@HSSD-D3220AM"],
            "version" : (None, "5506")
        }
}

def execute(context):
    return DiskFirmwareCheck(context).check()

class CheckRule():
    def __init__(self, devType, hasNas):
        self.devType = devType
        self.hasNas = hasNas

    def checkIsMeetRule(self, diskInfo):
        self.diskModel = diskInfo.get("Model", "")
        self.fwVersion = diskInfo.get("Firmware Version", "")
        if self.fwVersion == "" or self.diskModel == "":
            return False
        ret = self.meetCondition("condition1")
        if not baseUtil.isHighEndDev(self.devType) and not baseUtil.isHvsDev(self.devType):
            ret = ret or self.meetCondition("condition2")
        if self.hasNas:
            ret = ret or self.meetCondition("condition3") \
                  or self.meetCondition("condition4") or self.meetCondition("condition5")
        return ret

    def meetCondition(self, id):
        rule = RULE_DICT.get(id)
        models = rule.get("model")
        versions = rule.get("version")
        startVersion = versions[0]
        endVersion = versions[1]
        if not ((startVersion is None or self.fwVersion > startVersion)
                and (endVersion is None or self.fwVersion < endVersion)):
            return False
        for model in models:
            if model.startswith("="):
                model = model[1:]
                if self.diskModel == model:
                    return True
            elif model.startswith("@"):
                model = model[1:]
                if self.diskModel.startswith(model):
                    return True
        return False


class DiskFirmwareCheck():
    def __init__(self, context):
        self.logger = contextUtil.getLogger(context)
        self.cli = contextUtil.getCLI(context)
        self.lang = contextUtil.getLang(context)
        self.hasNas = False
        self.devType = contextUtil.getDevType(context)
        self.checkResults = []
        self.cliRets = []

    def check(self):
        try:
            self.logger.info("begin to DiskFirmwareCheck")
            diskInfos = self.queryDiskInfo()
            self.hasNas = self.hasFileSystem()
            checkRule = CheckRule(self.devType, self.hasNas)

            # 检查每一块硬盘
            for diskInfo in diskInfos:
                result = {}
                isMeet = checkRule.checkIsMeetRule(diskInfo)
                if isMeet:
                    result["ID"] = diskInfo.get("ID")
                    self.checkResults.append(result)
            if len(self.checkResults) == 0:
                return CheckStatus.PASS, "\n".join(self.cliRets), ""

            # 不通过\n 固件版本过低的硬盘如下：\n DAE000.0; DAE000.0; DAE000.0
            notpassedDiskIds = map(lambda x: x.get("ID"), self.checkResults)
            errMsg = baseUtil.getMsg(self.lang, "diskFwCheck.nopass.desc", ", ".join(notpassedDiskIds))
            return CheckStatus.WARNING, "\n".join(self.cliRets), errMsg
        except:
            self.logger.error("DiskFirmwareCheck exception.%s" % traceback.format_exc())
            return CheckStatus.NOCHECK, "\n".join(self.cliRets), baseUtil.getMsg(self.lang, "query.result.abnormal")


    def queryDiskInfo(self):
        cmd = "show disk general |filterColumn include columnList=ID,Manufacturer,Model,Firmware\sVersion"
        V1R1C00_cmd = r"show disk general |filterColumn include " \
                      r"colunmList=ID,Manufacturer,Model,Firmware\sVersion"
        status, cliRet, _ = cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        if not cliUtil.hasCliExecPrivilege(cliRet):
            status, cliRet, _ = cliUtil.excuteCmdInCliMode(self.cli,
                                                           V1R1C00_cmd, True,
                                                           self.lang)
        self.cliRets.append(cliRet)

        if cliUtil.queryResultWithNoRecord(cliRet):
            return []

        diskInfos = cliUtil.getHorizontalCliRet(cliRet)
        if status != True or not diskInfos:
            raise CliCmdException(ErrorCode(ErrorCodeSet.CLI_EXECUTE_CMD_FAILED))

        return diskInfos

    def hasFileSystem(self):
        cmd = "show file_system general"
        status, cliRet, _ = cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        self.cliRets.append(cliRet)
        # 不支持文件系统命令，无FS
        if not cliUtil.hasCliExecPrivilege(cliRet):
            return False

        # 命令执行成功，没有文件系统
        if cliUtil.queryResultWithNoRecord(cliRet):
            return False

        # 命令执行出错
        if status != True:
            raise CliCmdException(ErrorCode(ErrorCodeSet.CLI_EXECUTE_CMD_FAILED))

        # 其他，表示有FS
        return True
