# -*- coding: UTF-8 -*-

from java.lang import Throwable
from business.common import common
from frameone.base.constants import CheckStatus
from cbb.frame.context import contextUtil
from cbb.frame.cli import cliUtil
from cbb.frame.cli.execute_on_all_controllers import ExecuteOnAllControllers
from cbb.frame.cli.execute_on_all_controllers import ExeOnAllCtrlException
from cbb.frame.cli.execute_on_all_controllers import ExeOnAllCtrlContext
from cbb.frame.cli.execute_on_all_controllers import FuncResult
from cbb.frame.cli.execute_on_all_controllers import ResultType


def execute(context):
    """
    检查系统盘固件版本
    :param context:
    :return:
    """

    devType = contextUtil.getDevType(context)
    systemVersion = contextUtil.getCurVersion(context)
    logger = contextUtil.getLogger(context)

    logger.info("begin SystemDiskCheck")
    # 检查型号和版本
    if devType != '2200 V3' or systemVersion != 'V300R005C00SPC300':
        return CheckStatus.PASS, "No information", ""

    try:
        # 利用装饰器检查所有控制器
        check_result = _system_disk_check(ExeOnAllCtrlContext(context))
        return trans_ExeOnAllCtrl_result_to_evalu_result(check_result)
    except ExeOnAllCtrlException as exception:
        lang = contextUtil.getLang(context)
        common_err_msg = common.getMsg(lang, "query.result.abnormal")
        logger.error(exception.exception_msg)
        return CheckStatus.NOCHECK, "No information", common_err_msg


def trans_ExeOnAllCtrl_result_to_evalu_result(check_result):
    trans_dict = {ResultType.SUCCESS: CheckStatus.PASS,
                  ResultType.FAILED: CheckStatus.NOTPASS,
                  ResultType.NOT_FINISHED: CheckStatus.NOCHECK,
                  ResultType.NOT_SUPPORT: CheckStatus.NO_SUPPORT,
                  ResultType.WARNING: CheckStatus.WARNING}
    return trans_dict.get(check_result.result_type), \
        check_result.cli_ret, check_result.err_msg


@ExecuteOnAllControllers
def _system_disk_check(context):
    """
    # 检查系统盘（每个控制器都检查）
    :param context:
    :return:
    """
    return SystemDiskCheck(context).check()


class SystemDiskCheck:
    """
    检查系统盘固件版本
    """

    def __init__(self, context):
        self.logger = context.logger
        self.cli = context.dev_info.cli
        self.lang = context.lang
        self.cli_rets = []
        self.err_msgs = []
        self.common_err_msg = common.getMsg(self.lang, "query.result.abnormal")

    def _get_cli_ret(self):
        """
        返回原始信息
        :return: cliRet
        """
        if self.cli_rets:
            return "\n".join(self.cli_rets)
        else:
            return "No information"

    def _get_err_msg(self):
        """
        返回错误信息
        :return: cliRet
        """
        return "\n".join(self.err_msgs)

    def check(self):
        """
        检查升级风险（功能入口）
        :return:
        """

        try:
            # 进入diagnose模式
            flag, cli_ret, _ = cliUtil.enterDebugModeFromCliMode(self.cli,
                                                                 self.lang)
            self.cli_rets.append(cli_ret)
            if not flag:
                return FuncResult(ResultType.NOT_FINISHED, self._get_cli_ret(),
                                  self.common_err_msg)

            # 获取所有硬盘信息
            get_all_disk_cmd = "sddebug show disk"
            cmd_ret = cliUtil.execCliCmd(self.cli, get_all_disk_cmd, True)
            self.cli_rets.append(cmd_ret)
            all_disk_info = cliUtil.getHorizontalCliRet(cmd_ret)

            # 获取系统盘ID
            system_disk_id_list = []
            for disk_info in all_disk_info:
                disk_id = disk_info.get("SDID", "")
                disk_name = disk_info.get("BlkDev", "")
                if disk_id.isdigit() and int(disk_id) < 64 and disk_name:
                    system_disk_id_list.append(disk_name)
                    break
            if not system_disk_id_list:
                return FuncResult(ResultType.NOT_FINISHED, self._get_cli_ret(),
                                  self.common_err_msg)

            # 切换到minisystem执行disktool命令
            switch_flag, cli_ret = cliUtil.enterMinisystemFromDebugMode(
                self.cli,
                self.lang)
            self.cli_rets.append(cli_ret)
            if not switch_flag:
                return FuncResult(ResultType.NOT_FINISHED, self._get_cli_ret(),
                                  self.common_err_msg)

            # 获取系统盘Model和Firmware
            flag = ResultType.SUCCESS
            for disk_name in system_disk_id_list:
                system_disk_ret = \
                    cliUtil.execCliCmd(self.cli,
                                       "disktool -f i /dev/%s" % disk_name,
                                       True)
                self.cli_rets.append(system_disk_ret)
                disk_info_list = cliUtil.getVerticalCliRet(system_disk_ret)
                if not disk_info_list:
                    return FuncResult(ResultType.NOT_FINISHED,
                                      self._get_cli_ret(), self.common_err_msg)
                system_disk_info = disk_info_list[0]
                disk_model = system_disk_info.get("Device Model", "")
                disk_fw_version = system_disk_info.get("Firmware Version", "")
                # 系统盘信息异常
                if not (disk_model and disk_fw_version):
                    flag = ResultType.NOT_FINISHED
                    self.err_msgs.append(
                        common.getMsg(self.lang, "query.result.abnormal"))
                    break
                # 判断系统盘风险
                if disk_model == "RTMMB016VBV4BWXY" and \
                        disk_fw_version == "T1109":
                    flag = ResultType.FAILED
                    self.err_msgs.append(
                        common.getMsg(self.lang, "system_disk.check.nopass"))
                    # 只要有1块系统盘有风险，直接返回
                    break
            return FuncResult(flag, self._get_cli_ret(), self._get_err_msg())
        except Exception as e:
            self.logger.error(
                "SystemDiskCheck exception.%s" % str(e))
            return FuncResult(ResultType.NOT_FINISHED, self._get_cli_ret(),
                              self.common_err_msg)
        except Throwable as throwable:
            self.logger.error(
                "SystemDiskCheck exception.%s" % str(throwable))
            return FuncResult(ResultType.NOT_FINISHED, self._get_cli_ret(),
                              self.common_err_msg)
        finally:
            # 退出到cli模式
            ret = cliUtil.enterCliModeFromSomeModel(self.cli, self.lang)
            self.cli_rets.append(ret[1])
            self.logger.info(
                "enter cli mode from some model ret is %s" % str(ret))
            # 退出失败后为不影响后续检查项重新连接cli
            if not ret[0]:
                cliUtil.reConnectionCli(self.cli, self.logger)
