# -*- coding: UTF-8 -*-
from frameone.util import contextUtil, baseUtil
from frameone.base.constants import CheckStatus, CheckResult
from frameone.adapter.resourceService import ResrouceFactory
from frameone.cli import cliUtil
from frameone.base import config
from frameone.base.exception import UnCheckException
import traceback

# 涉及的问题版本及补丁版本
InvolvedSysVersion2HotpatchDict = {"V300R006C10SPC100":"--", "V500R007C00SPC100":"--", "V300R001C21":"--"}

# 执行入口
def execute(dataDict):
    checkRet = FailoverMemCheck(dataDict).execute()
    return checkRet.toScriptResult()

class FailoverMemCheck():
    def __init__(self, context):
        self.context = context
        self.originInfos = []
        self.logger = contextUtil.getLogger(context)
        self.lang = contextUtil.getLang(context)
        self.commonMsg = baseUtil.getMsg(self.lang, "query.result.abnormal")

    def execute(self):
        self.logger.info("enter failover mem excute.")
        try:
            # 标准1：特定版本才检查，其他不涉及
            currentVersion = contextUtil.getCurVersion(self.context)
            currentVersion = baseUtil.formatVersion(currentVersion)
            self.logger.info("currentVersion=%s" % currentVersion)
            if currentVersion not in InvolvedSysVersion2HotpatchDict.keys():
                self.logger.info("The device not need check failover memory.")
                return CheckResult(CheckStatus.NO_SUPPORT, "", "")

            # 标准2：如果版本有特定补丁，检查通过
            self.ssh = contextUtil.getCLI(self.context)
            neededHotPathVersion = InvolvedSysVersion2HotpatchDict.get(currentVersion)
            if neededHotPathVersion != "--":
                currentHotPatchVersion = self.getHotpathVersion()
                if currentHotPatchVersion >= neededHotPathVersion:
                    return CheckResult(CheckStatus.PASS, "", "")

            # 标准3：未打开端口漂移开关，报通过
            if not self.isFailoverSwitchOn():
                return CheckResult(CheckStatus.PASS, self.originInfos, "")

            # 标准4：系统剩余内存-25*端口个数 > 升级所需内存，则检查通过，否则，检查不通过
            remainMemDict = self.getCtrlRemainMem()
            neededMem = self.getNeededMemory()
            failoverIpDict = self.getFailoverIpNum()
            status = CheckStatus.PASS
            errMsg = ""
            for nodeId, remainMem in remainMemDict.items():
                # 该控端口个数
                failoverIpNum = failoverIpDict.get(nodeId, 0)
                failoverMem = 25 * failoverIpNum
                if int(remainMem) - failoverMem >= neededMem:
                    errMsg += baseUtil.getMsg(self.lang, "failover.mem.pass", nodeId)
                else:
                    status = CheckStatus.NOTPASS
                    errMsg += baseUtil.getMsg(self.lang, "failover.mem.notpass", (nodeId, remainMem, neededMem, failoverMem))

            return CheckResult(status, self.originInfos, errMsg)
        except:
            self.logger.error("failover memory check exception.[%s]" % traceback.format_exc())
            return CheckResult(CheckStatus.NOCHECK, self.originInfos, self.commonMsg)


    # 方法1:获取对应型号设备升级所需内存
    def getNeededMemory(self):
        devModel = contextUtil.getDevType(self.context)
        minMemory = 270
        if devModel in config.MEMORY_DEVS_180:
            minMemory = 180
        elif devModel in config.MEMORY_DEVS_200:
            minMemory = 200
        elif devModel in config.MEMORY_DEVS_400:
            minMemory = 400
        self.originInfos.append("current system needed memery is %s" % minMemory)
        self.logger.info("current system needed memery is %s" % minMemory)
        return minMemory

    # 方法2：获取当前热补丁版本
    def getHotpathVersion(self):
        cmd = "show upgrade package"
        (ok, cliRet, errorMsg) = cliUtil.excuteCmdInCliMode(self.ssh, cmd, True, self.lang)
        self.originInfos.append(cliRet)
        if ok != True:
            self.logger.info("the hotpatch version query error.")
            raise UnCheckException("excute cmd failed. cmd=%s" % cmd)
        beginIndex = cliRet.find('HotPatch Version')
        if beginIndex == -1:
            raise UnCheckException("not find hot patch version.")
        cliRetDictList = cliUtil.getHorizontalCliRet(cliRet[beginIndex:])
        if not cliRetDictList:
            raise UnCheckException("hot patch version parse error.")
        hotpatchVersion = cliRetDictList[0].get('Current Version')
        self.logger.info("the hotpatch version=%s" % hotpatchVersion)
        return hotpatchVersion

    # 方法3：查询端口漂移开关状态
    def isFailoverSwitchOn(self):
        cmd = "show logical_port failover_switch service_type=SAN"
        (ok, cliRet, errorMsg) = cliUtil.excuteCmdInDeveloperMode(self.ssh, cmd, True, self.lang)
        self.originInfos.append(cliRet)
        if not cliUtil.hasCliExecPrivilege(cliRet):
            return False
        if "off" in cliRet.lower():
            return False
        if "on" in cliRet.lower():
            return True
        raise UnCheckException("the echo not expected.")

    # 获取每个控制器剩余内存
    def getCtrlRemainMem(self):
        self.resrouceFactory = ResrouceFactory(self.context)
        recs = self.resrouceFactory.executeInterface("excuteDiagnoseCmd", "upd sysresourcecheck", 2)
        self.logger.info("excute tlv upd sysresourcecheck,recs=%s" % recs)
        ret = {}
        for rec in recs:
            nodeId = self.resrouceFactory.executeInterface("getParamStrValue", rec, 0)
            result = self.resrouceFactory.executeInterface("getParamStrValue", rec, 1)
            results = result.split(",")
            if len(results) == 2 and results[1].isdigit():
                ret[nodeId] = results[1].strip()
            else:
                raise UnCheckException("getCtrlRemainMem echo error.")
        self.originInfos.append("tlv cmd 'upd sysresourcecheck' result is \n %s" % ret)
        return ret

    # 方法4：获取每个控制器端口漂移的IP个数
    def getFailoverIpNum(self):
        ret = {}
        cmd = "show port general logic_type=Host_Port physical_type=ETH"
        (ok, cliRet, errorMsg) = cliUtil.excuteCmdInCliMode(self.ssh, cmd, True, self.lang)
        self.originInfos.append(cliRet)
        if cliUtil.queryResultWithNoRecord(cliRet):
            self.logger.info("no failover port.")
            return ret
        hostports = cliUtil.getHorizontalCliRet(cliRet)
        if len(hostports) == 0:
            raise UnCheckException("echo parse error, cmd=%s" % cmd)
        for port in hostports:
            if port.get("Running Status", "").lower() != "link up":
                continue
            ipv4 = port.get("IPv4 Address", "").strip()
            ipv6 = port.get("IPv6 Address", "").strip()
            ctrlId = self.__getCtrlIdByPortId(port.get("ID"))
            if ctrlId == "":
                self.originInfos.append("can not find ctrl by port location.location=%s" % port.get("ID"))
                continue
            if ctrlId not in ret:
                ret[ctrlId] = 0
            # 一个端口既有IPV4，又有IPV6，要算2个IP
            if ipv4 != "--" and ipv4 != "":
                ret[ctrlId] += 1
            if ipv6 != "--" and ipv6 != "":
                ret[ctrlId] += 1
        self.logger.info("the failover port=%s" % ret)
        return ret

    def __getCtrlIdByPortId(self, portId):
        # portId 示例：CTE0.R1.IOM0.P0; CTE0.L3.IOM0.P3; CTE0.A.IOM0.P0;
        relationMap = {
            "A": ["A", "R", "IOM0"],
            "B": ["B", "L", "IOM0"],
            "C": ["C", "R", "IOM1"],
            "D": ["D", "L", "IOM1"]
        }
        try:
            sectors = portId.upper().split(".")
            eng = sectors[0][-1]
            for key, value in relationMap.items():
                if value[0] in sectors[1] or (value[1] in portId and value[2] in portId):
                    return eng+key
            return ""
        except:
            return ""


