# -*- coding: UTF-8 -*-

import traceback
from frameone.util import contextUtil
from frameone.cli import cliUtil
from frameone.util import baseUtil
from frameone.base.constants import CheckStatus
from frameone.base.exception import CliCmdException, ErrorCodeSet, ErrorCode

from java.lang import Exception as JException

CARD_2X10GE_MODEL = "2x10ge optical interface module"
VER_HIGH_DEV = "V100R001C20"
VER_MID_LOW_DEV = "V200R002C20"


def execute(context):
    return InterfaceModule2x10GECheck(context).check()


class InterfaceModule2x10GECheck(object):
    def __init__(self, context):
        self.logger = contextUtil.getLogger(context)
        self.cli = contextUtil.getCLI(context)
        self.lang = contextUtil.getLang(context)
        self.dev_type = contextUtil.getDevType(context)
        self.cli_ret = []
        self.current_version = contextUtil.getCurVersion(context)
        self.target_version = contextUtil.getTargetVersion(context)

    def check(self):
        try:
            self.logger.info("[InterfaceModule2x10GECheck]begin to check.")

            cur_vrc_ver = baseUtil.getVRCVersion(self.current_version)
            target_vrc_ver = baseUtil.getVRCVersion(self.target_version)
            if baseUtil.isHvsDev(self.dev_type):
                if cur_vrc_ver >= VER_HIGH_DEV or \
                        target_vrc_ver < VER_HIGH_DEV:
                    return CheckStatus.PASS, "", ""
            else:
                if cur_vrc_ver >= VER_MID_LOW_DEV or \
                        target_vrc_ver < VER_MID_LOW_DEV:
                    return CheckStatus.PASS, "", ""

            interface_info_list = self.query_interface_info()
            interface_2x10GE_id_list = []
            # 检查每一张接口卡
            for interface_info in interface_info_list:
                interf_model = interface_info.get("Model", "")
                if interf_model.lower() == CARD_2X10GE_MODEL:
                    interface_id = interface_info.get("ID", "")
                    interface_2x10GE_id_list.append(interface_id)

            if len(interface_2x10GE_id_list) > 0:
                self.logger.info(
                    "[InterfaceModule2x10GECheck]exists 2x10GE card:%s"
                    % ",".join(interface_2x10GE_id_list))
                err_msg = baseUtil.getMsg(
                    self.lang, "InterfaceModule2x10GECheck.notpass",
                    ", ".join(interface_2x10GE_id_list))
                return CheckStatus.NOTPASS, "\n".join(self.cli_ret), err_msg

            return CheckStatus.PASS, "\n".join(self.cli_ret), ""
        except (Exception, JException):
            self.logger.error("InterfaceModule2x10GECheck exception.%s"
                              % traceback.format_exc())
            return CheckStatus.NOCHECK, "\n".join(self.cli_ret), \
                baseUtil.getMsg(self.lang, "query.result.abnormal")

    def query_interface_info(self):
        cmd = "show interface_module"
        status, cli_ret, _ = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang)
        self.cli_ret.append(cli_ret)

        if cliUtil.queryResultWithNoRecord(cli_ret):
            return []

        interface_info = cliUtil.getHorizontalCliRet(cli_ret)
        if status is not True or not interface_info:
            raise CliCmdException(ErrorCode(
                ErrorCodeSet.CLI_EXECUTE_CMD_FAILED))

        return interface_info
