# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2020. All rights reserved.
from business.common import common
from frameone.util import contextUtil
from frameone.base.constants import CheckStatus
from cbb.frame.cli import cliUtil
from cbb.frame.cli.execute_on_all_controllers import ExecuteOnAllControllers
from cbb.frame.cli.execute_on_all_controllers import ExeOnAllCtrlContext
from cbb.frame.cli.execute_on_all_controllers import ExeOnAllCtrlException
from cbb.frame.cli.execute_on_all_controllers import FuncResult
from cbb.frame.cli.execute_on_all_controllers import ResultType


def execute(context):
    """FC模式SmartIO卡类型错误

    :param context:
    :return:
    """
    return FcSmartIoCardTypeCheck(context).check()


class FcSmartIoCardTypeCheck:
    def __init__(self, context):
        self.context = context
        self.cli = contextUtil.getCLI(context)
        self.lang = contextUtil.getLang(context)
        self.logger = common.getLogger(
            contextUtil.getLogger(context), __file__)
        self.all_cli_ret = ""
        self.err_msg = common.getMsg(
            self.lang, "fc.smartio.cardtype.check.warning")
        self.common_err_msg = common.getMsg(self.lang, "query.result.abnormal")

    def check(self):
        try:
            # 检查升级路径
            dev_model = contextUtil.getDevType(self.context)
            if "2800" in dev_model:
                # 2800设备不涉及
                return CheckStatus.PASS, self.all_cli_ret, ""
            cur_version = contextUtil.getCurVersion(self.context)
            dest_version = contextUtil.getTargetVersion(self.context)
            if "DORADO" not in dev_model.upper():
                # V300R006C00SPC100及之前的版本->V300R006C50及之后的版本则继续
                if cur_version > "V300R006C00SPC100" \
                        or dest_version < "V300R006C50":
                    return CheckStatus.PASS, self.all_cli_ret, ""
            else:
                # Dorado V3 V300R001C00版本->DoradoV300R002C00及之后的版本则继续
                # Dorado V3 V300R001C01版本->DoradoV300R002C00及之后的版本则继续
                if cur_version != "V300R001C00" \
                        and cur_version != "V300R001C01" \
                        or dest_version < "V300R002C00":
                    return CheckStatus.PASS, self.all_cli_ret, ""

            # 获取SmartIO FC板载扣卡
            board_cards, cli_ret = self.get_board_cards()
            self.all_cli_ret += cli_ret

            # 获取可热拔插的SmartIO接口卡
            interface_cards, cli_ret = self.get_interface_cards()
            self.all_cli_ret += cli_ret

            if not board_cards and not interface_cards:
                return CheckStatus.PASS, self.all_cli_ret, ""

            # 检查FC卡详细信息
            result = check_board_card_mode(ExeOnAllCtrlContext(self.context))
            self.all_cli_ret += result.cli_ret
            self.logger.logInfo("fc result is {}".format(str(result)))
            if result.result_type != ResultType.SUCCESS:
                return CheckStatus.NOTPASS, self.all_cli_ret, self.err_msg

            # 检查热插拔卡详细信息
            result = check_interface_card_mode(
                ExeOnAllCtrlContext(self.context))
            self.all_cli_ret += result.cli_ret
            self.logger.logInfo("4 port result is {}".format(str(result)))
            if result.result_type != ResultType.SUCCESS:
                return CheckStatus.NOTPASS, self.all_cli_ret, self.err_msg

            return CheckStatus.PASS, self.all_cli_ret, ""
        except Exception as e:
            self.logger.logException(e)
            return CheckStatus.NOTPASS, self.all_cli_ret, self.common_err_msg
        except ExeOnAllCtrlException as e:
            self.logger.logException(e)
            return CheckStatus.NOTPASS, self.all_cli_ret, self.common_err_msg

    def get_board_cards(self):
        """获取SmartIO FC板载扣卡

        :param context:
        :return:
        """
        cmd = "show port general physical_type=FC"
        flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang)
        if flag is not True:
            raise Exception("can not get fc cards")

        cards = cliUtil.getHorizontalNostandardCliRet(cli_ret)
        board_cards = []
        for card in cards:
            card_id = card.get("ID")
            if card_id[-2:] == "H3":
                board_cards.append(card)

        self.logger.logInfo(
            "board_cards is {}".format(str(board_cards)))
        return board_cards, cli_ret

    def get_interface_cards(self):
        """获取可热拔插的SmartIO接口卡

        :param context:
        :return:
        """
        cmd = "show interface_module"
        flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang)
        if flag is not True:
            raise Exception("can not get 4 port cards")

        cards = cliUtil.getHorizontalNostandardCliRet(cli_ret)
        interface_cards = []
        for card in cards:
            card_model = card.get("Model")
            if card_model == "4 port SmartIO I/O Module":
                interface_cards.append(card)

        self.logger.logInfo(
            "interface_cards is {}".format(str(interface_cards)))
        return interface_cards, cli_ret


@ExecuteOnAllControllers
def check_board_card_mode(context):
    """检查FC卡详细信息

    :param context:
    :return:
    """
    cli = context.dev_info.cli
    lang = context.lang
    all_ret = ""
    err_msg = ""
    try:
        # 进入diagnose模式
        flag, cli_ret, err_msg = cliUtil.enterDebugModeFromCliMode(cli, lang)
        all_ret += cli_ret
        if flag is not True:
            return FuncResult(ResultType.NOT_FINISHED, all_ret, err_msg)

        # 获取板载FC扣卡的详细信息
        cmd = "drvunf showdetail 0x112003"
        cli_ret = cliUtil.execCliCmd(cli, cmd, True)
        all_ret += cli_ret

        if "Card type 0x0" in cli_ret:
            return FuncResult(ResultType.WARNING, all_ret, err_msg)

        return FuncResult(ResultType.SUCCESS, all_ret, err_msg)
    except Exception as e:
        raise Exception("can not check fc cards: {}".format(str(e)))
    finally:
        # 退出到cli模式
        cliUtil.enterCliModeFromSomeModel(cli, lang)


@ExecuteOnAllControllers
def check_interface_card_mode(context):
    """检查热插拔卡详细信息

    :param context:
    :return:
    """
    cli = context.dev_info.cli
    lang = context.lang
    all_ret = ""
    err_msg = ""
    cmd_list = [
        "drvunf showdetail 0x110003",
        "drvunf showdetail 0x110103",
        "drvunf showdetail 0x110203",
        "drvunf showdetail 0x110303",
        "drvunf showdetail 0x110403",
        "drvunf showdetail 0x110503",
        "drvunf showdetail 0x110603",
        "drvunf showdetail 0x110703",
    ]
    try:
        # 进入diagnose模式
        flag, cli_ret, err_msg = cliUtil.enterDebugModeFromCliMode(cli, lang)
        all_ret += cli_ret
        if flag is not True:
            return FuncResult(ResultType.NOT_FINISHED, all_ret, err_msg)

        # 获取热插拔卡的详细信息
        for cmd in cmd_list:
            cli_ret = cliUtil.execCliCmd(cli, cmd, True)
            all_ret += cli_ret
            if "Card type 0x2" in cli_ret:
                return FuncResult(ResultType.WARNING, all_ret, err_msg)

        return FuncResult(ResultType.SUCCESS, all_ret, err_msg)
    except Exception as e:
        raise Exception("can not check 4 port cards: {}".format(str(e)))
    finally:
        # 退出到cli模式
        cliUtil.enterCliModeFromSomeModel(cli, lang)
