# -*- coding: UTF-8 -*-

from frameone.util import contextUtil
from frameone.cli import cliUtil
from frameone.util import baseUtil
from frameone.base.constants import CheckStatus, CheckResult
from frameone.base.exception import CliCmdException, ErrorCodeSet, ErrorCode
import traceback
DORADO_RISK_SOFTWARE_VER = ["V300R001C00", "V300R001C01", 
                            "V300R001C00SPC100", "V300R001C01SPC100"]
HYBRID_RISK_SOFTWARE_VER = ["V300R003C20SPC100", "V300R003C20SPC200",
                            "V300R006C00", "V300R006C00SPC100"]
def execute(context):
    checker = HostRedundantLinkCheck(context)
    return checker.check()


class HostRedundantLinkCheck():
    
    def __init__(self, context):
        """
        checking whether a shadow host (which has no physical connection but designated with active
        initiators)is detected on current storage array, if so , a glitch might be triggered and cause 
        storage array's upgrade failure
        :param context:
        """
        self.context = context
        self.logger = contextUtil.getLogger(context)
        self.cli = contextUtil.getCLI(context)
        self.lang = contextUtil.getLang(context)
        self.errMsg = ""
        self.echos = ""
    def check(self):
        """

        :return:
        """
        riskHosts = []
        if not self._isCurStorageArrayVerAtRisk():
            self.logger.info("[HostRedundantLinkCheck] current device's version is not involved.")
            return CheckStatus.NO_SUPPORT, self.echos, self.errMsg
        
        cmd = "show host general"
        isSuc, echo, errMsg = cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        self.echos += echo
        if not isSuc:
            self.logger.info("[HostRedundantLinkCheck]failed to query host info.")
            return CheckStatus.NOCHECK, self.echos, self.errMsg
        hostInfoList = cliUtil.getHorizontalCliRet(echo)
        if not hostInfoList:
            return CheckStatus.PASS, self.echos, self.errMsg
        hostIds = []
        for hostInfo in hostInfoList:
            if hostInfo.get("ID").isdigit():
                hostIds.append(hostInfo.get("ID"))
        self.logger.info("[HostRedundantLinkCheck]current storage array's configured hosts:%s" %\
                          ",".join(hostIds))
        cmd = "show host lun host_id="
        noLunHosts = []
        failedLunInfoHosts = []
        for hostId in hostIds:
            isSuc, echo, errMsg = cliUtil.excuteCmdInCliMode(self.cli, cmd + hostId, True, self.lang)
            self.echos += echo
            if not isSuc:
                failedLunInfoHosts.append(hostId)
            if "successful" in echo:
                noLunHosts.append(hostId)
        self.logger.info("[HostRedundantLinkCheck]current storage array's no LUN hosts:%s" %\
                          ",".join(noLunHosts))
        failedInitiatorInfoHosts = []
        for noLunHost in noLunHosts:
            isSuc, isRisk = self._isHostWithActiveInitiators(noLunHost)
            if not isSuc:
                failedInitiatorInfoHosts.append(noLunHost)
            if isRisk:
                riskHosts.append(noLunHost)
        return self._capsulateResults(riskHosts, failedLunInfoHosts, failedInitiatorInfoHosts)
    def _capsulateResults(self, riskHosts, failedLunInfoHosts, failedInitiatorInfoHosts):
        """
        packaging results
        """   
        result = CheckStatus.PASS
        if failedLunInfoHosts:
            result = CheckStatus.NOCHECK
            self.errMsg += baseUtil.getMsg(self.lang, 
                                           "HostRedundantLinkCheck.host.lun.failed.check",
                                            ",".join(failedLunInfoHosts))  
        if failedInitiatorInfoHosts:
            result = CheckStatus.NOCHECK
            self.errMsg += baseUtil.getMsg(self.lang, 
                                           "HostRedundantLinkCheck.host.initr.failed.check",
                                            ",".join(failedInitiatorInfoHosts))
        if riskHosts:
            result= CheckStatus.NOTPASS
            self.logger.info("The following hosts is at risk: %s." % ",".join(riskHosts))
            self.errMsg += baseUtil.getMsg(self.lang, 
                                           "HostRedundantLinkCheck.follow.host.atRisk",
                                            ",".join(riskHosts))
        return result, self.echos, self.errMsg
    
    def _isHostWithActiveInitiators(self, hostId):
        """
        checking whether the host has no IB\ISCSI or FC initiators
        """
        cmd = "show initiator host_id=" + hostId
        isSuc, echo, errMsg = cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        self.echos += echo
        if not isSuc:
            self.logger.info("Host:%s failed to check FC&ISCSI initiators." % hostId)
            return False, False
        fcInitiatorPoi = echo.find("WWN")
        iscsiInitiatorPoi = echo.find("iSCSI IQN")
        if fcInitiatorPoi >= 0 :
            cuttedCliEcho = echo[fcInitiatorPoi - 2:iscsiInitiatorPoi - 2] \
                if iscsiInitiatorPoi > fcInitiatorPoi else echo[fcInitiatorPoi - 2:]
            isInitorOnline = self._isInitorConfiguredOnLine(cuttedCliEcho)
            if isInitorOnline:
                return True, True
            
        if iscsiInitiatorPoi >= 0:
            cuttedCliEcho = echo[iscsiInitiatorPoi - 2:fcInitiatorPoi - 2] \
                if fcInitiatorPoi > iscsiInitiatorPoi else echo[iscsiInitiatorPoi - 2:]
            isInitorOnline = self._isInitorConfiguredOnLine(cuttedCliEcho)
            if isInitorOnline:
                return True, True
            
        cmd = "show ib_initiator general host_id=" + hostId
        isSuc, echo, errMsg = cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        self.echos += echo
        if not isSuc:
            self.logger.info("Host:%s failed to check IB initiators." % hostId)
            return False, False
        if "successful" in echo.lower():
            return True, False
        isInitorOnline = self._isInitorConfiguredOnLine(echo)
        return True, isInitorOnline
    
    def _isInitorConfiguredOnLine(self, cuttedCliEcho):
        """
        checking wthether the initor is onlie
        """
        initiatorList = cliUtil.getHorizontalCliRet(cuttedCliEcho) 
        for initrInfo in initiatorList:
            if initrInfo.get("Running Status", "").lower() == "online":
                return True
        return False
    
    def _isCurStorageArrayVerAtRisk(self):
        """
        Only the storage arry of the following software version is required to be checked.
        V3R3C20SPC100,V3R3C20SPC200,V3R6C00,V3R6C00SPC100,
        Dorado V3R1C00,Dorado V3R1C01,Dorado V3R1C00SPC100,Dorado V3R1C01SPC100
        """
        currentVersion = contextUtil.getCurVersion(self.context)
        if "SPH" in currentVersion:
            currentVersion = currentVersion[:currentVersion.find("SPH")]
        curProdcutModel = contextUtil.getDevType(self.context)
        self.logger.info("current device's software version:%s, %s" % (currentVersion, curProdcutModel))
        if "dorado" in curProdcutModel.lower():
            return currentVersion in DORADO_RISK_SOFTWARE_VER
        else:
            return currentVersion in HYBRID_RISK_SOFTWARE_VER
        
        