# -*- coding: UTF-8 -*-
"""
 双活域升级前版本检查
"""

from frameone.util import contextUtil, baseUtil
from frameone.cli import cliUtil
from frameone.base.constants import CheckStatus, CheckResult
from business.common import common
from business.common import logger

from java.lang import Exception as JException

# 需要检查的设备型号
NEED_CHECK_DEV_TYPE_LIST = ["18500 V5", "18800 V5", "18500F V5", "18800F V5"]
# 支持双活域的源版本
NEED_CHECK_CURRENT_VER = "V500R007C30SPC100"
NEED_CHECK_PATCH_VER = "V500R007C30SPH107"
# 支持双活域的目标版本
NEED_CHECK_TARGET_VER = "V500R007C60SPC200"

# 继续检查的结果状态
CONTINUE_CHECK_STATUS = 5


def execute(data_dict):
    """
    检查项的入口函数
    :param data_dict: 上下文信息
    :return: 检查结果（结果状态、cli、错误消息）
    """
    check_result = hyper_metro_domain_config(data_dict).check()
    return check_result.toScriptResult()


class hyper_metro_domain_config(object):
    """
    双活域升级前版本检查类
    """
    def __init__(self, context):
        self.logger = logger.Logger(contextUtil.getLogger(context), __file__)
        self.lang = contextUtil.getLang(context)
        self.cli = contextUtil.getCLI(context)
        self.context = context
        self.origin_infos = []
        self.err_msg = ""
        self.common_msg = baseUtil.getMsg(self.lang, "query.result.abnormal")

    def check(self):
        """
        双活域配置检查类的入口函数
        :return: 检查结果
        """
        try:
            continue_flag, check_dev_flag = self._check_dev_info()
            if not continue_flag:
                return CheckResult(check_dev_flag, self.origin_infos,
                                   self.err_msg)

            hyper_metro_config_check_flag = self.hyper_metro_config_check()
            return CheckResult(hyper_metro_config_check_flag,
                               self.origin_infos, self.err_msg)
        except (Exception, JException) as ex:
            self.logger.logError("[hyper_metro_domain_ver]exe exception:%s"
                                 % str(ex))
            return CheckResult(CheckStatus.NOCHECK, self.origin_infos,
                               self.common_msg)

    def _check_dev_info(self):
        """
        检查设备型号、版本号等信息
        :return: 检查结果(True:继续检查，False：停止检查)，返回状态
        """
        dev_type = contextUtil.getDevType(self.context)
        self.origin_infos.append("device type:%s" % dev_type)
        self.logger.logInfo("[hyper_metro_domain_ver]dev type is %s"
                            % dev_type)
        if dev_type.upper() not in NEED_CHECK_DEV_TYPE_LIST:
            return False, CheckStatus.PASS

        current_version = contextUtil.getCurVersion(self.context)
        target_version = contextUtil.getTargetVersion(self.context)
        self.logger.logInfo("[hyper_metro_domain_ver]current version is %s, "
                            "target version is %s"
                            % (current_version, target_version))
        ret_flag, hot_patch_version, cli_ret, err_msg = \
            common.getHotPatchVersion(self.cli, self.lang)
        self.origin_infos.append(cli_ret)
        self.logger.logInfo("[hyper_metro_domain_ver]current patch version "
                            "is %s" % hot_patch_version)
        if not ret_flag:
            self.err_msg = err_msg
            return False, CheckStatus.NOCHECK
        if not (current_version.upper() == NEED_CHECK_CURRENT_VER and
                hot_patch_version.upper() >= NEED_CHECK_PATCH_VER and
                target_version.upper() < NEED_CHECK_TARGET_VER):
            return False, CheckStatus.PASS
        return True, None

    def hyper_metro_config_check(self):
        """
        执行cli命令检查双活域配置
        :return: 检查结果
        """
        cmd = r"show hyper_metro_domain general |filterColumn include " \
              r"columnList=ID,Remote\sDevice\sID"
        ret_flag, cli_ret, ret_msg = \
            cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        self.origin_infos.append(cli_ret)

        if not ret_flag:
            return CheckStatus.NOCHECK
        if cliUtil.queryResultWithNoRecord(cli_ret):
            return CheckStatus.PASS

        hyper_metro_info_list = cliUtil.getHorizontalCliRet(cli_ret)
        if len(hyper_metro_info_list) == 0:
            self.logger.logError("[hyper_metro_domain_ver]parse cli "
                                 "ret failed.")
            self.err_msg = self.common_msg
            return CheckStatus.NOCHECK

        hyper_metro_domain_dict = dict()
        for hyper_metro_info in hyper_metro_info_list:
            remote_device_id = hyper_metro_info.get("Remote Device ID", "")
            hyper_metro_domain_id = hyper_metro_info.get("ID", "")
            domain_list = hyper_metro_domain_dict.get(remote_device_id, [])
            domain_list.append(hyper_metro_domain_id)
            hyper_metro_domain_dict[remote_device_id] = domain_list
        for domain_list_value in hyper_metro_domain_dict.values():
            if len(domain_list_value) >= 2:
                self.err_msg = baseUtil.getMsg(
                    self.lang, "hyper_metro.domain.config.notpass")
                return CheckStatus.NOTPASS

        return CheckStatus.PASS
