# -*- coding: UTF-8 -*-

from frameone.util import contextUtil
from frameone.cli import cliUtil
from frameone.util import baseUtil
from frameone.base.constants import CheckStatus, CheckResult
from frameone.base.exception import CliCmdException, ErrorCodeSet, ErrorCode
from business.common import common
import traceback
import time
V3_RISK_SERIES_MIN_MAX = ["V300R003C00SPC100", "V300R006C10SPC100"]
V5_RISK_SERIES_MIN_MAX = ["V500R007C00", "V500R007C00SPC100"]
WHITE_LIST = {"V300R003C10SPC100" : "SPH118",
               "V300R003C20SPC200":"SPH213",
               "V300R005C00SPC300":"SPH308", 
               "V300R006C00SPC100":"SPH111",
               "V500R007C00SPC100":"SPH105",
               "V300R006C10SPC100":"SPH105"}
ITEM_ID = "cli_service_fullcopy_on_hyperMetro_LUN"
def execute(context):
    try:
        context["checkState"] = common.PROCESS_STATE_CHECKING
        common.threadUp(context, ITEM_ID)
        checker = HyperMetroFullCopyCheck(context)
    finally:
        try:
            common.finishProcess(context, ITEM_ID)
        finally:
            common.setProgress(context, common.PROGRESS_NUM_MAX, ITEM_ID)
    return checker.check()


class HyperMetroFullCopyCheck():
    
    def __init__(self, context):
        """
        when online upgrade requires temporary shut down hyper-metro feature, and VM migration operation is ongoing,
        a failure of VM migration might be triggered during the time when hyper-metro feature un-functioning.
        :param context:
        """
        self.context = context
        self.logger = contextUtil.getLogger(context)
        self.cli = contextUtil.getCLI(context)
        self.lang = contextUtil.getLang(context)
        self.errMsg = ""
        self.echos = ""
        self.maxCtlrNum = 0
        
    def check(self):
        """
        start checking
        :return:
        """
        try:
            riskHosts = []
            if not self._isCurStorageArrayVerAtRisk():
                self.logger.info("[HyperMetroFullCopyCheck] current device's version is not involved.")
                return CheckStatus.NO_SUPPORT, self.echos, self.errMsg
            
            cmd = "show hyper_metro_pair general"
            isSuc, echo, errMsg = cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
            self.echos += echo
            if not isSuc:
                if "^" in echo:
                    return CheckStatus.NO_SUPPORT, self.echos, self.errMsg 
                self.logger.info("[HyperMetroFullCopyCheck]failed to query hyper_metro info.")
                self.errMsg = baseUtil.getMsg(self.lang, "HyperMetro.feature.query.fail")
                return CheckStatus.NOCHECK, self.echos, self.errMsg
            if "successful" in echo:
                self.logger.info("[HyperMetroFullCopyCheck]No hyperMetro pair info was detected on device.")

                return CheckStatus.PASS, self.echos, self.errMsg

            return self._parseAllCtlrFullCopyStatus()

        except UncheckException, exp:
            self.logger.error("[HyperMetroFullCopyCheck] unchecked exception occurred.")
            self.errMsg = baseUtil.getMsg(self.lang, exp.errMsg)
            return CheckStatus.NOCHECK, self.echos, self.errMsg 
        except BaseException, exp:
            self.logger.error("[HyperMetroFullCopyCheck]exception occurred while processing script" + str(exp))
            self.errMsg = baseUtil.getMsg(self.lang, "query.result.abnormal")

            return CheckStatus.NOCHECK, self.echos, self.errMsg
        finally:
            # sometimes if CLI timeout is reached and array kills the connection,
            # a reconnection is required.
            if not self.cli.isConnected():
                self.logger.info("current cli is interrupted, now start to reconnect...")
                self.cli = contextUtil.getCLI(self.context)
            
    def _parseAllCtlrFullCopyStatus(self):
        """
        """
        failedCtlrs = []
        notPassCtlrs = []
        
        isSuc, cliRet, errMsg, ctlrsInfos = cliUtil.getControllerEngineTopography(self.cli, self.lang, self.logger)
        self.echos += cliRet
        if not isSuc:
            raise UncheckException("failed.quering.controller.topography")
        curEngineId = ctlrsInfos[0]
        curCtlrId = ctlrsInfos[1]
        engineNodeDict = ctlrsInfos[2]
        self.logger.info("[HyperMetroFullCopyCheck]engine topography : " + str(engineNodeDict))
        self.maxCtlrNum = len(engineNodeDict['0'])
        self.logger.info("[HyperMetroFullCopyCheck] now checking full-copy status on current connected engine...")
        self.logger.info("[HyperMetroFullCopyCheck]current controller id : " + curCtlrId)
        status, isSupportCrossEngine = cliUtil.isSupportSsh2RemoteCrossEngine(self.cli, self.lang, self.logger)
        if not status:
            raise UncheckException("query.result.abnormal")
        if isSupportCrossEngine:
            self.logger.info("[HyperMetroFullCopyCheck]current host supports cross-engine ssh jump.")
            failedCtlrs, notPassCtlrs, errMsg = self._getallCtlrFullCpyStatusCrossEngine(curCtlrId, engineNodeDict)
        else:
            failedCtlrs, notPassCtlrs, errMsg = self._getEngineFullCopyStatus(curEngineId, curCtlrId, engineNodeDict, self.cli)
            self.errMsg += errMsg
        unreachableEngines = []
        if not notPassCtlrs and not isSupportCrossEngine:
            unreachableEngines, failedCtlrsEng, notPassCtlrsOtrEng =self._checkOtherEngineCtlrStatus(engineNodeDict, curEngineId)
            failedCtlrs.extend(failedCtlrsEng)
            notPassCtlrs.extend(notPassCtlrsOtrEng)

        return self.__parseResults(unreachableEngines, failedCtlrs, notPassCtlrs)

    def __parseResults(self, unreachableEngines, failedCtlrs, notPassctlrs):
        """
        :param unreachableEngines the engine numbers whom we cannot connect to.
        :param failedCtlrs: 
        :param notPassctlrs:
        """
        result = CheckStatus.PASS
        if unreachableEngines:
            result = CheckStatus.NOCHECK
            self.errMsg += baseUtil.getMsg(self.lang, 
                                           "the.following.engine.failed.to.connect",
                                            ",".join(unreachableEngines))
        if failedCtlrs:
            result = CheckStatus.NOCHECK
            readableCtlrIds = map(lambda x:str(int(x)/self.maxCtlrNum) + ["A", "B", "C", "D"][int(x) % self.maxCtlrNum],
                                   failedCtlrs)
            
            self.errMsg += baseUtil.getMsg(self.lang, 
                                           "the.following.ctlr.failed.opearte",
                                            ",".join(readableCtlrIds))
        if notPassctlrs:
            result = CheckStatus.NOTPASS
            readableCtlrIds = map(lambda x:str(int(x)/self.maxCtlrNum) + ["A", "B", "C", "D"][int(x) % self.maxCtlrNum],
                                   notPassctlrs)
            
            self.errMsg += baseUtil.getMsg(self.lang, 
                                           "the.following.ctlr.fullcopy.not.pass",
                                            ",".join(notPassctlrs))
        return result, self.echos, self.errMsg
    def _checkOtherEngineCtlrStatus(self, engineMap, excludeEng):
        """

        :param engineMap:
        :param excludeEng:
        :return:
        """
        failedEngines = []
        unreachableEngines = []
        notPassctlrs = []
        failedCtlrs = []
        isSuc, cliRet, errMsg, managementPortInfo = cliUtil.getManagementPortInfo(self.cli, self.lang)
        if not isSuc:
            self.logger.info("failed fetching management ")
            raise UncheckException("query.result.abnormal") #
        engReachableIPDict = self._getAllEngineReachableIP(managementPortInfo)
        
        self.logger.info('All engine reachable IP:' + unicode(engReachableIPDict))

        for engineId in engineMap:
            if engineId == excludeEng:
                continue
            engMgtReachableIP = engReachableIPDict.get(engineId)
            self.logger.info("Begin connect to controller %s on engine %s." % (engMgtReachableIP, engineId))
            cilConnection = contextUtil.createCliConnection(self.context, engMgtReachableIP)
            if not cilConnection:
                self.logger.info("connection failed.") 
                unreachableEngines.append(engineId)
                continue
            
            isSuc, cliRet, errMsg, ctlrsInfos = cliUtil.getControllerEngineTopography(cilConnection, self.lang, self.logger)
            if not isSuc:
                failedEngines.append(engineId)
                continue
            curCtlrId = ctlrsInfos[1]
            self.logger.info("[HyperMetroFullCopyCheck]current controller id : " + curCtlrId)
            failedCtlrsEng, notPassCtlrsEng, errMsg = self._getEngineFullCopyStatus(engineId, curCtlrId, engineMap, cilConnection)
            failedCtlrs.extend(failedCtlrsEng)
            notPassctlrs.extend(notPassCtlrsEng)
            self.errMsg += errMsg
            cliUtil.closeCliConnection(cilConnection)
        return unreachableEngines, failedCtlrs, notPassctlrs
    def _getallCtlrFullCpyStatusCrossEngine(self, curCtlrId, engineNodeDict):
        """
        
        :param:curCtlrId
        :param:engineNodeDict
        """
        failedCtlrs = []
        notPassCtlrs = []
        errMsg = ""
        self.echos += "\nOn Controller %s:\n" % curCtlrId
        self.logger.info("[HyperMetroFullCopyCheck]checking controller %s 's full-copy status" % curCtlrId)
        hasError, hasBusiness = self._checkCtlrFullCopyRunningStatus(self.cli, curCtlrId)
        if hasError:
            failedCtlrs.append(curCtlrId)
        if hasBusiness:
            notPassCtlrs.append(curCtlrId)
        ctlrIds = []
        for enginId in engineNodeDict:
            ctlrIds.extend(engineNodeDict.get(enginId))
        self.logger.info("[HyperMetroFullCopyCheck]all ctlr ids listed as :" + ",".join(ctlrIds))
        for ctlrId in ctlrIds:
            if str(ctlrId) == str(curCtlrId):
                continue
            self.echos += "\nOn Controller %s:\n" % ctlrId
            cmd = "sshtoremoteExt %s" % ctlrId
            self.logger.info("[HyperMetroFullCopyCheck]connecting to % controller using cross engine module"
                              % ctlrId)
            passwd = self.context.get("dev").getLoginUser().getPassword()
            isSuc, echos, errMsg = cliUtil.sshToRemoteContr(self.cli, cmd, passwd, self.lang)
            self.echos += echos
            del passwd
            if not isSuc:
                failedCtlrs.append(ctlrId)
                continue
            
            hasError, hasBusiness = self._checkCtlrFullCopyRunningStatus(self.cli, ctlrId)
            if hasError:
                failedCtlrs.append(ctlrId)
            if hasBusiness:
                notPassCtlrs.append(ctlrId)
            cliUtil.exitHeartbeatCli(self.cli, self.lang)
        
        return failedCtlrs, notPassCtlrs, errMsg
            
    def _getEngineFullCopyStatus(self, curEngineId, curCtlrId, engineNodeDict, sshCon):
        """
        giving engine, Check fullcopy status on all ctlrs belonging to this engine
        :param:
        """
        failedCtlrs = []
        notPassCtlrs = []
        errMsg = ""
        self.echos += "\nOn Controller %s:\n" % curCtlrId
        self.logger.info("[HyperMetroFullCopyCheck]checking controller %s 's full-copy status" % curCtlrId)
        hasError, hasBusiness = self._checkCtlrFullCopyRunningStatus(sshCon, curCtlrId)
        if hasError:
            failedCtlrs.append(curCtlrId)
        if hasBusiness:
            notPassCtlrs.append(curCtlrId)
        for todoCtlrId in engineNodeDict[curEngineId]:
            if str(todoCtlrId) == str(curCtlrId):
                continue
            ctlrNodeId = str(int(todoCtlrId) % self.maxCtlrNum)
            self.echos += "\nOn Controller %s:\n" % ctlrNodeId
            if self.maxCtlrNum == 2:
                cmd = "sshtoremote"
            else:
                cmd = "sshtoremoteExt %s" % ctlrNodeId
            self.logger.info("[HyperMetroFullCopyCheck]connecting to % controller" % todoCtlrId)
            passwd = self.context.get("dev").getLoginUser().getPassword()
            isSuc, echos, errMsg = cliUtil.sshToRemoteContr(sshCon, cmd, passwd, self.lang)
            self.echos += echos
            del passwd
            if not isSuc:
                failedCtlrs.append(todoCtlrId)
                continue
            
            hasError, hasBusiness = self._checkCtlrFullCopyRunningStatus(sshCon, todoCtlrId)
            if hasError:
                failedCtlrs.append(ctlrNodeId)
            if hasBusiness:
                notPassCtlrs.append(ctlrNodeId)
            cliUtil.exitHeartbeatCli(self.cli, self.lang)
        
        return failedCtlrs, notPassCtlrs, errMsg
    
    def _getAllEngineReachableIP(self, mgtPortDictList):
        """
        @summary: Get reachable IP address of every engine.
        @param mgtPortDictList: Management port IP address information dictionary.
        @return: {engineId: reachable_IP_address}
        """
        engineIpDict = {}
        for managementPortInfoDict in mgtPortDictList:
            ip = managementPortInfoDict.get("IPv4 Address")
            portRunningStatus = managementPortInfoDict.get("Running Status")
            if portRunningStatus == "Link Up" and ip not in ["", "--"]:
                managementPortId = managementPortInfoDict.get("ID")
                engineId = managementPortId.split(".")[0][-1:]
                if (engineId not in engineIpDict) and ("DSW" not in managementPortId):
                    engineIpDict.setdefault(engineId, ip)
        
        return engineIpDict


    def _checkCtlrFullCopyRunningStatus(self, sshCon, ctlrNo):
        """

        :param sshCon: the ssh connection instance of current connected controller
        :param ctlrNo: the controller number
        :return:
        """
        self.logger.info("now checking full copy strategy on controller:" + str(ctlrNo))
        cmd = "fcopy showjobcount"
        hasError = False
        hasBusiness = False
        for _ in range(10):

            isOk, echos, errMsg = cliUtil.executeCmdInDebugMode(sshCon, cmd, True, self.lang)
            self.echos += echos
            if not isOk:
                hasError = True
                continue
            for line in echos.lower().splitlines():
                if "count:" in line:
                    count = line.split(":")[-1].strip()
                    self.logger.info("the %d time cmd result count is %s" %(_, count))
                    if count.isdigit() and int(count) > 0:
                        hasBusiness = True
                        break
            time.sleep(1)
        self.logger.info("ctlr %s 's checking result : hasError: %s, hasBusiness:%s" %
                         (ctlrNo, str(hasError), str(hasBusiness)))
        return hasError, hasBusiness


    def _isCurStorageArrayVerAtRisk(self):
        """
        Only the storage arry of the following software version is required to be checked.
        V3_RISK_SERIES_MIN_MAX = ["V300R003C00SPC100", "V300R006C10SPC100"]
        V5_RISK_SERIES_MIN_MAX = ["V500R007C00", "V500R007C00SPC100"]
        
        the following software hot patch has solved the problem, which shall be excluded:
         V300R003C10SPC100+SPH118 V300R003C20SPC200+SPH213 V300R005C00SPC300+SPH308 
         V300R006C00SPC100+SPH111 V500R007C00SPC100/V300R006C00SPC100+SPH105；
        """
        isSuc, hotPatchVersion, cliRet, errMsg =common.getHotPatchVersion(self.cli, self.lang)
        if not isSuc:
            raise Exception("query.result.abnormal")
        currentVersion = contextUtil.getCurVersion(self.context)
        self.echos += cliRet
        
        if currentVersion >= V3_RISK_SERIES_MIN_MAX[0] and\
                currentVersion <= V3_RISK_SERIES_MIN_MAX[1] or\
                    currentVersion >= V5_RISK_SERIES_MIN_MAX[0] and\
                        currentVersion <= V5_RISK_SERIES_MIN_MAX[1]:
            self.logger.info("current device's software version:%s" % (currentVersion))
            
            return WHITE_LIST.get(currentVersion) is None or WHITE_LIST.get(currentVersion, "") > hotPatchVersion
        
        return False

class UncheckException(Exception):
    def __init__(self, errMsg):
        Exception.__init__(self)
        self.errMsg = errMsg