# -*- coding: UTF-8 -*-
import traceback
from frameone.util import contextUtil
from frameone.cli import cliUtil
from frameone.util import baseUtil
from frameone.base.constants import CheckStatus
from function import StorageContextUtil

INVOLVE_DEV_TYPE_DICT = {
    "V3": ["18500 V3", "18500F V3", "18800 V3", "18800F V3"],
    "V5": ["6800 V5", "6800F V5", "18500 V5", "18500F V5", "18800 V5", "18800F V5"],
    "DoradoV3": ["Dorado6000 V3", "Dorado18000 V3"]
}
RISK_SOFTWARE_VERSION_DICT = {
    "V3": {"Min": "V300R003C00", "Max": "V300R006C50SPC200"},
    "V5": {"Min": "V500R007C00", "Max": "V500R007C00SPC100"},
    "DoradoV3": {"Min": "V300R002C00", "Max": "V300R002C00"}
}
CACHE_ENOUGH_CAPACITY = 1024 * 1024
# 可解决问题的补丁
SOLVE_RISK_PATCH_VERSION_DICT = {
    "V300R003C00SPC100": {"patch": "V300R003C00SPH190", "equals": True},
    "V300R003C10SPC100": {"patch": "V300R003C10SPH190", "equals": True},
    "V300R003C20SPC200": {"patch": "V300R003C20SPH217", "equals": False},
    "V300R006C20": {"patch": "V300R006C20SPH013", "equals": False},
    "V300R006C50SPC100": {"patch": "V300R006C50SPH103", "equals": False}
}
# 需要排除的补丁范围
NORISK_URGENCY_PATCH_VERSION_DICT = {
    "V300R003C20SPC200": {"Min": "V300R003C20SPH270", "Max": "V300R003C20SPH279"},
    "V300R006C20": {"Min": "V300R006C20SPH070", "Max": "V300R006C20SPH079"},
    "V300R006C50SPC100": {"Min": "V300R006C50SPH170", "Max": "V300R006C50SPH179"}
}

def execute(context):
    """
    控制器间链路检查
    """
    return ControllerLinkCheck(context).run(context)


class ControllerLinkCheck:

    def __init__(self,context):
        self.lang = contextUtil.getLang(context)
        self.logger = contextUtil.getLogger(context)
        self.cli = contextUtil.getSSH(context)
        self.isPreUpgradeCheckScene = StorageContextUtil.isPreUpgradeCheckScene(context)
        self.dev = contextUtil.getDevObj(context)
        self.cliRet = ""
        self.errMsg = ""

    def run(self,context):
        """
        执行控制器间链路检查
        :param context: 设备信息
        :return: 评估结果
        """
        try:
            softwareCurrentVersion = contextUtil.getCurVersion(context)
            softwareTargetVersion = contextUtil.getTargetVersion(context)

            #风险系列、风险版本不涉及补丁的初判断
            softwareDevType = self.__isInvolveDevType(contextUtil.getDevType(context))
            if softwareDevType not in RISK_SOFTWARE_VERSION_DICT:
                return CheckStatus.PASS, self.cliRet, self.errMsg

            if not self.__isRiskVersion(softwareDevType, softwareCurrentVersion) and not self.__isRiskVersion(softwareDevType, softwareTargetVersion):
                return CheckStatus.PASS, self.cliRet, self.errMsg

            # 查看容量
            checkStatus, cacheCapacity = self.__getDevCacheCapacity()
            if checkStatus is CheckStatus.NOCHECK:
                return checkStatus, self.cliRet, self.errMsg

            #容量判断
            if not self.__isCacheCapacityEnough(cacheCapacity):
                return CheckStatus.PASS, self.cliRet, self.errMsg

            # 获取补丁版本
            isExcuteNormal, softwareCurrentVersion, patchCurrentVersion = cliUtil.getSystemVersion(self.cli, self.lang)
            if not isExcuteNormal:
                self.errMsg = baseUtil.getMsg(self.lang, "query.result.abnormal")
                self.logger.error("Query result abnormal about service cifs.")
                return CheckStatus.NOCHECK, self.cliRet, self.errMsg
            self.cliRet += "\n" + patchCurrentVersion

            #涉及补丁的风险版本终判断
            return self.__getRiskVerJudgeRes(softwareDevType, softwareCurrentVersion, softwareTargetVersion, patchCurrentVersion), self.cliRet, self.errMsg

        except Exception as exception:
            self.logger.error("There are some exception occur :%s" % traceback.format_exc())
            return CheckStatus.NOCHECK, self.cliRet, baseUtil.getMsg(self.lang, 'query.result.abnormal')

    def __isInvolveDevType(self, devType):
        """
        转换设备类型为简称，便于查找风险版本
        :param devType: 设备类型
        :return: 设备类型简称
        """
        devType = devType.strip()
        for shortDevType in INVOLVE_DEV_TYPE_DICT.keys():
            devTypeList = INVOLVE_DEV_TYPE_DICT.get(shortDevType)
            if devType in devTypeList:
                return shortDevType
        return None

    def __getDevCacheCapacity(self):
        """
        执行cmd命令并对成功与否进行判断
        :return:（评估结果，Cli回显解析结果）
        """
        cmd = "show controller general"
        self.logger.info(cmd)
        isCmdExecNormal, cliRet, self.errMsg = cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        self.cliRet += '\n' + cliRet

        cacheCapacity = ""
        # 判断回显是否为Command executed successfully
        if cliUtil.queryResultWithNoRecord(cliRet):
            self.logger.error("CliRet is 'Command executed successfully'")
            return CheckStatus.NOCHECK, cacheCapacity

        # 命令是否执行正常
        if not isCmdExecNormal:
            self.logger.error("Abnormal execution of commands")
            return CheckStatus.NOCHECK, cacheCapacity

        cliRetLinesList = cliUtil.getVerticalCliRet(cliRet)

        if len(cliRetLinesList) == 0:
            self.errMsg = baseUtil.getMsg(self.lang, "query.result.abnormal")
            self.logger.error("Query result abnormal about service cifs.")
            return CheckStatus.NOCHECK, cacheCapacity

        # 容量获取
        cacheCapacity = cliRetLinesList[0].get("Cache Capacity", "")
        if len(cacheCapacity) == 0:
            self.logger.error("Not found Cache Capacity in CliRet")
            return CheckStatus.NOCHECK, cacheCapacity

        return CheckStatus.PASS, cacheCapacity

    def __getRiskVerJudgeRes(self, softwareDevType, softwareCurrentVersion, softwareTargetVersion, patchCurrentVersion):
        """
        获得带补丁风险版本判断结果
        :param softwareDevType: 设备型号
        :param softwareCurrentVersion: 当前版本
        :param softwareTargetVersion: 目标版本
        :param patchCurrentVersion: 当前补丁版本
        :return: 判断结果CheckStatus.PASS/CheckStatus.NOTPASS
        """
        CVCheckStatus = self.__isRiskCurrentVersionWithPatch( softwareDevType, softwareCurrentVersion, patchCurrentVersion);
        TVCheckStatus, isChoosePatch = self.__isRiskTargetVersionWithPatch(softwareDevType, softwareTargetVersion);

        #如果工具可以确认“升级前版本有风险，升级后版本有风险但选择了同时升级系统和激活热补丁功能，
        #且补丁版本正确”，要提示升级前有风险，升级目标版本有风险（防止解决升级前风险后，不采用同时升级系统和激活热补丁功能）。
        if not TVCheckStatus and CVCheckStatus and isChoosePatch:
            self.errMsg += baseUtil.getMsg(self.lang, "LinCheck.RiskCurrentVer.RiskTargetVerWithPath.notpass",
                                           (softwareCurrentVersion, softwareTargetVersion))
        if not TVCheckStatus and not CVCheckStatus:
            return CheckStatus.PASS
        return CheckStatus.NOTPASS


    def __isRiskCurrentVersionWithPatch(self, softwareDevType, softwareCurrentVersion, patchCurrentVersion):
        """
        # 当前版本是否通过涉及补丁
        :param softwareDevType: 设备类型
        :param softwareCurrentVersion: 当前版本
        :param patchCurrentVersion: 当前版本补丁
        :return: 是否是风险版本
        """
        if self.__isRiskVersion(softwareDevType, softwareCurrentVersion):
            if softwareCurrentVersion in SOLVE_RISK_PATCH_VERSION_DICT:
                if self.__isSolveRiskPatchVersion(softwareCurrentVersion, patchCurrentVersion):
                    return False
                error_key = \
                    "LinCheck.CurrentRisk.HasPatch.notpass"
                if self.only_risk_patch(softwareCurrentVersion):
                    error_key = \
                        "LinCheck.CurrentRisk.HasEqualsPatch.notpass"
                self.errMsg += \
                    baseUtil.getMsg(
                        self.lang, error_key,
                        (softwareCurrentVersion,
                         self.get_risk_patch(softwareCurrentVersion)))
                return True
            self.errMsg += baseUtil.getMsg(self.lang, "LinCheck.CurrentRisk.notpass", softwareCurrentVersion)
            return True
        return False
    def __isRiskTargetVersionWithPatch(self, softwareDevType, softwareTargetVersion):
        """
        # 目标版本是否通过涉及热补丁
        :param softwareDevType: 设备型号
        :param softwareTargetVersion: 目标版本
        :return: （目标版本是否是风险版本，是否选择了热补丁（设备升级）使得版本无风险）
        """
        if self.__isRiskVersion(softwareDevType, softwareTargetVersion):
            upgPatchPkgVer = self.dev.getUpgPatchPkgVer()
            if not self.isPreUpgradeCheckScene and softwareTargetVersion in SOLVE_RISK_PATCH_VERSION_DICT \
                    and self.__isSolveRiskPatchVersion(softwareTargetVersion, upgPatchPkgVer):
                return False, True
            self.errMsg += baseUtil.getMsg(self.lang, "LinCheck.TargetRisk.notpass", softwareTargetVersion)
            return True, False
        return False, False

    def __isCacheCapacityEnough(self, cacheCapacity):
        """
        判断设备Cache容量是否大于等于1TB
        :param cacheCapacity: 命令回显中Cache容量字符串 String
        :return:是否满足容量要求 True/False Boolean
        """
        cacheCapacity = cacheCapacity.strip()
        cacheUnit = cacheCapacity[-2:]
        cacheCapacityNum = float(cacheCapacity[:-2])

        return cacheCapacityNum * self.__transCapacityUnit(cacheUnit) >= CACHE_ENOUGH_CAPACITY

    def __transCapacityUnit(self, cacheUnit):
        """
        容量单位的转换
        :param cacheUnit: 容量单位 String
        :return: 以M为单位返回值 int
        """
        if cacheUnit == "MB":
            return 1
        if cacheUnit == "GB":
            return 1024
        if cacheUnit == "TB":
            return 1024 * 1024
        if cacheUnit == "PB":
            return 1024 * 1024 *1024

    def __isRiskVersion(self, devType, version):
        """
        是否是风险版本判断
        :param devType: 设备系列
        :param version: 系统版本
        :return: True/False
        """
        return RISK_SOFTWARE_VERSION_DICT.get(devType).get("Min") <= version <= RISK_SOFTWARE_VERSION_DICT.get(devType).get("Max")

    def __isSolveRiskPatchVersion(self, version, patchCurrentVersion):
        """
        是否为风险版本的正确补丁
        :param version: 系统版本
        :param patchCurrentVersion: 补丁版本
        :return: True/False
        """
        solve_risk_patch = self.get_risk_patch(version)
        equals_patch = self.only_risk_patch(version)
        # 唯一指定正确补丁版本检查
        if equals_patch is True:
            return patchCurrentVersion == solve_risk_patch
        # 判断补丁版本是否为可解决问题补丁范围内
        if not bool(solve_risk_patch) or \
                solve_risk_patch > patchCurrentVersion:
            return False
        # 判断补丁是否在被排除补丁范围中
        exclude_min_ver = NORISK_URGENCY_PATCH_VERSION_DICT.get(version,
                                                                {}).get(
            "Min")
        exclude_max_ver = NORISK_URGENCY_PATCH_VERSION_DICT.get(version,
                                                                {}).get(
            "Max")
        exclude_ver_flag = False
        if exclude_min_ver is not None:
            exclude_ver_flag = patchCurrentVersion >= exclude_min_ver
        if exclude_max_ver is not None:
            exclude_ver_flag = exclude_ver_flag and \
                               patchCurrentVersion <= exclude_max_ver
        return not exclude_ver_flag

    def get_risk_patch(self, version):
        """
        获取版本可以解决问题的热补丁
        :param version:当前阵列版本
        :return:热补丁版本
        """
        if version in SOLVE_RISK_PATCH_VERSION_DICT.keys() \
                and SOLVE_RISK_PATCH_VERSION_DICT.get(
                version).get("patch") is not None:
            return SOLVE_RISK_PATCH_VERSION_DICT.get(version).get("patch")
        return ""

    def only_risk_patch(self, version):
        """
        热补丁是否唯一匹配
        :param version:当前阵列版本
        :return: 是，否
        """
        if version in SOLVE_RISK_PATCH_VERSION_DICT.keys() \
                and SOLVE_RISK_PATCH_VERSION_DICT.get(
                version).get("equals") is not None:
            return SOLVE_RISK_PATCH_VERSION_DICT.get(version).get("equals")
        return False
