#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2019. All rights reserved.

"""
@time: 2019/08/29 
@file: check_cli_software_controllersCheckOfMutilEngine.py 
@function: 控制器数量检查（大于8控）
"""
from java.lang import Throwable
from frameone.util import contextUtil
from frameone.util import baseUtil
from business.common import logger
from frameone.cli import cliUtil
from frameone.base.constants import CheckStatus


class ShortDevType:
    V3 = 1  # V3系列型号
    V5 = 2  # V5系列型号


INVOLVE_DEV_TYPE_DICT = {
    ShortDevType.V3: ("18500 V3", "18500F V3", "18800 V3", "18800F V3"),
    ShortDevType.V5: ("18500 V5", "18500F V5", "18800 V5", "18800F V5")
}
INVOLVE_CURRENT_VERSION_RANGE_DICT = {
    ShortDevType.V3: {"Min": "V300R003C00", "Max": "V300R006C30"},
    ShortDevType.V5: {"Min": "V500R007C00", "Max": "V500R007C20"}
}
INVOLVE_TARGET_VERSION_RANGE_DICT = {
    ShortDevType.V3: {"Min": "V300R006C30", "Max": "V300R006C50SPC100"},
    ShortDevType.V5: {"Min": "V500R007C20", "Max": "V500R007C30SPC100"}
}


def execute(context):
    return ControllersCheckOfMutilEngine(context).run()


class ControllersCheckOfMutilEngine:

    def __init__(self, context):
        self._lang = contextUtil.getLang(context)
        self._cli = contextUtil.getSSH(context)
        self._logger = logger.Logger(contextUtil.getLogger(context), __file__)
        self._cli_ret = ""
        self._err_msg = ""
        self._context = context

    def run(self):
        try:
            # step 1 检查版本
            current_version = contextUtil.getCurVersion(self._context)
            target_version = contextUtil.getTargetVersion(self._context)
            dev_type = contextUtil.getDevType(self._context)
            short_dev_type = self._trans_involve_type(dev_type)
            self._cli_ret = "current version:%s\ntarget version:%s\ndevice type:%s" % \
                            (current_version, target_version, dev_type)
            if not self._is_involve_current_version(short_dev_type, current_version) or \
                    not self._is_involve_target_version(short_dev_type, target_version):
                return CheckStatus.PASS, self._cli_ret, self._err_msg

            # step 2 检查控制器数
            return self._check_num_of_controllers(), self._cli_ret, self._err_msg

        except Exception as exception:
            self._logger.logException(exception)
            return CheckStatus.NOCHECK, self._cli_ret, baseUtil.getMsg(self._lang, 'query.result.abnormal')
        except Throwable as throwable:
            self._logger.logException(throwable)
            return CheckStatus.NOCHECK, self._cli_ret, baseUtil.getMsg(self._lang, 'query.result.abnormal')

    def _check_num_of_controllers(self):
        """
        检查控制器数量
        :return: 通过与否 CheckStatus
        """
        is_exe_successful, unsure_value, self._err_msg = cliUtil.getControllerIdList(self._cli, self._lang)
        if not is_exe_successful:
            self._cli_ret = unsure_value
            self._logger.logError("Exception to find controller number command.")
            return CheckStatus.NOCHECK
        controlList = unsure_value
        self._cli_ret += "\ncontrollers id:%s" % ",".join(str(i) for i in controlList)
        if len(controlList) > 8:
            self._err_msg = baseUtil.getMsg(self._lang, "Controllers.Check.Of.Mutil.Engine.notpass")
            return CheckStatus.NOTPASS
        return CheckStatus.PASS

    @staticmethod
    def _trans_involve_type(dev_type):
        """
        转换设备类型为简称，便于匹配涉及版本
        :param dev_type: 设备类型 str
        :return: 设备类型简称 str
        """
        dev_type = dev_type.strip()
        for short_dev_type in INVOLVE_DEV_TYPE_DICT.keys():
            dev_type_list = INVOLVE_DEV_TYPE_DICT.get(short_dev_type)
            if dev_type in dev_type_list:
                return short_dev_type
        return ""

    @staticmethod
    def _is_involve_current_version(short_dev_type, current_version):
        """
        当前版本是否涉及版本
        :param short_dev_type: 设备类型简称 str
        :param current_version:当前版本 str
        :return:True/False
        """
        if short_dev_type not in INVOLVE_CURRENT_VERSION_RANGE_DICT:
            return False
        involve_version_dict = INVOLVE_CURRENT_VERSION_RANGE_DICT.get(short_dev_type)
        return involve_version_dict.get("Min") <= current_version <= involve_version_dict.get("Max")

    @staticmethod
    def _is_involve_target_version(short_dev_type, target_version):
        """
        目标版本是否是涉及版本
        :param short_dev_type:设备类型简称 str
        :param target_version:目标版本 str
        :return: True/False
        """
        if short_dev_type not in INVOLVE_TARGET_VERSION_RANGE_DICT:
            return False
        involve_version_dict = INVOLVE_TARGET_VERSION_RANGE_DICT.get(short_dev_type)
        return involve_version_dict.get("Min") <= target_version <= involve_version_dict.get("Max")
