# -*- coding: UTF-8 -*-

from frameone.util import contextUtil
from frameone.cli import cliUtil
from frameone.util import baseUtil
from frameone.base.constants import CheckStatus
from frameone.base.exception import CliCmdException, ErrorCodeSet, ErrorCode
import traceback

HDDFW_DEV_TYPE = "Dorado5000 V3" #只涉及Dorado5000 V3型号

HDDFW_NUM = {("Dorado5000 V3", "NVME") : 35,
             ("Dorado5000 V3", "SAS") : 100,
             }

HDDFW_DEV_DEST_VER = ["V300R001C21SPC100", "V300R001C30", "V300R001C30SPC100"]

def execute(context):
    return DiskFWConfCheck(context).check()

class DiskFWConfCheck():
    def __init__(self, context):
        self.context = context
        self.logger = contextUtil.getLogger(context)
        self.cli = contextUtil.getCLI(context)
        self.lang = contextUtil.getLang(context)
        self.devType = contextUtil.getDevType(context)
        self.destVer = contextUtil.getTargetVersion(context)
        self.checkResults = []
        self.allCliRets = ""
        self.diskType = 'SAS'
        self.errMsg = ''

    def check(self):
        try:
            self.logger.info("begin to check disk fw conf.")
            self.logger.info("Current device type:%s" %self.devType)
            if HDDFW_DEV_TYPE != self.devType:
                return CheckStatus.NO_SUPPORT, "", ""

            self.logger.info("Upgrade dest version:%s" % self.destVer)
            if self.destVer not in HDDFW_DEV_DEST_VER:
                return CheckStatus.NO_SUPPORT, "", ""

            #查询硬盘类型（SAS或者NVME）
            self.queryDiskType()
            self.logger.info("Upgrade device disk type:%s" % self.diskType)

            return self._parseAllCtlrFWConf()
        except:
            self.logger.error("DiskFirmwareConfCheck exception.%s" % traceback.format_exc())
            return CheckStatus.NOCHECK, self.allCliRets, baseUtil.getMsg(self.lang, "query.result.abnormal")
        finally:
            if not self.cli.isConnected():
                self.logger.info("current cli is interrupted, now start to reconnect...")
                self.cli = contextUtil.getCLI(self.context)

    def _parseAllCtlrFWConf(self):
        failedCtlrs = []
        notPassCtlrs = []

        isSuc, cliRet, errMsg, ctlrsInfos = cliUtil.getControllerEngineTopography(self.cli, self.lang, self.logger)
        self.allCliRets += "\n%s" %cliRet
        if not isSuc:
            raise UncheckException("failed.quering.controller.topography")
        curEngineId = ctlrsInfos[0]
        curCtlrId = ctlrsInfos[1]
        engineNodeDict = ctlrsInfos[2]
        self.logger.info("[DiskFirmwareConfCheck]engine topography : " + str(engineNodeDict))
        self.maxCtlrNum = len(engineNodeDict['0'])
        self.logger.info("[DiskFirmwareConfCheck] now checking fw conf on current connected engine...")
        self.logger.info("[DiskFirmwareConfCheck]current controller id : " + curCtlrId)

        failedCtlrs, notPassCtlrs, errMsg = self._getEngineDiskFWConfs(curEngineId, curCtlrId, engineNodeDict, self.cli)
        self.errMsg += errMsg

        unreachableEngines = []
        unreachableEngines, failedCtlrsEng, notPassCtlrsOtrEng =self._checkOtherEngineCtlrFWConf(engineNodeDict, curEngineId)
        failedCtlrs.extend(failedCtlrsEng)
        notPassCtlrs.extend(notPassCtlrsOtrEng)
        self.logger.info("[DiskFirmwareConfCheck]check result unreachableEngines: " + unicode(unreachableEngines))
        self.logger.info("[DiskFirmwareConfCheck]check result failedCtlrs: " + unicode(failedCtlrs))
        self.logger.info("[DiskFirmwareConfCheck]check result notPassCtlrs: " + unicode(notPassCtlrs))
        return self.__parseResults(unreachableEngines, failedCtlrs, notPassCtlrs)

    def __parseResults(self, unreachableEngines, failedCtlrs, notPassctlrs):
        """
        :param unreachableEngines the engine numbers whom we cannot connect to.
        :param failedCtlrs: 
        :param notPassctlrs:
        """
        result = CheckStatus.PASS
        if notPassctlrs:
            result = CheckStatus.NOTPASS
            self.errMsg += baseUtil.getMsg(self.lang, "disk.fw.conf.result.notpass")
            return result, self.allCliRets, self.errMsg

        if unreachableEngines:
            result = CheckStatus.NOCHECK
            self.errMsg += baseUtil.getMsg(self.lang,
                                           "connect.engine.failed",
                                           (",".join(unreachableEngines), ",".join(unreachableEngines)))
        if failedCtlrs:
            result = CheckStatus.NOCHECK
            readableCtlrIds = map(
                lambda x: str(int(x) / self.maxCtlrNum) + ["A", "B", "C", "D"][int(x) % self.maxCtlrNum],
                failedCtlrs)

            self.errMsg += baseUtil.getMsg(self.lang,
                                           "disk.fw.conf.ctlr.failed",
                                           ",".join(readableCtlrIds))
        return result, self.allCliRets, self.errMsg

    def _checkCtlrDiskFWConf(self, sshCon, ctlrNo):
        """
        :param sshCon: the ssh connection instance of current connected controller
        :param ctlrNo: the controller number
        """
        self.logger.info("checking disk fw conf on controller:" + str(ctlrNo))
        cmd = "hdm show diskfwconf"
        hasError = True
        isStandardNum = True
        isOk, cliRet, errMsg = cliUtil.executeCmdInDebugMode(sshCon, cmd, True, self.lang)
        self.allCliRets += "\n%s" %cliRet
        if not isOk:
            hasError = True
            return (hasError, isStandardNum)

        diskfwconfRetList = cliUtil.getHorizontalDebugRet(cliRet)
        if not diskfwconfRetList:
            raise CliCmdException(ErrorCode(ErrorCodeSet.CLI_EXECUTE_CMD_FAILED))

        devHddfwStandNum = HDDFW_NUM.get((self.devType, self.diskType))
        if devHddfwStandNum and len(diskfwconfRetList) > devHddfwStandNum:
            isStandardNum = False
            return (False, isStandardNum)

        self.logger.info("ctlr %s 's checking result : hasError: %s, isStandardNum:%s" %
                         (ctlrNo, str(hasError), str(isStandardNum)))
        return (False, True)

    def _getEngineDiskFWConfs(self, curEngineId, curCtlrId, engineNodeDict, sshCon):
        """
        giving engine, Check disk fw conf on all ctlrs belonging to this engine
        :param:
        """
        failedCtlrs = []
        notPassCtlrs = []
        errMsg = ""
        self.allCliRets += "\nOn Controller %s:\n" % curCtlrId
        self.logger.info("[DiskFirmwareConfCheck]checking controller %s 's disk fw conf." % curCtlrId)
        hasError, isStandardNum = self._checkCtlrDiskFWConf(sshCon, curCtlrId)
        if hasError:
            failedCtlrs.append(curCtlrId)
        if not isStandardNum:
            notPassCtlrs.append(curCtlrId)
        for todoCtlrId in engineNodeDict[curEngineId]:
            if str(todoCtlrId) == str(curCtlrId):
                continue
            ctlrNodeId = str(int(todoCtlrId) % self.maxCtlrNum)
            self.allCliRets += "\nOn Controller %s:\n" % todoCtlrId
            if self.maxCtlrNum == 2:
                cmd = "sshtoremote"
            else:
                cmd = "sshtoremoteExt %s" % ctlrNodeId
            self.logger.info("[DiskFirmwareConfCheck]connecting to % controller" % todoCtlrId)
            passwd = self.context.get("dev").getLoginUser().getPassword()
            isSuc, echos, errMsg = cliUtil.sshToRemoteContr(sshCon, cmd, passwd, self.lang)
            self.allCliRets += "\n%s" %echos
            del passwd
            if not isSuc:
                failedCtlrs.append(todoCtlrId)
                continue

            hasError, isStandardNum = self._checkCtlrDiskFWConf(sshCon, todoCtlrId)
            if hasError:
                failedCtlrs.append(ctlrNodeId)
            if not isStandardNum:
                notPassCtlrs.append(ctlrNodeId)
            cliUtil.exitHeartbeatCli(self.cli, self.lang)

        return failedCtlrs, notPassCtlrs, errMsg

    def _checkOtherEngineCtlrFWConf(self, engineMap, excludeEng):
        """

        :param engineMap:
        :param excludeEng:
        :return:
        """
        failedEngines = []
        unreachableEngines = []
        notPassctlrs = []
        failedCtlrs = []
        isSuc, cliRet, errMsg, managementPortInfo = cliUtil.getManagementPortInfo(self.cli, self.lang)
        if not isSuc:
            self.logger.info("failed fetching management ")
            raise UncheckException("query.result.abnormal")
        engReachableIPDict = self._getAllEngineReachableIP(managementPortInfo)

        self.logger.info('All engine reachable IP:' + unicode(engReachableIPDict))

        for engineId in engineMap:
            if engineId == excludeEng:
                continue
            engMgtReachableIPList = engReachableIPDict.get(engineId)
            cilConnection = None
            for engMgtReachableIP in engMgtReachableIPList:
                self.logger.info("Begin connect to controller %s on engine %s." % (engMgtReachableIP, engineId))
                cilConnection = contextUtil.createCliConnection(self.context, engMgtReachableIP)
                if cilConnection:
                    self.logger.info("connection success.")
                    break
            if not cilConnection:
                self.logger.info("connection failed.")
                unreachableEngines.append(engineId)
                continue

            isSuc, cliRet, errMsg, ctlrsInfos = cliUtil.getControllerEngineTopography(cilConnection, self.lang,
                                                                                      self.logger)
            if not isSuc:
                failedEngines.append(engineId)
                continue
            curCtlrId = ctlrsInfos[1] #ctlrsInfos第1个值为当前连接的控制器ID
            self.logger.info("[DiskFirmwareConfCheck]current controller id : " + curCtlrId)
            failedCtlrsEng, notPassCtlrsEng, errMsg = self._getEngineDiskFWConfs(engineId, curCtlrId, engineMap,
                                                                                    cilConnection)
            failedCtlrs.extend(failedCtlrsEng)
            notPassctlrs.extend(notPassCtlrsEng)
            self.errMsg += errMsg
            cliUtil.closeCliConnection(cilConnection)
        return unreachableEngines, failedCtlrs, notPassctlrs

    def queryDiskType(self):
        '''
        @ summary 查询硬盘类型（硬盘type返回SAS时为SAS，）
        :return: 
        '''
        cmd = "show disk general |filterColumn include columnList=Type"
        status, cliRet, errMsg = cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        self.allCliRets += '\n%s' %cliRet
        if False == status:
            raise CliCmdException(ErrorCode(ErrorCodeSet.CLI_EXECUTE_CMD_FAILED))

        if cliUtil.queryResultWithNoRecord(cliRet):
            raise CliCmdException(ErrorCode(ErrorCodeSet.CLI_EXECUTE_CMD_FAILED))

        diskInfos = cliUtil.getHorizontalCliRet(cliRet)
        if status != True or not diskInfos:
            raise CliCmdException(ErrorCode(ErrorCodeSet.CLI_EXECUTE_CMD_FAILED))

        diskTypeTemp = diskInfos[0].get("Type", "")
        self.logger.info("Current device distType:%s" % diskTypeTemp)
        if self.isNvmeDisk(diskTypeTemp):
            self.diskType = "NVME"

        return

    def _getAllEngineReachableIP(self, mgtPortDictList):
        """
        @summary: Get reachable IP address of every engine.
        @param mgtPortDictList: Management port IP address information dictionary.
        @return: {engineId: reachable_IP_address}
        """
        engineIpDict = {}
        for managementPortInfoDict in mgtPortDictList:
            ip = managementPortInfoDict.get("IPv4 Address")
            portRunningStatus = managementPortInfoDict.get("Running Status")
            if portRunningStatus == "Link Up" and ip not in ["", "--"]:
                managementPortId = managementPortInfoDict.get("ID")
                engineId = managementPortId.split(".")[0][-1:]
                if "DSW" not in managementPortId:
                    engineIpList = engineIpDict.get(engineId, [])
                    engineIpList.append(ip)
                    engineIpDict.setdefault(engineId, engineIpList)

        return engineIpDict

    def isNvmeDisk(self, diskModel):
        return "NVME" in diskModel.upper()


class UncheckException(Exception):
    def __init__(self, errMsg):
        Exception.__init__(self)
        self.errMsg = errMsg