#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2019. All rights reserved.

"""
@time: 2019/11/24
@file: check_cli_software_lunOrFsReclaimTaskCheck.py
@function: 	空间回收任务存在性检查
"""

from java.lang import Throwable

from business.common import logger
from cbb.frame.cli.execute_on_all_controllers import ExecuteOnAllControllers
from cbb.frame.cli.execute_on_all_controllers import ExeOnAllCtrlContext
from cbb.frame.cli.execute_on_all_controllers import ExeOnAllCtrlException
from cbb.frame.cli.execute_on_all_controllers import FuncResult
from cbb.frame.cli.execute_on_all_controllers import ResultType
from cbb.frame.cli import cliUtil as cbbCliUtil
from frameone.base.constants import CheckStatus
from frameone.cli import cliUtil
from frameone.util import baseUtil
from frameone.util import contextUtil

RISK_TARGET_VERSION_TUPLE = [
    {"Min": "V300R006C50", "Max": "V300R006C60SPC101"}
]
RISK_CURRENT_VERSION_TUPLE = [
    {"Min": "V300R002C00", "Max": "V300R006C30"}
]
DISK_FORMAT_MODE_RISK_CURRENT_VERSION_TUPLE = [
    {"Min": "V300R003C20SPC200", "Max": "V300R003C20SPC200"},
    {"Min": "V300R006C00SPC100", "Max": "V300R006C30"}
]
DORADO_DEVICE_TYPE = "Dorado"
CONTINUE_CHECK_STATUS = 5


def execute(context):
    return lunOrFsReclaimTaskCheck(context).executingCheck()


class lunOrFsReclaimTaskCheck:
    def __init__(self, context):
        self._lang = contextUtil.getLang(context)
        self._cli = contextUtil.getSSH(context)
        self._logger = logger.Logger(contextUtil.getLogger(context), __file__)
        self._dev_type = contextUtil.getDevType(context)
        self._current_version = contextUtil.getCurVersion(context)
        self._target_version = contextUtil.getTargetVersion(context)
        self._cli_ret_list = list()
        self._err_msg = ""
        self._context = context

    def executingCheck(self):
        """
        执行空间回收任务检查
        :return: 检查结果
        """
        try:
            # step 1 型号版本检查
            if self._check_dev_basic_info() is not CONTINUE_CHECK_STATUS:
                cli_ret = self.get_cli_ret(self._cli_ret_list)
                return CheckStatus.PASS, cli_ret, self._err_msg

            # step 2 硬盘域Format Mode类型检查
            check_result = self._check_disk_domain_format_mode()
            if check_result is not CONTINUE_CHECK_STATUS:
                cli_ret = self.get_cli_ret(self._cli_ret_list)
                return check_result, cli_ret, self._err_msg

            # step 3 回收任务检查
            exe_context = ExeOnAllCtrlContext(self._context)
            check_result = self._check_reclaim_task(exe_context)
            check_status, cli_ret, self._err_msg = \
                self.trans_ExeOnAllCtrl_result_to_evalu_result(
                    check_result)

            self._cli_ret_list.append(cli_ret)
            cli_ret = self.get_cli_ret(self._cli_ret_list)
            # 添加失败处理的信息
            self._add_failed_deal_err_msg(check_status)
            return check_status, cli_ret, self._err_msg
        except ExeOnAllCtrlException as exception:
            self._logger.logError(exception.exception_msg)
            cli_ret = self.get_cli_ret(self._cli_ret_list)
            err_msg = baseUtil.getMsg(self._lang, 'query.result.abnormal')
            return CheckStatus.NOCHECK, cli_ret, err_msg
        except Exception as exception:
            self._logger.logException(exception)
            cli_ret = self.get_cli_ret(self._cli_ret_list)
            err_msg = baseUtil.getMsg(self._lang, 'query.result.abnormal')
            return CheckStatus.NOCHECK, cli_ret, err_msg
        except Throwable as throwable:
            self._logger.logException(throwable)
            cli_ret = self.get_cli_ret(self._cli_ret_list)
            err_msg = baseUtil.getMsg(self._lang, 'query.result.abnormal')
            return CheckStatus.NOCHECK, cli_ret, err_msg
        finally:
            # 退出到cli模式
            flag, _, _ = cliUtil.enterCliModeFromSomeModel(self._cli,
                                                           self._lang)
            self._logger.logInfo("Exit to cli under exe check exception.")
            # 退出失败后为不影响后续检查项重新连接cli
            if not flag:
                self._logger.logInfo("Reconnection cli under exit to cli "
                                     "failed.")
                cbbCliUtil.reConnectionCli(self._cli, self._logger)

    def trans_ExeOnAllCtrl_result_to_evalu_result(self, check_result):
        trans_dict = {ResultType.SUCCESS: CheckStatus.PASS,
                      ResultType.FAILED: CheckStatus.NOTPASS,
                      ResultType.NOT_FINISHED: CheckStatus.NOCHECK,
                      ResultType.NOT_SUPPORT: CheckStatus.NO_SUPPORT,
                      ResultType.WARNING: CheckStatus.WARNING}
        return trans_dict.get(check_result.result_type), \
            check_result.cli_ret, check_result.err_msg

    def _check_dev_basic_info(self):
        """
        检查设备基本信息（型号，目标版本，当前版本）
        :return: 检查结果
        """
        self._cli_ret_list.append(
            "current version:%s\ntarget version:%s\ndevice type:%s" %
            (self._current_version, self._target_version, self._dev_type))
        if self._is_dorado_device_type():
            return CheckStatus.PASS
        if not self._is_risk_target_version():
            return CheckStatus.PASS
        if not self._is_risk_current_version():
            return CheckStatus.PASS
        return CONTINUE_CHECK_STATUS

    def _is_dorado_device_type(self):
        """
        是否是dorado设备型号
        :return: 是否是（True/False)
        """
        return DORADO_DEVICE_TYPE in self._dev_type

    def _is_risk_current_version(self):
        """
        当前版本是否是风险的版本
        :return: 是否是（True/False)
        """
        for risk_dict in RISK_CURRENT_VERSION_TUPLE:
            if risk_dict.get("Min") <= self._current_version <= risk_dict.get(
                    "Max"):
                return True
        return False

    def _is_risk_target_version(self):
        """
        升级目标版本是否是风险的版本
        :return:
        """
        for risk_dict in RISK_TARGET_VERSION_TUPLE:
            if risk_dict.get("Min") <= self._target_version <= risk_dict.get(
                    "Max"):
                return True
        return False

    def _check_disk_domain_format_mode(self):
        """
        检查硬盘域Format Mode类型
        :return: 检查结果
        """
        if not self._is_format_mode_risk_current_version():
            return CONTINUE_CHECK_STATUS
        # 进入开发者模式
        flag, cli_ret, self._err_msg = cliUtil.enterDeveloperMode(self._cli,
                                                                  self._lang)
        self._cli_ret_list.append(cli_ret)
        if not flag:
            self._logger.logError("Enter developer mode failed.")
            return CheckStatus.NOCHECK
        # 查当前所有硬盘域ID
        exe_result, disk_domain_id_list = self._query_disk_domain_id()
        if exe_result is not CONTINUE_CHECK_STATUS:
            return exe_result
        # 检查所有硬盘域类型
        check_result = self._check_format_mode_type(disk_domain_id_list)
        # 退出开发者模式
        cliUtil.developerMode2CliMode(self._cli)
        return check_result

    def _is_format_mode_risk_current_version(self):
        """
        是否是硬盘域Format Mode相关的风险的当前版本
        :return: 是否是（True/False)
        """
        for risk_dict in DISK_FORMAT_MODE_RISK_CURRENT_VERSION_TUPLE:
            if risk_dict.get("Min") <= self._current_version <= risk_dict.get(
                    "Max"):
                return True
        return False

    def _query_disk_domain_id(self):
        """
        查询硬盘域id
        :return: 查询结果、查询的硬盘域id列表
        """
        cmd = "show disk_domain general"
        is_exe_success, cli_ret, self._err_msg = cliUtil.excuteCmdInCliMode(
            self._cli, cmd, True, self._lang)
        self._cli_ret_list.append(cli_ret)

        disk_domain_id_list = list()
        # 命令是否可用
        if not is_exe_success:
            self._logger.logError("[_query_disk_domain_id]"
                                  "Abnormal execution of commands")
            return CheckStatus.NOCHECK, disk_domain_id_list
        # 是否有执行cli命令的权限
        if not cliUtil.hasCliExecPrivilege(cli_ret):
            self._logger.logError("[_query_disk_domain_id]No permission to "
                                  "execute the cli command")
            return CheckStatus.NOCHECK, disk_domain_id_list
        # 判断回显是否为Command executed successfully
        if cliUtil.queryResultWithNoRecord(cli_ret):
            self._logger.logInfo("[_query_disk_domain_id]CliRet is 'Command "
                                 "executed successfully'")
            return CheckStatus.PASS, disk_domain_id_list
        cli_ret_lines_list = cliUtil.getHorizontalCliRet(cli_ret)
        if len(cli_ret_lines_list) == 0:
            self._err_msg = baseUtil.getMsg(self._lang,
                                            "query.result.abnormal")
            self._logger.logError("Disk domain table parse exceptional.")
            return CheckStatus.NOCHECK, disk_domain_id_list
        for cli_ret_line in cli_ret_lines_list:
            disk_domain_id_list.append(cli_ret_line.get("ID"))
        return CONTINUE_CHECK_STATUS, disk_domain_id_list

    def _check_format_mode_type(self, disk_domain_id_list):
        """
        检查硬盘域的Format mode类型
        :param disk_domain_id_list: 硬盘域id List
        :return: 检查结果
        """
        correct_format_mode_type = "Disk Extent Mix Format"
        for disk_domain_id in disk_domain_id_list:
            format_mode_type = self._query_format_mode_type(disk_domain_id)
            if len(format_mode_type) == 0:
                return CheckStatus.NOCHECK
            if correct_format_mode_type not in format_mode_type:
                return CONTINUE_CHECK_STATUS
        return CheckStatus.PASS

    def _query_format_mode_type(self, disk_domain_id):
        """
        查询硬盘域的Format mode类型
        :param disk_domain_id: 硬盘域id
        :return: Format mode类型
        """
        cmd = "show disk_domain format_mode disk_domain_id=%s" % disk_domain_id
        is_exe_success, cli_ret, self._err_msg = cliUtil.excuteCmdInCliMode(
            self._cli, cmd, True, self._lang)
        self._cli_ret_list.append(cli_ret)

        format_mode_type = ""
        # 命令是否可用
        if not is_exe_success:
            self._logger.logError("[_query_format_mode_type]Abnormal "
                                  "execution of commands")
            return format_mode_type
        # 是否有执行cli命令的权限
        if not cliUtil.hasCliExecPrivilege(cli_ret):
            self._logger.logError("[_query_format_mode_type]"
                                  "No permission to execute the cli command")
            return format_mode_type
        cli_ret_lines_list = cliUtil.getVerticalCliRet(cli_ret)
        if len(cli_ret_lines_list) == 0:
            self._err_msg = baseUtil.getMsg(self._lang,
                                            "query.result.abnormal")
            self._logger.logError("Disk domain table parse exceptional.")
            return format_mode_type
        format_mode_type = cli_ret_lines_list[0].get("Format Mode", "")
        return format_mode_type

    @staticmethod
    @ExecuteOnAllControllers
    def _check_reclaim_task(context):
        """
        回收任务的存在性检查
        :param context: 设备上下文
        :return: 检查结果
        """
        lang = context.lang
        cli = context.dev_info.cli
        log = context.logger

        # 进入debug模式
        is_exe_success, cli_ret, err_msg = cliUtil.enterDebugMode(cli, lang)

        if not is_exe_success:
            log.logError("[_check_reclaim_task_beingness]"
                         "Failed to enter developer debug view.")
            return FuncResult(ResultType.NOT_FINISHED, cli_ret, err_msg)
        cli_ret_list = list()
        cli_ret_list.append(cli_ret)
        # 是否存在回收任务
        check_result = \
            lunOrFsReclaimTaskCheck._check_existence_of_reclaim_task(cli, lang,
                                                                     log)
        cli_ret_list.append(check_result.cli_ret)
        if check_result.result_type != ResultType.FAILED:
            cliUtil.exitDebugModeToCliMode(cli)
            all_cli_ret = lunOrFsReclaimTaskCheck.get_cli_ret(cli_ret_list)
            return FuncResult(check_result.result_type, all_cli_ret,
                              check_result.err_msg)

        # 是否存在回收失败任务
        check_result = \
            lunOrFsReclaimTaskCheck._check_existence_of_failed_reclaim_task(
                cli, lang, log)
        cli_ret_list.append(check_result.cli_ret)
        cliUtil.exitDebugModeToCliMode(cli)
        all_cli_ret = lunOrFsReclaimTaskCheck.get_cli_ret(cli_ret_list)
        if check_result.result_type == ResultType.NOT_FINISHED:
            return FuncResult(ResultType.NOT_FINISHED, all_cli_ret,
                              check_result.err_msg)
        if check_result.result_type == ResultType.FAILED:
            err_msg = baseUtil.getMsg(lang, "Exist.Reclaim.Failed.Task")
            return FuncResult(ResultType.FAILED, all_cli_ret, err_msg)
        if check_result.result_type == ResultType.SUCCESS:
            err_msg = baseUtil.getMsg(lang, "Not.Exist.Reclaim.Failed.Task")
            return FuncResult(ResultType.WARNING, all_cli_ret, err_msg)

    @staticmethod
    def _check_existence_of_reclaim_task(cli, lang, log):
        """
        检查是否存在回收任务
        :param cli:  cli连接
        :param lang: 语言
        :param log: 日志
        :return: 检查结果
        """
        # 查询回收任务
        cmd = "cmm show prlist mid 215"
        is_exe_success, cli_ret, err_msg = \
            cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
        if not is_exe_success:
            log.logError("[_check_existence_of_reclaim_task]"
                         "The query reclaim task failed")
            return FuncResult(ResultType.NOT_FINISHED, cli_ret, err_msg)

        # 判断回显是否为Command executed successfully
        if cliUtil.queryResultWithNoRecord(cli_ret):
            log.logInfo("[_check_existence_of_reclaim_task]CliRet is "
                        "'Command executed successfully'")
            return FuncResult(ResultType.SUCCESS, cli_ret, err_msg)

        cli_ret_lines_list = cliUtil.getHorizontalCliRet(cli_ret)
        if len(cli_ret_lines_list) == 0:
            err_msg = baseUtil.getMsg(lang, "query.result.abnormal")
            log.logError("[_check_existence_of_reclaim_task]"
                         "Parse reclaim task cliRet exception.")
            return FuncResult(ResultType.NOT_FINISHED, cli_ret, err_msg)

        # 回收任务检查
        for cli_ret_line in cli_ret_lines_list:
            part_name = cli_ret_line.get("PartName", "").strip()
            partition_name = cli_ret_line.get("PartitionName", "").strip()
            use_cnt = cli_ret_line.get("UsedCnt", "").strip()
            if part_name == "reclaim!volroot" or partition_name == \
                    "reclaim volroot":
                if use_cnt != "0":
                    log.logInfo("[_check_existence_of_reclaim_task]"
                                "Exist reclaim task")
                    return FuncResult(ResultType.FAILED, cli_ret, err_msg)

        log.logInfo("[_check_existence_of_reclaim_task]"
                    "Not exist reclaim task")
        return FuncResult(ResultType.SUCCESS, cli_ret, "")

    @staticmethod
    def _check_existence_of_failed_reclaim_task(cli, lang, log):
        """
        检查是否存在回收失败任务
        :param cli:  cli连接
        :param lang: 语言
        :param log: 日志
        :return: 检查结果
        """
        cmd = "spa counter"
        is_exe_success, cli_ret, err_msg = \
            cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
        if not is_exe_success:
            log.logError("[_check_existence_of_failed_reclaim_task]"
                         "The query failed reclaim task failed")
            return FuncResult(ResultType.NOT_FINISHED, cli_ret, err_msg)

        # 判断回显是否为Command executed successfully
        if cliUtil.queryResultWithNoRecord(cli_ret):
            log.logInfo("[_check_existence_of_failed_reclaim_task]CliRet is "
                        "'Command executed successfully'")
            return FuncResult(ResultType.SUCCESS, cli_ret, err_msg)

        cli_ret_lines_list = cliUtil.getHorizontalCliRet(cli_ret)
        if len(cli_ret_lines_list) == 0:
            err_msg = baseUtil.getMsg(lang, "query.result.abnormal")
            log.logError("[_check_existence_of_failed_reclaim_task]"
                         "Parse failed reclaim task cliRet exception.")
            return FuncResult(ResultType.NOT_FINISHED, cli_ret, err_msg)
        # 回收失败任务检查
        for cli_ret_line in cli_ret_lines_list:
            step_name = cli_ret_line.get("Step Name", "").strip()
            return_bad = cli_ret_line.get("Return Bad", "").strip()
            if step_name == "reclaim vol end":
                if return_bad != "0":
                    log.logInfo("[_check_existence_of_failed_reclaim_task]"
                                "Exist reclaim failed task")
                    return FuncResult(ResultType.FAILED, cli_ret, "")
        log.logInfo("[_check_existence_of_failed_reclaim_task]"
                    "Not exist reclaim failed task")
        return FuncResult(ResultType.SUCCESS, cli_ret, "")

    def _add_failed_deal_err_msg(self, check_result):
        """
        添加失败处理的信息
        :param check_result: 回收任务检查结果
        """
        if check_result == CheckStatus.NOTPASS:
            err_msg = baseUtil.getMsg(self._lang, "Existence.Space.Reclaim."
                                                  "Task.notpass")
            self._err_msg = "%s\n%s" % (self._err_msg, err_msg)
        elif check_result == CheckStatus.WARNING:
            err_msg = baseUtil.getMsg(self._lang, "Existence.Space.Reclaim."
                                                  "Task.warning")
            self._err_msg = "%s\n%s" % (self._err_msg, err_msg)

    @staticmethod
    def get_cli_ret(cli_ret_list):
        """
        返回原始信息
        :return: 回显
        """
        if cli_ret_list:
            return "\n".join(cli_ret_list)
        else:
            return "No original cli information"
