# -*- coding: UTF-8 -*-
from frameone.cli import cliUtil
from business.common import common
from frameone.util import contextUtil

# 多硬盘域风险检查风险版本
MULTI_DD_RISK_VERSION = {
    "V300R002C00SPC100": "V300R002C00SPH105",
    "V300R002C10SPC100": "V300R002C10SPH112",
    "V300R002C20": "V300R002C20SPH002",
}

PRODUCT_RET = "Product Version:{}\nHotPatch Version:{}\n"


def execute(context):
    """
    多硬盘域风险检查
    :param context:上下文
    :return:
    """
    lang = contextUtil.getLang(context)
    logger = common.getLogger(contextUtil.getLogger(context), __file__)
    cli = contextUtil.getSSH(context)

    pool_multi_dd_check = PoolMultiDDCheck(cli, lang, logger, context)
    flag = pool_multi_dd_check.execute_check()
    return common.getUpgEvaluationRs(
        flag,
        "\n".join(pool_multi_dd_check.all_ret_list),
        pool_multi_dd_check.err_msg,
    )


class PoolMultiDDCheck:
    def __init__(self, cli, lang, logger, context):
        self.cli = cli
        self.lang = lang
        self.logger = logger
        self.context = context
        self.all_ret_list = []
        self.err_msg = ""
        self.disk_domain_id_list = []
        self.ctrl_domain_dict = {}

    def execute_check(self):
        """
        检查总入口
        :return: 是否通过，错误提示
        """
        try:
            flag, p_version, p_patch = cliUtil.getSystemVersion(
                self.cli, self.lang
            )
            self.all_ret_list.append(PRODUCT_RET.format(p_version, p_patch))
            if not self.is_risk_version(p_version, p_patch):
                return True

            flag = self.parse_disk_domain_ids()
            self.logger.logInfo(
                "domain info is:{}".format(str(self.disk_domain_id_list))
            )
            if flag is not True:
                return flag

            # 获取所有domain详情
            for domain_id in self.disk_domain_id_list:
                flag = self.get_disk_domain_detail_info(domain_id)
                self.logger.logInfo(
                    "ctrl and domain info:{}".format(
                        str(self.ctrl_domain_dict)
                    )
                )
                if flag is not True:
                    return flag

            return self.check_multi_dd()
        except Exception as e:
            self.logger.logException(e)
            err_key = "query.result.abnormal"
            self.err_msg = common.getMsg(self.lang, err_key)
            return cliUtil.RESULT_NOCHECK

    def check_multi_dd(self):
        """
        检查控制器所属的硬盘域是否大于1个
        :return: True: 通过，False 不通过
        """
        err_msg_list = []
        err_key = "pool.multidd.not.pass"
        for ctrl, domain_list in self.ctrl_domain_dict.items():
            if len(domain_list) <= 1:
                continue
            err_msg_list.append(
                common.getMsg(
                    self.lang, err_key, (ctrl, ",".join(domain_list)),
                )
            )
        if err_msg_list:
            self.err_msg = "".join(err_msg_list)
            return False
        return True

    def is_risk_version(self, p_version, p_patch):
        """
        检查是否风险版本
        :param p_version: 版本信息
        :param p_patch: 补丁信息
        :return: False 非风险；True 有风险
        """
        if p_version not in MULTI_DD_RISK_VERSION:
            return False

        if p_patch == "--":
            return True

        if p_patch >= MULTI_DD_RISK_VERSION.get(p_version):
            return False

        return True

    def parse_disk_domain_ids(self):
        """
        获取硬盘域信息
        :return:
        """
        cmd = "show disk_domain general"
        flag, ret, self.err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, False, self.lang
        )
        self.all_ret_list.append(ret)
        if flag is not True:
            return flag

        # 没有硬盘域信息
        if cliUtil.queryResultWithNoRecord(ret):
            return True

        cli_ret_line = cliUtil.getHorizontalCliRet(ret)
        for line in cli_ret_line:
            disk_domain_id = line.get("ID")
            self.disk_domain_id_list.append(disk_domain_id)
        return True

    def get_disk_domain_detail_info(self, domain_id):
        """
        获取硬盘域控制器和domain对应信息
        :return:
        """
        cmd = "show disk_domain general disk_domain_id={}".format(domain_id)
        flag, ret, self.err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, False, self.lang
        )
        self.all_ret_list.append(ret)
        if flag is not True:
            return flag

        ret_info_list = cliUtil.getVerticalCliRet(ret)
        ctrl_list = []
        for line in ret_info_list:
            ctrl_list = line.get("Controller", "").split(",")
        for ctrl in ctrl_list:
            tmp_list = self.ctrl_domain_dict.get(ctrl, [])
            tmp_list.append(domain_id)
            self.ctrl_domain_dict[ctrl] = list(set(tmp_list))
        return True
