# -*- coding: UTF-8 -*-
from business.common import common
from frameone.cli import cliUtil
from frameone.base.constants import CheckStatus
from frameone.util import baseUtil
from frameone.util import contextUtil

RISK_VERSION_DICT = {
    "V300R002C20": "V300R002C20SPH005",
    "V300R006C60": "",
    "V300R006C61": "V300R006C61SPH005",
    "V500R007C50": "",
    "V500R007C61": "V500R007C61SPH005"}

def execute(context):

    """
        背景：
        20200703德国大众金融VWFS 18800 V5 V500R007C61存储性能问题，
        以及俄罗斯内政部Dorado V3 V300R002C20性能问题，都是1822 FC固件缺陷。
        主机读业务，芯片收到cmd命令到返回第一帧data的过程中FC链路发生IO超时，
        芯片收到驱动下发的abort命令，abort处理中链头超时IO没有摘链，
        导致链表后续IO不能调度，这条链接上的IO超时。
        buglist为BUGLIST202006230005。
        检查标准：
        1确认升级目标版本是否为DoradoV300R002C20,V300R006C60,
        V300R006C61,V500R007C61,V500R007C50，如果不是则检查结果为通过，
        否则继续检查；
        2 若步骤2中存在FC端口Max Speed为32000的记录，
        则检查结果为不通过，否则检查结果为通过。
    :param context:
    :return:
    """

    smart_io_fc_check = SmartIOFcCheck(context)
    flag, msg = smart_io_fc_check.check()
    return flag, "\n".join(smart_io_fc_check.all_ret_list), msg


class SmartIOFcCheck:
    def __init__(self, context):
        self.context = context
        self.cli = contextUtil.getCLI(context)
        self.lang = contextUtil.getLang(context)
        self.logger = common.getLogger(
            contextUtil.getLogger(context), __file__)
        self.all_ret_list = list()
        self.common_err_msg = common.getMsg(self.lang, "query.result.abnormal")

    def check(self):
        try:
            dest_version = contextUtil.getTargetVersion(self.context)
            patch_version = self.context.get('dev').getUpgPatchPkgVer()
            if not patch_version:
                patch_version = '--'
            self.all_ret_list.append(
                "Upgrade package version:{}\nUpgrade patch "
                "package version:{}".format(dest_version, patch_version))
            self.logger.logInfo("Upgrade package version:{};\nUpgrade patch "
                                "package version:{}.".format(dest_version,
                                                             patch_version))
            # 判断是否为风险版本
            if self.check_risk_version(dest_version, patch_version):
                return CheckStatus.PASS, ""

            not_pass_id_list = self.get_1822_fc_card_id()
            if not not_pass_id_list:
                return CheckStatus.PASS, ''
            err_msg = common.getMsg(self.lang, "fc.smart.io.check.warning",
                                    (dest_version,
                                     ', '.join(not_pass_id_list)))
            return CheckStatus.NOTPASS, err_msg
        except Exception as e:
            self.logger.logException(e)
            return CheckStatus.NOCHECK, self.common_err_msg

    def get_1822_fc_card_id(self):
        """获取1822的FC卡信息
        :return: 返回1822的fc卡ID
        """
        cmd = "show port general physical_type=FC"
        flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang)
        self.all_ret_list.append(cli_ret)
        if flag is not True:
            raise Exception("can not get fc cards")

        cards = cliUtil.getHorizontalNostandardCliRet(cli_ret)
        not_pass_id_list = list()
        for card in cards:
            card_id = card.get("ID", "--")
            max_speed = card.get("Max Speed(Mbps)", "")
            if max_speed.strip() == "32000":
                not_pass_id_list.append(card_id)

        self.logger.logInfo(
            "board_cards is {}".format(str(not_pass_id_list)))
        return not_pass_id_list

    def check_risk_version(self, soft_version, patch_version):
        """
        检查是否为风险版本
        :param soft_version: 目标软件版本
        :param patch_version: 目标补丁版本
        :return: 有风险的为False
        """
        soft_version_no_spc = soft_version.split("SPC")[0]
        if soft_version_no_spc in RISK_VERSION_DICT:
            risk_hot_patch = RISK_VERSION_DICT.get(soft_version)
            if risk_hot_patch and patch_version >= risk_hot_patch:
                return True
            return False
        return True
