# -*- coding: UTF-8 -*-

from frameone.util import contextUtil
from function import StorageContextUtil
from cbb.frame.base.constants import CheckStatus
from cbb.frame.base import baseUtil
from cbb.frame.cli.cliUtil import get_system_version_with_ret, exist_sas_interface_card

# 针对升级到目标版本问题检查
TARGET_VERSION_PATCH_CHECK = {
    "V500R007C70SPC100 Kunpeng": {
        "need_patch": "",
        "condition": "is_high_end_sas_card"
    },
    "V500R007C70SPC200 Kunpeng": {
        "need_patch": "",
        "condition": "is_high_end_sas_card"
    },
    "V5R7C60SPC100 Kunpeng": {
        "need_patch": "V500R007C60SPH115",
        "condition": "can_not_upgrade_to_6_1_0"
    },
    "V500R007C60SPC300 Kunpeng": {
        "need_patch": "V500R007C60SPH335",
        "condition": "can_not_upgrade_to_6_1_0"
    },
    "V500R007C71SPC100 Kunpeng": {
        "need_patch": "V500R007C71SPH152",
        "condition": "can_not_upgrade_to_6_1_0"
    },
    "V500R007C73SPC100 Kunpeng": {
        "need_patch": "V500R007C73SPH123",
        "condition": "can_not_upgrade_to_6_1_0"
    },
    "6.1.0": {
        "need_patch": "SPH36",
        "condition": "can_not_upgrade_to_6_1_0"
    },
    "6.0.1": {
        "need_patch": "SPH35",
        "condition": "can_not_upgrade_to_6_1_0"
    },
    "6.0.0": {
        "need_patch": "SPH11",
        "condition": "can_not_upgrade_to_6_1_0"
    },
    "6.1.RC3": {
        "need_patch": "SPH1",
        "condition": "can_not_upgrade_to_6_1_0"
    },
    # 升级到6.1.2需要升级中安装6.1.2.SPH10及之后补丁BUGLIST202204290021
    "6.1.2": {
        "need_patch": "SPH66",
        "condition": "can_not_upgrade_to_6_1_0"
    },
    "6.1.3": {
        "need_patch": "SPH31",
        "condition": "can_not_upgrade_to_6_1_0"
    },
    "6.1.5": {
        "need_patch": "SPH33",
        "condition": "can_not_upgrade_to_6_1_0",
        "other": {"patch_version": "SPH1",
                  "product_model": ["OceanStor Dorado 2000", "OceanStor 2910"]}
    },
    "6.1.6": {
        "need_patch": "SPH18",
        "condition": "can_not_upgrade_to_6_1_0",
        "other": {"patch_version": "--",
                  "product_model": ["OceanStor 2910"]}
    },
    "6.1.7": {
        "need_patch": "SPH6",
        "condition": "can_not_upgrade_to_6_1_0",
        "other": {"patch_version": "--",
                  "product_model": ["OceanStor 2910"]}
    },
}


def execute(context):
    return HotPatchBeforeUpgrade(context).check()


class HotPatchBeforeUpgrade(object):
    def __init__(self, context):
        self.context = context
        self.logger = contextUtil.getLogger(context)
        self.cli = contextUtil.getCLI(context)
        self.lang = contextUtil.getLang(context)
        self.all_ret = []
        self.err_msg = ''
        self.dev = context.get("dev")
        self.dest_version = str(self.dev.getDestVersion())
        self.upgrade_mode = str(self.dev.getUpgradeMode())
        self.dev_type = str(self.dev.getDeviceType())
        self.current_version = ""
        self.patch_version = ""

    def check(self):
        try:
            if self.dest_version not in TARGET_VERSION_PATCH_CHECK:
                self.logger.info("not involved version:{}".format(self.dest_version))
                return CheckStatus.PASS, "", ""
            self.ini_basic_info()
            status, err_msg = self.check_target_version_patch()
            return status, "\n".join(self.all_ret), err_msg
        except Exception as e:
            self.logger.error("Hot patch check exception.{}".format(str(e)))
            return CheckStatus.NOCHECK, "\n".join(self.all_ret), baseUtil.getPyResource(self.lang,
                                                                                        "query.result.abnormal")

    def ini_basic_info(self):
        self.logger.info("begin to check hot patch.")
        flag, self.current_version, self.patch_version, cli_ret = get_system_version_with_ret(self.cli, self.lang)
        self.all_ret.append(cli_ret)
        if not flag:
            raise Exception("query version failed.")
        self.logger.info("current_version={},patch_version={},dest_version={},upgrade_mode={}"
                         .format(self.current_version, self.patch_version, self.dest_version, self.upgrade_mode))

    @staticmethod
    def compare_patch_version(version1, version2):
        if version1.startswith("SPH") and version2.startswith("SPH"):
            version1 = int(version1.replace("SPH", "").strip())
            version2 = int(version2.replace("SPH", "").strip())
        if version1 == version2:
            return 0
        return 1 if version1 > version2 else -1

    def check_target_version_patch(self):
        version_config = TARGET_VERSION_PATCH_CHECK.get(self.dest_version)
        need_patch_ver = version_config.get("need_patch", "")

        # 升级场景已选择目标版本对应的补丁包
        is_pre_upgrade_scene = StorageContextUtil.isPreUpgradeCheckScene(self.context)
        select_patch_ver = str(self.dev.getUpgPatchPkgVer()).strip()
        self.logger.info("is pre upgrade scene:{}, select_patch_ver={}".format(
            is_pre_upgrade_scene, select_patch_ver))

        other = version_config.get("other")
        self.logger.info("current version exist other scene:{}".format(other))
        if not is_pre_upgrade_scene and self.dev.getUpgPatchPkgVer() and need_patch_ver:
            # 是否存在例外的场景
            if self.check_target_version(other, select_patch_ver, need_patch_ver):
                self.logger.info("has need install patch.")
                return CheckStatus.PASS, ""

        condition = version_config.get("condition", "")
        self.logger.info("condition={}".format(condition))
        if not condition:
            self.logger.info("condition is none")
            return CheckStatus.NOTPASS, self.get_target_version_risk_msg(need_patch_ver)
        condition_method = getattr(self, condition)
        if condition_method(version_config):
            self.logger.info("condition is meet.")
            if self.need_other_patch_version(other):
                need_patch_ver = other.get("patch_version")
            if need_patch_ver == "--":
                return CheckStatus.PASS, ""
            return CheckStatus.NOTPASS, self.get_target_version_risk_msg(need_patch_ver)
        self.logger.info("condition is not meet.")
        return CheckStatus.PASS, ""

    def get_target_version_risk_msg(self, need_patch_ver):
        if need_patch_ver:
            return baseUtil.getPyResource(self.lang, "upgrade.need.select.patch").format(need_patch_ver)
        return baseUtil.getPyResource(self.lang, "upgrade.target.exist.risk")

    # 反射调用
    def is_high_end_sas_card(self, version_config):
        self.logger.info("condition config={}".format(version_config))
        if not baseUtil.isV5V6HighEnd(self.dev_type):
            return False
        flag, exist_sas_card, cli_ret = exist_sas_interface_card(self.cli, self.lang)
        self.all_ret.append(cli_ret)
        if not flag:
            raise Exception("query sas card failed.")
        return exist_sas_card

    def can_not_upgrade_to_6_1_0(self, version_config):
        self.logger.info("condition config={}".format(version_config))
        need_patch_ver = version_config.get("need_patch", "")
        if self.upgrade_mode.upper() == "APOLLO" \
                and self.compare_patch_version(self.patch_version,
                                               need_patch_ver) >= 0:
            return False
        return True

    def need_other_patch_version(self, other):
        return other and self.dev_type in other.get("product_model")

    def check_target_version(self, other, select_patch_ver, need_patch_ver):
        if other:
            other_patch_version = other.get("patch_version")
            other_product_models = other.get("product_model")
            if self.dev_type in other_product_models:
                return self.compare_patch_version(select_patch_ver, other_patch_version) >= 0
            return self.compare_patch_version(select_patch_ver, need_patch_ver) >= 0
        return self.compare_patch_version(select_patch_ver, need_patch_ver) >= 0
