# -*- coding: UTF-8 -*-

from frameone.util import contextUtil
from cbb.frame.base import baseUtil
from frameone.base.constants import CheckStatus
from frameone.cli import cliUtil
from cbb.frame.cli.cliUtil import get_system_version_with_ret

# 检查设备版本范围
DEV_VERSION_LIST = ["V300R001C00SPC100", "V300R001C01SPC100",
                    "V300R001C21SPC100"]
# 升级目标版本的范围
TARGET_VERSION_LIST = ["V300R001C30SPC100", "V300R002C00SPC100",
                       "V300R002C10SPC100", "V300R002C20"]

ARRAY_CURRENT_VERSION_LIST = ["V300R001C01SPC100", "V300R001C21SPC100"]

ARRAY_TARGET_VERSION = "V300R002C10SPC100"

HOST_LUN_0_PATCH_PKG_VER = "V300R002C10SPH118"


def execute(context):
    """
    检查方法：
    步骤1 以admin用户登录设备；
    步骤2 执行命令：show upgrade package，查询系统软件版本信息；
    步骤3 执行命令：show host general，查询并记录阵列创建的所有逻辑主机[Host ID]；
    步骤4 执行命令：show host lun host_id=[Host ID]，Host ID为步骤3中记录的主机Host ID，查询并记录主机映射的Host LUN ID；
    检查标准：
    1 若步骤2中系统软件为Dorado V300R001C00SPC100、Dorado V300R001C01SPC100、Dorado V300R001C21SPC100，则继续检查，否则检查结果为通过；
    2、若升级的目标版本为：Dorado V300R001C30SPC100、Dorado V300R002C00SPC100、Dorado V300R002C10SPC100、Dorado V300R002C20,则继续检查，否则检查通过。
    3 若步骤4中记录的Host LUN ID有值为0的情况，则为检查不通过，否则检查通过；
    修复建议：
    若检查结果不通过，请参照以下方式处理：
    1、若升级的目标版本为Dorado V300R002C10SPC100
        a、若当前系统版本为Dorado V300R001C01SPC100或Dorado V300R001C21SPC100，请在升级策略设置时同时选择Dorado V300R002C10SPH118及之后补丁进行升级；
        b、若当前系统版本为Dorado V300R001C00SPC100，请先升级到Dorado V300R001C21SPC100，然后再升级到目标版本，并在升级策略设置时同时选择Dorado V300R002C10SPH118及之后补丁进行升级。
    2、若升级的目标版本为其他版本，请联系技术支持工程师。

    :param context:
    :return:
    """
    return CheckConfigHostLunId0(context).check()


class CheckConfigHostLunId0():
    def __init__(self, context):
        self.context = context
        self.logger = contextUtil.getLogger(context)
        self.cli = contextUtil.getCLI(context)
        self.lang = contextUtil.getLang(context)
        self.target_version = contextUtil.getTargetVersion(context)
        self.all_ret = []
        self.err_msg = ''

    def check(self):
        try:
            self.logger.info("begin to check upgrade host lun id 0.")
            flag, software_version, patch_version, cli_ret = get_system_version_with_ret(
                self.cli, self.lang)
            self.all_ret.append(cli_ret)
            self.logger.info(
                "upgrade_hostlunid0 software_version:{}".format(
                    software_version))
            if not flag:
                self.err_msg = baseUtil.getPyResource(self.lang,
                                                      "query.result.abnormal")
                self.logger.error("Version query result abnormal.")
                return CheckStatus.NOCHECK, self.get_all_ret(), self.err_msg
            # 1 若步骤2中系统软件为Dorado V300R001C00SPC100、Dorado V300R001C01SPC100、Dorado V300R001C21SPC100，
            # 则继续检查，否则检查结果为通过；
            if software_version not in DEV_VERSION_LIST:
                self.logger.info("software_version not in limit.")
                return CheckStatus.PASS, self.get_all_ret(), ""
            # 2、若升级的目标版本为：Dorado V300R001C30SPC100、Dorado V300R002C00SPC100、Dorado V300R002C10SPC100、Dorado V300R002C20,
            # 则继续检查，否则检查通过。
            self.logger.info("target_version:{}".format(self.target_version))
            if self.target_version not in TARGET_VERSION_LIST:
                return CheckStatus.PASS, self.get_all_ret(), ""

            flag = self.check_array_upg_patch(software_version)
            self.logger.info("check_array_upg_patch result:{}".format(flag))
            if flag:
                return CheckStatus.PASS, self.get_all_ret(), ""

            # 3、查询并记录阵列创建的所有逻辑主机[Host ID]；
            flag, host_ids, err_msg = self.get_host_id_list()
            if flag is not True:
                return flag, self.get_all_ret(), err_msg
            self.logger.info("get_host_ids hostids:{}".format(host_ids))
            flag, host_id_list, err_msg = self.get_host_lun_id_0_host(host_ids)
            if flag is not True:
                return flag, self.get_all_ret(), err_msg
            if not host_id_list:
                return CheckStatus.PASS, self.get_all_ret(), ""
            err_msg = baseUtil.getPyResource(self.lang, "check.config.host.lun.id.0.notpass")
            return CheckStatus.NOTPASS, self.get_all_ret(), err_msg
        except Exception as e:
            self.logger.error("HotpatchBeforeUpgrade exception.{}".format(str(e)))
            return CheckStatus.NOCHECK, self.get_all_ret(), baseUtil.getMsg(
                self.lang,
                "query.result.abnormal")

    def check_array_upg_patch(self, software_version):
        """
        升级的目标版本为Dorado V300R002C10SPC100
        若当前系统版本为Dorado V300R001C01SPC100或Dorado V300R001C21SPC100，
        请在升级策略设置时同时选择Dorado V300R002C10SPH118及之后补丁进行升级，检查通过；
        :param software_version:当前版本
        :return: 设备升级过程中打补丁的结果
        """

        if software_version not in ARRAY_CURRENT_VERSION_LIST:
            return False
        if self.target_version != ARRAY_TARGET_VERSION:
            return False

        dev = contextUtil.getDevObj(self.context)
        upg_patch_pkg_ver = dev.getUpgPatchPkgVer()
        if not upg_patch_pkg_ver:
            return False
        self.all_ret.append("Patch package version:{}".format(upg_patch_pkg_ver))
        # 设备升级中升级过程中打补丁 Dorado V300R002C10SPH118及之后补丁进行升级
        upg_patch_pkg_vrc_ver = baseUtil.getVRCVersion(upg_patch_pkg_ver)
        host_lun_0_patch_vrc_ver = baseUtil.getVRCVersion(HOST_LUN_0_PATCH_PKG_VER)
        self.logger.info("upg_patch_pkg_vrc_ver:{} host_lun_0_patch_vrc_ver:{}".format(
                upg_patch_pkg_ver, host_lun_0_patch_vrc_ver))
        if upg_patch_pkg_vrc_ver == host_lun_0_patch_vrc_ver and upg_patch_pkg_ver >= HOST_LUN_0_PATCH_PKG_VER:
            return True

    def get_all_ret(self):
        return "\n".join(self.all_ret)

    def get_host_id_list(self):
        """步骤3 执行命令：show host general，查询并记录阵列创建的所有逻辑主机[Host ID]；
            :return: host id列表
                """
        host_ids = []
        cmd = "show host general"
        flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(self.cli, cmd, True,
                                                            self.lang)
        self.all_ret.append(cli_ret)
        if flag is not True:
            self.err_msg = baseUtil.getPyResource(self.lang,
                                                  "query.result.abnormal")
            self.logger.error("get_host_ids query result abnormal.")
            return flag, host_ids, self.err_msg
        hosts = cliUtil.getHorizontalCliRet(cli_ret)
        self.logger.info("get_host_ids hosts:{}".format(hosts))
        host_ids = [host.get("ID") for host in hosts]
        return True, host_ids, ""

    def get_host_lun_id_0_host(self, host_ids):
        """
        若步骤4中记录的Host LUN ID有值为0的情况，则为检查不通过，否则检查通过；
        :param host_ids:
        :return:
        """
        self.logger.info("get_hostlunid0_hosts start.")
        host_id_list = []
        for host_id in host_ids:
            flag, host_id, err_msg = self.check_host_lun_id_by_host(host_id)
            if flag is not True:
                return flag, host_id_list, err_msg
            if host_id:
                host_id_list.append(host_id)
        return True, host_id_list, ""

    def check_host_lun_id_by_host(self, host_id):
        """
        查看记录的Host LUN ID有值为0的host id
        :param host_id: 主机ID
        :return:  flag，host_id主机ID，err_msg错误信息
        """
        self.logger.info("check_hostlunid_by_host start.")
        cmd = "show host lun host_id={}".format(host_id)
        flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(self.cli, cmd, True,
                                                            self.lang)
        self.all_ret.append(cli_ret)
        if flag is not True:
            self.err_msg = baseUtil.getPyResource(self.lang,
                                                  "query.result.abnormal")
            self.logger.error("check_hostlunid_by_host query result abnormal.")
            return CheckStatus.NOCHECK, "", self.err_msg
        hostlun_infos = cliUtil.getHorizontalCliRet(cli_ret)
        for host_lun_info in hostlun_infos:
            host_lun_id = host_lun_info.get("Host LUN ID")
            if host_lun_id == "0":
                return True, host_id, ""
        return True, "", ""
