# -*- coding: utf-8 -*-


from business.common.base_check import BaseCheckItem
from frameone.base.constants import CheckStatus
from frameone.util import baseUtil
from cbb.frame.base import config
from cbb.frame.rest import restData
from cbb.frame.rest import restUtil
from cbb.business.operate.fru.common import FuncFactory
from utils import Products
from java.lang import Exception as JException

# 不兼容的接口卡
INTERFACE_NOT_COMPATIBLE_ENUM = "2388"


def execute(context):
    """
    接口卡兼容性检查
    @param context: 上下文
    @return: 检查是否通过
    """
    return CheckInterfaceCompatible(context).execute_check()


class CheckInterfaceCompatible(BaseCheckItem):

    def execute_check(self):
        try:
            # 检查型号版本
            if not self.is_risk_model_and_version():
                return CheckStatus.NO_SUPPORT, self.get_cli_ret(), ""
            # 检查接口卡
            if self.check_interface():
                return CheckStatus.PASS, self.get_cli_ret(), ""
            err_msg = baseUtil.getMsg(self._lang, "check.intf.compatible.notpass")
            return CheckStatus.NOTPASS, self.get_cli_ret(), err_msg
        except (Exception, JException) as exception:
            self._logger.logException(exception)
            return CheckStatus.NOCHECK, self.get_cli_ret(), baseUtil.getMsg(self._lang, "query.result.abnormal")

    def check_interface(self):
        # 获取所有接口卡，
        interface_list = FuncFactory.getFruListInfo(self._context, restData.Enum.ObjEnum.INTF_MODULE)
        # 检查是否有100G FE NoF接口卡,枚举：2388，如果有，返回False
        for interface_info in interface_list:
            model = restUtil.Tlv2Rest.getRecordValue(interface_info, restData.Hardware.IntfModule.MODEL)
            if str(model) == INTERFACE_NOT_COMPATIBLE_ENUM:
                return False
        return True

    def is_risk_model_and_version(self):
        # 不是Dorado V6高端或者新融合高端，则不是风险型号
        self._cli_ret_list.append("dev type: {}".format(self._dev_type))
        if self._dev_type not in config.DORADO_DEVS_V6_HIGH or self._dev_type in config.OCEAN_PROTECT:
            return False
        # 如果软件版本大于6.1.6，目标版本不大于6.1.6，则不是风险版本
        self._cli_ret_list.append("dev verison: {}, target_version: {}".format(self._current_version,
                                                                               self._target_version))
        # 如果起始版本大于等于6.1.7RC1，则不是风险版本（6.1.99小于6.1.7RC1）
        if Products.compareVersion(self._current_version, "6.1.7RC1") >= 0 \
                and not self._current_version.startswith("6.1.99"):
            return False
        # 如果目标版本大于等于6.1.7RC1，则是风险版本，否则不是风险版本（6.1.99小于6.1.7RC1）
        return Products.compareVersion(self._target_version, "6.1.7RC1") >= 0 \
            and not self._target_version.startswith("6.1.99")

    def get_cli_ret(self):
        """
        返回原始信息
        :return: 回显
        """
        return "\n\n".join(self._cli_ret_list) if self._cli_ret_list else "No original cli information"
