# -*- coding: UTF-8 -*-

import traceback
from frameone.util import contextUtil, baseUtil
from frameone.base.constants import CheckStatus, CheckResult
from frameone.adapter.resourceService import ResrouceFactory

offLineCurrentVersionList = ["V300R001C00", "V300R001C00SPC100", "V300R001C01"]

def execute(dataDict):
    checkResult = IsRssStatusSupportOnline(dataDict).check()
    return checkResult.toScriptResult()

class IsRssStatusSupportOnline():
    
    def __init__(self, context):
        self.logger = contextUtil.getLogger(context)
        self.lang = contextUtil.getLang(context)
        self.currentVersion = contextUtil.getCurVersion(context)
        self.devType = contextUtil.getDevType(context)
        self.context = context
        self.originInfos = []
        self.commonMsg = baseUtil.getMsg(self.lang, "query.result.abnormal")

    def check(self):
        try:
            # 离线版本检查
            setedModel = contextUtil.getUpgradeModel(self.context)
            if setedModel == "OFFLINE":
                upgradeVersion = contextUtil.getTargetVersion(self.context)
                self.logger.info("upgradeVersion is  %s,currentVersion is %s" % (upgradeVersion, self.currentVersion))
                if not baseUtil.isDoradoDev(self.devType) or self.currentVersion not in offLineCurrentVersionList or upgradeVersion < "V300R001C01SPC100":
                    return CheckResult(CheckStatus.NO_SUPPORT, self.originInfos)

            self.resrouceFactory = ResrouceFactory(self.context)
            recs = self.resrouceFactory.executeInterface("excuteDiagnoseCmd", "rsf showupdate", 1)
            retMap = self.parseRecs(recs)
            if not retMap:
                # 没有数据
                return CheckResult(CheckStatus.NOCHECK, self.originInfos, self.commonMsg)
            # 只会有1条数据
            status, errMsg = retMap.values()[0]
            return CheckResult(status, self.originInfos, errMsg)
        except:
            self.logger.error("rssStatusSupportOnline check excption [%s]" % traceback.format_exc())
            return CheckResult(CheckStatus.NOCHECK, self.originInfos, self.commonMsg)
    
    def parseRecs(self, recs):
        retMap = {}
        CLONE_FAIL_KEY = "CLONE.fail"
        CHECK_NOTPASS_KEY = "ISRssStatusSupportOnline.notpass"

        if baseUtil.isDoradoDev(self.devType):
            CLONE_FAIL_KEY = "CLONE.fail.dorado"
            if self.currentVersion < "V300R001C21":
                CHECK_NOTPASS_KEY = "ISRssStatusSupportOnline.notpass.beforeDoradoC21"
            else:
                CHECK_NOTPASS_KEY = "ISRssStatusSupportOnline.notpass.dorado"

        for rec in recs:
            nodeId = self.resrouceFactory.executeInterface("getParamStrValue", rec, 0)
            result = self.resrouceFactory.executeInterface("getParamStrValue", rec, 1)
            self.originInfos.append("nodeId=%s, result=%s" % (nodeId, result))
            self.logger.info("result:" + result)
            if result == "Rss can update.":
                #执行成功
                itemState = CheckStatus.PASS
                suggession = baseUtil.getMsg(self.lang, "ISRssStatusSupportOnline.pass")
            elif result == "Rss can not update.":
                #执行失败
                itemState = CheckStatus.NOTPASS
                suggession = baseUtil.getMsg(self.lang, CHECK_NOTPASS_KEY)
            elif result == "TimeOut":
                #执行超时
                itemState = CheckStatus.NOCHECK
                suggession = baseUtil.getMsg(self.lang, "ISRssStatusSupportOnline.timeout")
            else:
                itemState = CheckStatus.NOCHECK
                suggession = baseUtil.getMsg(self.lang, "ISRssStatusSupportOnline.error")
                (ok, errorKeyLst) = self.failureCharacteristics(result)
                self.logger.info("failureCharacteristics ok:" + str(ok))
                if ok:
                    itemState = CheckStatus.NOTPASS
                    suggession = ""
                    suggession_common = baseUtil.getMsg(self.lang, "common.fail")
                    for oneError in errorKeyLst:
                        oneErrorRes = baseUtil.getMsg(self.lang, oneError)
                        if oneError == "CLONE.fail":
                            oneErrorRes = baseUtil.getMsg(self.lang, CLONE_FAIL_KEY)
                        suggession += '\n %s) ' % str(errorKeyLst.index(oneError)+1) + oneErrorRes
                    suggession = suggession_common + suggession

                elif "fail.v" in result.lower():
                    itemState = CheckStatus.PASS
                    suggession = baseUtil.getMsg(self.lang, "ISRssStatusSupportOnline.pass")
            retMap[nodeId] = [itemState, suggession]
        return retMap

    def failureCharacteristics(self, result):
        failureLst = ["SNAPSHOT.fail", "CLONE.fail", "LUNCPY.fail", "LM.fail", "RM.fail", "LMR.fail", "REPRM.fail", "REPHV.fail", "REPHC.fail", "REPFSMOVE.fail"]
        errorKeyLst = result.split(';')
        errorKeyLst = list(set(errorKeyLst).intersection(set(failureLst)))
        if not errorKeyLst:
            return False, errorKeyLst
        if "REPRM.fail" in errorKeyLst and "RM.fail" in errorKeyLst:
            errorKeyLst.remove("REPRM.fail")
        return True, errorKeyLst
