# -*- coding: UTF-8 -*-
from frameone.util import contextUtil, baseUtil
from frameone.base.constants import CheckStatus, CheckResult
from business.common.config import CPU_PARTITION_USAGE_CONFIG
from frameone.adapter.resourceService import ResrouceFactory
import traceback

PATITION_ID_CHECK_DICT = {
    "7"  : 50, # normal分区
    "9"  : 60, # rv分区
    "10" : 60  # kv分区
}

# 执行入口
def execute(dataDict):

    # 检查设备版本，只有满足特定的设备型号和目标版本才进行该检查
    logger = contextUtil.getLogger(dataDict)
    currentVersion = contextUtil.getCurVersion(dataDict)

    logger.info("currentVersion=%s" % currentVersion)
    if currentVersion < CPU_PARTITION_USAGE_CONFIG["srcVersionAtLeast"]:
        logger.info("The device not need check controller memory.")
        return CheckStatus.NO_SUPPORT, "", ""

    ##执行升级前检查
    checkRet = CpuPartitionUsageCheck(dataDict).run()
    return checkRet.toScriptResult()

class CpuPartitionUsageCheck():
    def __init__(self, dataDict):
        self.dataDict = dataDict
        self.originInfos = []
        self.logger = contextUtil.getLogger(dataDict)
        self.lang = contextUtil.getLang(dataDict)
        self.commonMsg = baseUtil.getMsg(self.lang, "query.result.abnormal")

    def run(self):
        self.logger.info("enter CpuPartitionUsageCheck run.")
        checkResultMap = {}
        try:
            self.resrouceFactory = ResrouceFactory(self.dataDict)
            # 分别查询每个分区所有控制器的CPU利用率
            # 每个分区的结果为：{"0A":"20","0B":"23","0C":"error", "0D":"nocmd"}
            for patitionId in PATITION_ID_CHECK_DICT:
                cmd = "iod cpuusage show %s" % patitionId
                recs = self.resrouceFactory.executeInterface("excuteDiagnoseCmd", cmd, 2)
                checkResultMap[patitionId] = self.parseRecs(recs)
            self.originInfos.append("pationId cpu usage=%s" % checkResultMap)

            # 按控制器ID进行数据重组
            mergeResult = {}
            for patitionId in checkResultMap.keys():
                patitionObj = checkResultMap.get(patitionId, {})
                for nodeId in patitionObj.keys():
                    resultList = mergeResult.get(nodeId, [])
                    average = patitionObj.get(nodeId, "")
                    if average == "nocmd": # 对没有该命令时进行兼容
                        resultList.append("pass")
                    elif average.isdigit() and int(average) < PATITION_ID_CHECK_DICT.get(patitionId):
                        resultList.append("pass")
                    elif average.isdigit() and int(average) >= PATITION_ID_CHECK_DICT.get(patitionId):
                        resultList.append("nopass")
                    else:
                        resultList.append("error")
                    mergeResult[nodeId] = resultList
            self.logger.info("mergeResult=%s" % mergeResult)

            #将数据结果进行状态合并{"OA":["pass","nopas","pass"],"OB":["error","error","pass"]}
            for nodeId in mergeResult.keys():
                if "nopass" in mergeResult[nodeId]:
                    mergeResult[nodeId] = CheckStatus.NOTPASS
                elif "error" in mergeResult[nodeId]:
                    mergeResult[nodeId] = CheckStatus.NOCHECK
                else:
                    mergeResult[nodeId] = CheckStatus.PASS
            self.logger.info("mergeResult=%s" % mergeResult)
            status = CheckStatus.getMergeStatus(mergeResult.values())
            errMsg = ""
            for nodeId, result in mergeResult.items():
                errMsg += baseUtil.getMsg(self.lang, "CpuPartitionUsageCheck." + result, nodeId)
            return CheckResult(status, self.originInfos, errMsg)
        except:
            self.logger.error("CpuPartitionUsageCheck exception.[%s]" % traceback.format_exc())
            return CheckResult(CheckStatus.NOCHECK, self.originInfos, self.commonMsg)

    # 输出格式为{"0A":"pass","0B":"nopass","0C":"error"}
    def parseRecs(self, recs):
        retMap = {}
        for rec in recs:
            nodeId = self.resrouceFactory.executeInterface("getParamStrValue", rec, 0)
            result = self.resrouceFactory.executeInterface("getParamStrValue", rec, 1)
            self.logger.info("result:" + result)
            resultList = result.split(":")
            if "Try Find Help :" in result:
                retMap[nodeId] = "nocmd"
            elif len(resultList) != 2:
                retMap[nodeId] = "error"
            else:
                retMap[nodeId] = str(resultList[1]).strip()
        return retMap
