# -*- coding: utf-8 -*-

"""
@time: 2021/1/28
@file: check_tgt_check_shard_port_switch.py
@function: 共享卡双活ALUA特殊模式0风险检查
"""
from java.lang import Throwable
from business.common import logger
from frameone.base.constants import CheckStatus
from frameone.cli import cliUtil
from frameone.util import baseUtil
from frameone.util import contextUtil
from cbb.frame.base import config

# 风险当前版本组
RISK_CURRENT_VERSION_TUPLE = (
    "V500R007C60SPC200 Kunpeng",
    "V500R007C60SPC300 Kunpeng",
    "V500R007C70 Kunpeng",
    "V500R007C70SPC100 Kunpeng"
)


def execute(context):
    return checkShardPortSwitch(context).executingCheck()


class checkShardPortSwitch:
    def __init__(self, context):
        self._lang = contextUtil.getLang(context)
        self._cli = contextUtil.getSSH(context)
        self._logger = logger.Logger(contextUtil.getLogger(context), __file__)
        self._dev_type = contextUtil.getDevType(context)
        self._current_version = contextUtil.getCurVersion(context)
        self._target_version = contextUtil.getTargetVersion(context)
        self._cli_ret_list = list()
        self._err_msg = ""

    def is_risk_dev_type(self):
        self._cli_ret_list.append("dev type: {}".format(self._dev_type))
        return " V5" in self._dev_type and (self._dev_type.startswith("68") or
                                            self._dev_type.startswith("18"))

    def is_risk_dev_version(self):
        self._cli_ret_list.append("dev version: {}".format(
            self._current_version))
        return self._current_version in RISK_CURRENT_VERSION_TUPLE

    def executingCheck(self):
        """
        执行共享卡双活ALUA特殊模式0风险检查
        :return: 检查结果
        """
        try:
            if self._dev_type in config.OCEAN_STOR_MICRO_DEVS or self._dev_type in config.OCEAN_STOR_COMPUTING_DEVS:
                self._logger.logInfo("compute or micro dev is not support.")
                return CheckStatus.NO_SUPPORT, self.get_cli_ret(), ""
            if not self.is_risk_dev_type() or not self.is_risk_dev_version():
                return CheckStatus.PASS, self.get_cli_ret(), ""
            lun_infos = self._obtain_mode0_using_hyper_metro_lun_infos()
            if not lun_infos:
                return CheckStatus.PASS, self.get_cli_ret(), ""
            return self._create_no_pass_result(self._had_pause_hyper_metro(
                [info.get("running_status") for info in lun_infos]))
        except (Exception, Throwable) as exception:
            self._logger.logException(exception)
            err_msg = baseUtil.getMsg(self._lang, 'query.result.abnormal')
            return CheckStatus.NOCHECK, self.get_cli_ret(), err_msg

    def _had_pause_hyper_metro(self, hyper_metro_lun_running_status_list):
        """
        是否暂停了双活
        :param hyper_metro_lun_running_status_list: 双活lun的running status
        :return: 是 True/ 否 False
        """
        if not hyper_metro_lun_running_status_list:
            return False
        for status in hyper_metro_lun_running_status_list:
            if status.lower() != "paused":
                return False
        return True

    def _create_no_pass_result(self, had_pause_hyper_metro):
        if self._current_version == "V500R007C60SPC200 Kunpeng" and \
                self._target_version == "V500R007C60SPC300 Kunpeng":
            if had_pause_hyper_metro:
                err_msg = baseUtil.getMsg(
                    self._lang,
                    'check.shard.port.switch.c60spc200.pause.hyperMetro')
            else:
                err_msg = baseUtil.getMsg(
                    self._lang,
                    'check.shard.port.switch.c60spc200.nopause.hyperMetro')
            return CheckStatus.WARNING, self.get_cli_ret(), err_msg

        if self._target_version >= "V500R007C70SPC200 Kunpeng" and \
                had_pause_hyper_metro:
            err_msg = baseUtil.getMsg(
                self._lang, 'check.shard.port.switch.other.pause.hyperMetro')
            return CheckStatus.WARNING, self.get_cli_ret(), err_msg
        err_msg = baseUtil.getMsg(
            self._lang, 'check.shard.port.switch.notpass')
        return CheckStatus.NOTPASS, self.get_cli_ret(), err_msg

    def _obtain_mode0_using_hyper_metro_lun_infos(self):
        """
        获取特殊模式零使用的双活lun信息
        :return: 特殊模式零使用的双活lun信息
        """
        hyper_metro_lun_infos = self._query_hyper_metro_lun_infos()
        mode0_using_hyper_metro_lun_infos = list()
        if not hyper_metro_lun_infos:
            return mode0_using_hyper_metro_lun_infos
        mode0_using_lun_ids = self._get_special_mode0_port_using_lun_ids()
        for info in hyper_metro_lun_infos:
            if info.get("lun_id") in mode0_using_lun_ids:
                mode0_using_hyper_metro_lun_infos.append(info)
        self._logger.logInfo("mode0_using_hyper_metro_lun_infos: {}".format(
            mode0_using_hyper_metro_lun_infos))
        return mode0_using_hyper_metro_lun_infos

    def _query_hyper_metro_lun_infos(self):
        """
        查询系统中双活LUN的ID信息
        :return: list/lun ids
        """
        cmd = "show hyper_metro_pair general |filterRow column=Type predict=" \
              "equal_to value=LUN |filterColumn include columnList=ID," \
              "Local\sID,Running\sStatus"
        is_exe_success, cli_ret, self._err_msg = cliUtil.excuteCmdInCliMode(
            self._cli, cmd, True, self._lang)
        self._cli_ret_list.append(cli_ret)

        lun_local_infos = list()
        if is_exe_success is cliUtil.RESULT_NOSUPPORT:
            return lun_local_infos

        # 命令是否可用
        if is_exe_success is not True:
            raise Exception("query hyper metro lun ids failed.")
        cli_ret_lines_list = cliUtil.getHorizontalCliRet(cli_ret)
        # 记录双活LUN的ID信息（Local ID）
        for cli_ret_line in cli_ret_lines_list:
            lun_id = cli_ret_line.get("Local ID", "")
            running_status = cli_ret_line.get("Running Status", "")
            if lun_id:
                lun_local_infos.append({"lun_id": lun_id,
                                        "running_status": running_status})
        self._logger.logInfo("hyper metro lun infos: {}".format(str(
            lun_local_infos)))
        return lun_local_infos

    def _get_special_mode0_port_using_lun_ids(self):
        """
        获取使用特殊模式0的lun ids
        :return: list/lun ids
        """
        total_lun_ids = list()
        mode0_host_ids = self._query_host_ids_of_using_special_mode0()
        for host_id in mode0_host_ids:
            total_lun_ids.extend(self._query_host_mapping_lun_id(host_id))
        return total_lun_ids

    def _query_host_ids_of_using_special_mode0(self):
        """
        查询使用特殊模式0的主机ids
        :return: list/主机ids
        """
        cmd = "show initiator initiator_type=FC isfree=no"
        is_exe_success, cli_ret, self._err_msg = cliUtil.excuteCmdInCliMode(
            self._cli, cmd, True, self._lang)
        self._cli_ret_list.append(cli_ret)
        host_ids = set()
        if is_exe_success is cliUtil.RESULT_NOSUPPORT:
            return host_ids
        # 命令是否可用
        if is_exe_success is not True:
            raise Exception("query initiator failed")
        cli_ret_lines_list = cliUtil.getHorizontalCliRet(cli_ret)
        for cli_ret_line in cli_ret_lines_list:
            if "mode0" == cli_ret_line.get("Special Mode Type", "").lower():
                host_id = cli_ret_line.get("Host ID", "")
                if host_id:
                    host_ids.add(host_id)
        self._logger.logInfo("using special mode zero`s host ids: {}".format(
            str(host_ids)))
        return host_ids

    def _query_host_mapping_lun_id(self, host_id):
        """
        查询映射主机的 LUN IDs
        :param host_id: 主机ID
        :return: LUN IDs
        """
        cmd = "show host lun host_id=%s" % host_id
        is_exe_success, cli_ret, self._err_msg = cliUtil.excuteCmdInCliMode(
            self._cli, cmd, True, self._lang)
        self._cli_ret_list.append(cli_ret)
        lun_ids = set()
        if is_exe_success is cliUtil.RESULT_NOSUPPORT:
            return lun_ids
        # 命令是否可用
        if is_exe_success is not True:
            raise Exception("query host mapping lun id failed")
        # 执行命令后回显的列表
        cli_ret_lines_list = cliUtil.getHorizontalCliRet(cli_ret)
        for cli_ret_line in cli_ret_lines_list:
            lun_id = cli_ret_line.get("LUN ID", "")
            if lun_id:
                lun_ids.add(lun_id)
        self._logger.logInfo("host [{}] mapping lun ids: [{}]".format(
            host_id, str(lun_ids)))
        return lun_ids

    def get_cli_ret(self):
        """
        返回原始信息
        :return: 回显
        """
        if self._cli_ret_list:
            return "\n\n".join(self._cli_ret_list)
        else:
            return "No original cli information"
