# -*- coding: UTF-8 -*-
import re
from business.common import common
from frameone.util import baseUtil
from frameone.util import contextUtil
from frameone.adapter.resourceService import ResrouceFactory
from frameone.base.constants import CheckStatus
from cbb.frame.base import product


TLV_CMD = "upg checkhostpath"


def execute(context):
    """
    检查型号范围 V3,V5 DORADO V3
    检查版本范围 V300R002C20SPC200及之后
    :param context:
    :return:
    """
    return FrontLinkCheck(context).check()


class FrontLinkCheck:
    def __init__(self, context):
        self.lang = contextUtil.getLang(context)
        self.logger = common.getLogger(contextUtil.getLogger(context),
                                       __file__)
        self.context = context
        self.origin_infos = []
        self.err_msg = []
        self.resrouce_factory = None

    def check(self):
        try:
            self.logger.logInfo('START FRONT LINK CHECK')
            current_version = contextUtil.getCurVersion(self.context)
            current_type = contextUtil.getDevType(self.context)
            self.origin_infos.append(
                baseUtil.getMsg(self.lang, "front.end.lnk.device.info",
                                (current_type, current_version)))
            # Kunpeng和AA，及V300R002C20SPC200以下版本不检查直接通过
            if self.check_not_bug_device(current_type, current_version):
                return CheckStatus.PASS, "", ""
            self.resrouce_factory = ResrouceFactory(self.context)
            check_result = self.check_tlv_result()
            return check_result, "\n".join(self.origin_infos), "\n".join(
                self.err_msg)
        except Exception as exception:
            self.logger.logException(exception)
            return CheckStatus.NOCHECK, "\n".join(
                self.origin_infos), "\n".join(self.err_msg)

    def check_not_bug_device(self, current_type, current_version):
        """
        检查是否为问题版本
        :param current_type:
        :param current_version:
        :return:
        """
        # kunpeng不检查
        if 'KUNPENG' in current_type.upper() \
                and current_version >= 'V500R007C60':
            return True
        # AA不检查
        if contextUtil.isDoradoDev(
                self.context) and product.isDigitalVer(current_version):
            return True
        # Dorado,Dorado NAS需要检查所有版本
        if contextUtil.isDoradoDev(
                self.context) or baseUtil.isDoradoNasDev(current_type):
            return False
        # 融合存储该版本之下不检查
        if current_version < 'V300R003C20SPC200':
            return True
        return False

    def check_tlv_result(self):
        """
        检查TLV Rest接口回显
        :param recs:
        :return:
        """
        recs = self.resrouce_factory.executeInterface(
            "excuteDiagnoseCmd",
            TLV_CMD, 2)
        self.origin_infos.append(str(recs))
        check_result = CheckStatus.PASS
        no_check = False
        for rec in recs:
            node_id = self.resrouce_factory.executeInterface(
                "getParamStrValue", rec, 0)
            result = self.resrouce_factory.executeInterface(
                "getParamStrValue", rec, 1)
            link_id = self.resrouce_factory.executeInterface(
                "getParamStrValue", rec, 2)
            # 不支持该接口或返回接口结果为正常
            if self.not_support_tlv(result) or result == "True":
                continue
            # 不冗余
            if result == "False":
                suggession = baseUtil.getMsg(self.lang,
                                             "front.end.lnk.notpass")
                if link_id:
                    suggession = baseUtil.getMsg(
                        self.lang, "front.end.lnk.notpass.detail",
                        link_id)
                check_result = CheckStatus.NOTPASS
            # 控制器检查超时
            elif result == "TimeOut":
                suggession = baseUtil.getMsg(self.lang,
                                             "front.end.lnk.timeout")
                no_check = True
            # 其他异常
            else:
                suggession = baseUtil.getMsg(self.lang,
                                             "front.end.lnk.error")
                no_check = True
            if bool(suggession):
                self.err_msg.append(
                    baseUtil.getMsg(self.lang,
                                    "front.end.lnk.controller.check.error",
                                    (node_id, suggession)))
        if no_check and check_result != CheckStatus.NOTPASS:
            return CheckStatus.NOCHECK
        return check_result

    def not_support_tlv(self, result):
        """
        通过回显检查是否支持tlv接口
        :param result:
        :return:
        """
        if re.search("input command error", result,
                     re.IGNORECASE) is not None:
            self.logger.logInfo("product do not support"
                                " FrontLinkCheck TLV. ")
            return True
        return False
