# -*- coding: UTF-8 -*-

from function import StorageContextUtil
from function import StorageResource

from frameone.cli import cliUtil
from java.lang import Exception as JException

import json


def execute(context):
    '''
    @summary: 入口程序 执行自定义命令并保存回显
    @param context: 上下文对象
    '''

    logger = StorageContextUtil.getLogger(context)
    lang = StorageContextUtil.getLang(context)

    try:
        host_lun_executer = HostLunExecuter(context)
        flag, host_lun_info_str = host_lun_executer.get_host_lun_info()

        dev_sn = StorageContextUtil.getDevSN(context)
        _save_device_info(context, dev_sn+"_hostLunInfo", host_lun_info_str)
        return True, ""
    except (Exception, JException) as e:
        logger.error("[saveDevDataCustom] error: " + str(e))
        return False, StorageResource.getMsg(lang, "query.result.abnormal")


def _save_device_info(context, item, save_info):
    '''
    @summary: 保存相关的设备信息
    @param context: 上下文对象
    '''
    logger = StorageContextUtil.getLogger(context)
    scene_data_instance = StorageContextUtil.getSceneDataExchangeObj(context)
    logger.info("[_saveDeviceInfo] save item: " + item)
    scene_data_instance.saveKV(item, save_info)


class HostLunExecuter():
    """
    查询阵列侧的所有主机、及影响的启动器和LUN信息
    返回的json数据格式如下：
    {
    "faliCommand":"",
    "cliRets": "",
    "host": [{"hostID": "hostid1", "initiator": ["wwn1","wwn2"],
        "lunWWN": [{"ID": "id1", "WWN": "wwn1"}, {"ID": "id2", "WWN": "wwn2"}]
        },
        {"hostID": "hostid1", "initiator": ["wwn1","wwn2"],
        "lunWWN": [{"ID": "id1", "WWN": "wwn1"}, {"ID": "id2", "WWN": "wwn2"}]
        }]
    }
    """
    def __init__(self, context):
        self.context = context
        self.logger = StorageContextUtil.getLogger(context)
        self.lang = StorageContextUtil.getLang(context)
        self.cli = StorageContextUtil.getSSH(context)
        self.result = dict(failCommand="",
                           cliRets="", result="nocheck", host=[])
        self.lun_info_dict = {}

    def append_cli_rets(self, cli_ret):
        all_cli_rets = self.result.get("cliRets")
        all_cli_rets += cli_ret + "\n"
        self.result["cliRets"] = all_cli_rets

    def get_all_lun_wwn_id(self):
        cmd = "show lun general"
        status, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang)
        self.append_cli_rets(cli_ret)
        if status is not True:
            self.logger.error(
                "[get_all_lun_wwn_id] get lun general error msg:%s" % err_msg)
            self.result["failCommand"] = cmd
            return False

        cli_dict_list = cliUtil.getHorizontalNostandardCliRet(cli_ret)
        for lunRet in cli_dict_list:
            lun_id = lunRet.get("ID")
            lun_wwn = lunRet.get("WWN")
            self.lun_info_dict[lun_id] = lun_wwn
        return True

    def _get_initiator_cmd_info(self, cmd, initiator_info_dict):
        status, cli_ret, _ = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang)
        self.append_cli_rets(cli_ret)
        if cliUtil.isSupportFeature(cli_ret):
            return True
        if status is not True:
            self.logger.error(
                "[_get_initiator_cmd_info] exe cmd error:%s" % cmd)
            self.result["failCommand"] = cmd
            return False
        cli_dict_list = cliUtil.getHorizontalCliRet(cli_ret)
        for initiator_ret in cli_dict_list:
            run_status = initiator_ret.get("Running Status")
            if run_status != "Online":
                continue
            host_id = initiator_ret.get("Host ID", "")
            initiator_wwn = initiator_ret.get("WWN", "")
            if not initiator_wwn:
                initiator_wwn = initiator_ret.get("iSCSI IQN", "")
            initiator_list = initiator_info_dict.get(host_id)
            if not initiator_list:
                initiator_list = []
                initiator_info_dict[host_id] = initiator_list
            initiator_list.append(initiator_wwn)
        return True

    def _get_host_by_initiator(self):
        initiator_info_dict = {}
        cmd_list = ["show initiator isfree=no initiator_type=FC",
                    "show initiator isfree=no initiator_type=iSCSI"]
        for cmd in cmd_list:
            flag = self._get_initiator_cmd_info(cmd, initiator_info_dict)
            if not flag:
                return False, initiator_info_dict

        return True, initiator_info_dict

    def get_host_lun_info(self):
        flag = self.get_all_lun_wwn_id()
        if not flag:
            self.result['result'] = "False"
            return False, json.dumps(self.result)
        if len(self.lun_info_dict) == 0:
            self.logger.info("[get_host_lun_info]the lun num is 0.")
            self.result['result'] = "True"
            return True, json.dumps(self.result)

        flag, initiator_info_dict = self._get_host_by_initiator()
        if not flag:
            self.logger.error(
                "[get_host_lun_info]get host by initiator error.")
            self.result['result'] = "False"
            return False, json.dumps(self.result)
        if len(initiator_info_dict) == 0:
            self.logger.info("[get_host_lun_info]the initiator num is 0.")
            self.result['result'] = "True"
            return True, json.dumps(self.result)

        # 查询单个主机下的LUN信息
        host_ret_list = self.result.get("host")
        cmd = "show host lun host_id=%s"
        for host_id, initiator_list in initiator_info_dict.items():
            host_dict = dict(hostID=host_id)
            host_dict["initiator"] = initiator_list
            status, cli_ret, _ = cliUtil.excuteCmdInCliMode(
                self.cli, cmd % host_id, True, self.lang)
            self.append_cli_rets(cli_ret)
            if status is not True:
                self.logger.error("[get_host_lun_info] exe cmd error:%s" % cmd)
                self.result["failCommand"] = cmd % host_id
                return False, json.dumps(self.result)
            lun_wwn_dict_list = []
            host_dict["lunWWN"] = lun_wwn_dict_list
            cli_dict_list = cliUtil.getHorizontalCliRet(cli_ret)
            for hostLun in cli_dict_list:
                lun_id = hostLun.get("LUN ID", "")
                lun_wwn = self.lun_info_dict.get(lun_id)
                host_lun_id = hostLun.get("Host LUN ID", "")
                lun_wwn_dict = dict(ID=lun_id, WWN=lun_wwn,
                                    HOST_LUN_ID=host_lun_id)
                lun_wwn_dict_list.append(lun_wwn_dict)
            host_ret_list.append(host_dict)

        return True, json.dumps(self.result)
