# -*- coding: UTF-8 -*-
import json
from function import StorageContextUtil
from frameone.cli import cliUtil
from frameone.util import contextUtil

FAIL_TO_GET_CLI_MODE = "lun.capacity.consistency.check." \
                       "cannot.get.storage.climode"
STORAGE_EXECUTE_ERROR = "lun.capacity.consistency.check." \
                        "cannot.get.storage.null"


class ShowLunCapacityInPreciseMode:
    """
    LUN主机存储容量一致性检查，阵列数据收集
    """
    def __init__(self, context):
        self.context = context
        self.logger = StorageContextUtil.getLogger(context)
        self.cli = contextUtil.getCLI(context)
        self.lang = StorageContextUtil.getLang(context)
        self.all_cli_ret = []
        self.lun_infos = {}
        self.execute_error_keys = []
        self.fail_cmds = []

    def execute(self):
        try:
            # 获取当前cli显示模式
            cli_mode_flag, climode_ret, climode_error = \
                self.get_cli_capacity_mode()
            if cli_mode_flag is not True:
                self.build_result()
                return False
            cli_config_info_dict_list = cliUtil.getVerticalCliRet(
                climode_ret)
            capacity_mode = ""
            if bool(cli_config_info_dict_list):
                cli_config_info_dict = cli_config_info_dict_list[0]
                capacity_mode = cli_config_info_dict.get("Capacity Mode",
                                                         "")
            # 无法获取CLI显示模式
            if not bool(capacity_mode):
                self.execute_error_keys.append(FAIL_TO_GET_CLI_MODE)
                self.build_result()
                return False
            # 是否需要切换模式
            if capacity_mode.lower() != "precise":
                change_mode_ret = self.change_capacity_mode("precise")
                if change_mode_ret[0] is not True:
                    self.build_result()
                    return False
            # 获取LUN信息
            lun_ret = self.show_lun_capacity()
            if lun_ret[0] is not True:
                self.build_result()
                return False
            # 解析LUN
            error_capacity_lun_ids = self.get_lun_capacity(lun_ret[1])
            # 解析容量显示不全的LUN
            self.get_error_capacity_lun_info(error_capacity_lun_ids)
        except Exception as ex:
            self.logger.error("execute error" + str(ex))
            self.execute_error_keys.append(STORAGE_EXECUTE_ERROR)
        finally:
            # 将cli模式还原
            self.change_capacity_mode(capacity_mode)
        # 装载数据
        self.build_result()
        return True

    def build_result(self):
        dev_sn = StorageContextUtil.getDevSN(self.context)
        scene_data_instance = StorageContextUtil.getSceneDataExchangeObj(
            self.context)
        luns_in_precise_mode = {"cli_rets": "\n".join(self.all_cli_ret),
                                "execute_error": self.execute_error_keys,
                                "fail_cmds": ",".join(self.fail_cmds),
                                "lun_wwn_capacitys": self.lun_infos}
        item = dev_sn + "_luns_in_precise_mode"
        scene_data_instance.saveKV(item, json.dumps(luns_in_precise_mode))

    def get_cli_capacity_mode(self):
        """
        @summary: 获取cli显示模式
        @param cli: cli链接
        @return: cmd_ret 命令回文
        """
        cmd = "show cli configuration"
        cmd_ret = cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        self.all_cli_ret.append(cmd_ret[1])
        if cmd_ret[0] is not True:
            self.fail_cmds.append(cmd)
        return cmd_ret

    def change_capacity_mode(self, capacity_mode):
        """
        @summary: 切换cli显示模式
        @return: cmd_ret 命令回显
        :param capacity_mode:
        """
        cmd = "change cli capacity_mode=%s" % capacity_mode
        cmd_ret = cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        self.all_cli_ret.append(cmd_ret[1])
        if cmd_ret[0] is not True:
            self.fail_cmds.append(cmd)
        return cmd_ret

    def show_lun_capacity(self):
        cmd = "show lun general|filterColumn include " \
              "columnList=ID,WWN,Capacity"
        # 至多查询二十分钟
        cmd_result = self.cli.execCmdWithTimout(cmd, 2 * 10 * 60)
        cmd_ret = cliUtil.analysis_cli_result(cmd_result, self.lang)

        # HVS V1R1C00/C10
        if cmd_ret[0] == cliUtil.RESULT_NOCHECK:
            cmd = cmd.replace("columnList", "colunmList")
            cmd_result = self.cli.execCmdWithTimout(cmd, 2 * 10 * 60)
            cmd_ret = cliUtil.analysis_cli_result(cmd_result, self.lang)

        self.all_cli_ret.append(cmd_result)
        if cmd_ret[0] is not True:
            self.fail_cmds.append(cmd)
        return cmd_ret

    def get_lun_capacity(self, cli_ret):
        cli_ret_list = cliUtil.getHorizontalNostandardCliRet(cli_ret)
        error_capacity_lun_ids = []
        for cli_ret_dict in cli_ret_list:
            lun_capacity = cli_ret_dict.get("Capacity")
            lun_id = cli_ret_dict.get("ID")
            lun_wwn = cli_ret_dict.get("WWN")
            # 容量显示不全，记录并统一处理
            if '..' in lun_capacity:
                error_capacity_lun_ids.append(lun_id)
            # 去除单位
            if lun_capacity.upper().endswith("B"):
                lun_capacity = lun_capacity[:-1]
            self.lun_infos[lun_wwn] = lun_capacity
        return error_capacity_lun_ids

    def get_error_capacity_lun_info(self, lun_ids):
        """
        容量显示不全的lun补充详情命令查询
        :param lun_ids:
        :return:
        """
        for lun_id in lun_ids:
            cmd = "show lun general lun_id=%s" % lun_id
            flag, cmd_info, err_info = cliUtil.excuteCmdInCliMode(self.cli,
                                                                  cmd,
                                                                  True,
                                                                  self.lang)
            # 如果详情查询失败，记录回显和失败命令
            if flag is not True:
                self.all_cli_ret.append(cmd_info)
                self.fail_cmds.append(cmd)
                continue
            capacity, wwn = \
                self.get_lun_disk_capacity_and_location(
                    cmd_info)
            self.lun_infos[wwn] = capacity

    def get_lun_disk_capacity_and_location(self, cmd_ret):
        """
        @summary: 根据LUN信息获取Disk Location字段值和容量
        @param cmd_ret: 单个lun的信息
        @return: （capacity, wwn）
        """
        lun_info_dict_list = cliUtil.getVerticalCliRet(cmd_ret)
        for lunInfoDict in lun_info_dict_list:
            if "Disk Location" in lunInfoDict \
                    and "Capacity" in lunInfoDict and "WWN" in lunInfoDict:
                capacity = lunInfoDict.get("Capacity")
                wwn = lunInfoDict.get("WWN")
                # 去除单位
                if capacity.upper().endswith("B"):
                    capacity = capacity[:-1]
                return capacity, wwn
        return "",  ""
