# -*- coding: UTF-8 -*-
import os
import re
import time
import ast

from frameone.base import config
from frameone.rest import restData
from frameone.rest import restDataType
from frameone.util import baseUtil
from frameone.util import contextUtil
from frameone.util import jsonUtil
from com.huawei.ism.tlv.lang import UnsignedInt64  # @UnresolvedImport
from com.huawei.ism.exception import IsmException
from java.lang import Exception as JException

REST_CAN_NOT_EXECUTE = "1077936891" #命令执行失败

class Tlv2Rest:
    '''
     @summary:  将tlv命令封装为rest参数下发
    '''

    @staticmethod
    def convertRecord2Dict(jsonRecord):
        """
        @summary: 将Json格式的回显转换成字典
        @param jsonRecord: REST返回的数据. 
        """
        return jsonUtil.jsonStr2Dict(jsonRecord)

    @staticmethod
    def getParamsJsonStr(paramDict):
        """
        @summary: 将参数字典转化为json字符串
        @param paramDict: 参数字典. 
        """
        if paramDict is None or len(paramDict) == 0:
            return ""
        return jsonUtil.dict2JsonStr(paramDict)

    '''
    Function: 生成TLV参数列表
    Params: allParam=所有的TLV参数，中间以逗号隔开
    Return: TLV参数列表
    '''

    @staticmethod
    def getParamList(*allParam):
        paramList = []
        for param in allParam:
            paramList.append(param)
        return paramList

    @staticmethod
    def getRecordValue(data, field):
        """
        @summary: 获取指定属性Key值转化后成指定类型并返回
        @param data: REST返回的单个DATA数据（类型为字典）. 
            {   propertyIndex0:{ “type”:“int”，“value”：propertyValue }，
                propertyIndex1: { “type”:“int”，“value”：propertyValue }，
                …
                }
        @param field: 包含 index, type。其中key是DATA中的属性。TYPE是该属性的值。 
        """
        try:
            fieldIndex = field.get("index")
            fieldType = field.get("type")
            paramInfo = data.get(str(fieldIndex))
            if paramInfo == None:
                paramInfo = ''
            paramValue = paramInfo.get("value")

            if fieldType == restDataType.DATA_TYPE_UNSIGN_INT or \
                            fieldType == restDataType.DATA_TYPE_SIGN_INT or \
                            fieldType == restDataType.DATA_TYPE_BOOL or \
                            fieldType == restDataType.DATA_TYPE_ENUM or \
                            fieldType == restDataType.DATA_TYPE_TIME:
                return int(paramValue)
            elif fieldType == restDataType.DATA_TYPE_ARRAY:
                return ast.literal_eval(paramValue)
            elif fieldType == restDataType.DATA_TYPE_U64LONG_TYPE:
                return long(paramValue)
            else:
                return unicode(paramValue)
        except Exception, e:
            raise Exception(config.ERR_CODE_CAN_NOT_EXECUTE, "exec cmd return data format error: %s" % str(e))

    @staticmethod
    def getData(record):
        """
        @summary: 获取回显中的data数据
        @param record: rest回显
        @return : 返回data数据
        """
        if record is None or len(record) == 0:
            return None

        return record.get("data")

    @staticmethod
    def getErrInfo(record):
        """
        @summary: 获取回显中的error数据
        @param record: rest回显
        @return : 返回error数据
        """
        if record is None or len(record) == 0:
            return None

        return record.get("error")

    @staticmethod
    def getErrorTetail(record, field):
        fieldkey = field.get("index")
        fieldType = field.get("type")
        paramValue = record.get(fieldkey)

        if fieldType == restDataType.DATA_TYPE_UNSIGN_INT or \
                        fieldType == restDataType.DATA_TYPE_SIGN_INT or \
                        fieldType == restDataType.DATA_TYPE_BOOL or \
                        fieldType == restDataType.DATA_TYPE_ENUM or \
                        fieldType == restDataType.DATA_TYPE_TIME:
            return int(paramValue)

        elif fieldType == restDataType.DATA_TYPE_U64LONG_TYPE:
            return UnsignedInt64(paramValue)
        else:
            return unicode(paramValue)

    @staticmethod
    def tlvInfo2Json(cmd, params):
        '''
        @summary: 将tlv命令参数转换为rest下发的json字符串
        @param cmd:tlv命令字，restData的TlvCmd类
        params:参数列表
        @return: restJsonStr:为下发rest命令使用的json字符串
        '''
        restParams = []
        paramDict = {}

        for param in params:
            if len(param) < 2:
                continue
            pIndex = param[0].get("index")
            paramDict["index"] = pIndex

            # type要和DM约定
            pType = param[0].get("type")
            paramDict["type"] = pType

            pValue = param[1]
            paramDict["value"] = str(pValue)

            restParams.append(paramDict)
            paramDict = {}

        restJsonDict = {"cmd": cmd.get('cmd'),
                        "timeout": cmd.get('timeout'),
                        "params": restParams
                        }
        paramJson = jsonUtil.dict2JsonStr(restJsonDict)
        paramJson = paramJson.replace("\'", "\"").replace("\"[{", "\'[{").replace("}]\"", "}]\'").replace("\"{",
                                                                                                          "{").replace(
            "}\"", "}")
        return paramJson

    @staticmethod
    def getRestUri(baseUri, objUri):
        """
        @summary: 获取REST接口使用的uri连接
        @param objUri: objUri为各对象差异的URI
        """
        return os.path.join(baseUri, objUri)

    @staticmethod
    def execCmd(rest, cmd, params, isBatch=True, isShowLog=True, restMethod=restData.Enum.RestMethod.POST,
                timeOut=config.REST_DEFAULT_TIMEOUT):
        '''
         @summary:  将tlv命令参数转换为rest下发的json字符串
         @param rest:restFactory类
                cmd:tlv命令字，restData的TlvCmd类
                params:参数列表
                isBatch:批量执行标志
        @return: records:返回回显
        '''
        restConn = rest.getRest()
        baseUri = rest.getBaseUri()
        uri = Tlv2Rest.getRestUri(baseUri, restData.Uri.TLV_COMMUNICATION)
        restJsonStr = Tlv2Rest.tlvInfo2Json(cmd, params)
        data = []
        for i in range(config.REST_DEFAULT_RETRY_TIMES + 1):
            try:
                if isShowLog:
                    restRecord = restConn.execPost(uri, restJsonStr).getContent()
                else:
                    restRecord = restConn.execPostWithNoLog(uri, restJsonStr).getContent()

                records = Tlv2Rest.convertRecord2Dict(restRecord)

            except Exception, e:
                raise Exception(config.ERR_CODE_CAN_NOT_EXECUTE, e)

            # 取得返回数据
            data = Tlv2Rest.getData(records)
            if not isinstance(data, list):
                data = []

            # 由于rest返回结果不区分批量和单条数据，都为列表，当为单条数据时取第一条数据返回。
            if not isBatch:
                if len(data) > 0:
                    data = data[0]
                else:
                    data = {}

            # 取得返回异常
            errInfo = Tlv2Rest.getErrInfo(records)

            if errInfo is not None:
                # 取得异常Code
                errCode = Tlv2Rest.getErrorTetail(errInfo, restData.ErrorInfo.CODE)
                # 为默认错误码则为正常情况
                if errCode == config.ERR_CODE_NORMAL:
                    return data

                # 通信异常进行重试
                if i < config.REST_DEFAULT_RETRY_TIMES and errCode in config.ERR_CODE_FAIL_COMMUNICATION:
                    time.sleep(30)  # 间隔30s重新下发
                    continue

                # 取得异常描述
                description = Tlv2Rest.getErrorTetail(errInfo, restData.ErrorInfo.DESCRIPTION)
                # 无法处理的错误码，以异常形式抛出。
                raise Exception(errCode, description)
            else:
                # 设定错误消息和错误码
                raise Exception(config.ERR_CODE_ABEND, "exec cmd retun error! errinfo is None.")

            return data

        return data

    @staticmethod
    def changeBMC(rest):
        '''
         @summary:  修改BMC信息
        '''
        cmd = restData.TlvCmd.CHANGE_BMC_INFO
        params = []
        param1 = (restData.Hardware.BmcInfoDefine.FLAG, restData.Enum.BmcChangeModeEnum.CHANGE)
        params.append(param1)
        return Tlv2Rest.execCmd(rest, cmd, params)

    @staticmethod
    def getFullProduct(rest):
        '''
                         查询阵列全版本
        '''
        cmd = restData.TlvCmd.OM_MSG_OP_LST_VER
        params = []
        param1 = (
        restData.Upgrade.CmoCrVerInfoS.CMO_VER_PACKAGE_TYPE, restData.Enum.UpgradePkgTypeEnum.UPGRADE_PKGTYPE_TYPE_SYS)
        params.append(param1)
        return Tlv2Rest.execCmd(rest, cmd, params, False)

    @staticmethod
    def logRecList(context, recordList, strDesc=None):
        # 构造记录字符串
        logger = baseUtil.getLogger(contextUtil.getLogger(context), __file__)
        strRec = ""
        if None == recordList or len(recordList) == 0:
            strRec = str(None)
        else:
            recNum = len(recordList)
            for index in range(0, recNum):
                strRec = strRec + str(recordList[index]) + "\n"

        # 添加描述
        strHead = "\n"
        if type(strDesc) == str:
            strHead = strHead + strDesc + "\n"
        # 打印记录(尽量将所有的函数调用关系均打印出来)
        logger.logInfo(strHead
                       + "====================logRecList BEGIN====================\n"
                       + strRec
                       + "====================logRecList END====================\n"
                       )
        return

    @staticmethod
    def getEnclosureRecords(rest):
        '''
        @summary: 批量查询框信息
        @param rest: rest连接
        @return: rest回显
        '''
        cmd = restData.TlvCmd.GET_BATCH_NEXT
        param1 = (restData.PublicAttributes.TYPE, restData.Enum.ObjEnum.ENCLOSURE)
        params = Tlv2Rest.getParamList(param1)
        return Tlv2Rest.execCmd(rest, cmd, params)

    @staticmethod
    def getCtrlEnclosureHeight(rest, logger):
        '''
        @summary: 获取当前集群框高
        @param rest: rest对象
        '''
        records = Tlv2Rest.getEnclosureRecords(rest)
        for record in records:
            logicType = Graph2Rest.getRecordValue(record, restData.Hardware.Enclosure.LOGIC_TYPE)
            logger.logInfo("====get getCtrlEnclosureHeight====logicType=[%s]." % str(logicType))
            if logicType == restData.Enum.EncLogicTypeEnum.CTRL:
                height = Graph2Rest.getRecordValue(record, restData.Hardware.Enclosure.HEIGHT)
                logger.logInfo(
                    "====get getCtrlEnclosureHeight====logicType=[%s]====height=[%s]." % (str(logicType), str(height)))
                return height

    @staticmethod
    def getUserInfo(rest, User):
        '''
        @summary: 获取当前集群用户信息
        @param rest: rest对象
        @param User: 用户名
        '''
        cmd = restData.TlvCmd.GET_BATCH_NEXT
        params = []
        strConditions = Tlv2Rest.getQueryConditionString(
            [{"index": 2, "type": restData.Enum.QueryType.EQUAL, "value": User}, ], None, None)
        param0 = (restData.PublicAttributesExtend.QUERY_CONDITION, strConditions)
        param1 = (restData.PublicAttributes.TYPE, restData.Enum.ObjEnum.USER)
        params.append(param0)
        params.append(param1)
        return Tlv2Rest.execCmd(rest, cmd, params, False)

    @staticmethod
    def getSysRecord(rest):
        '''
        @summary: 查询系统信息
        @param rest: rest连接
        @param baseUri: rest基础链接
        @return: 系统回显
        '''
        cmd = restData.TlvCmd.GET
        params = []
        param1 = (restData.Sys.System.TYPE, restData.Enum.ObjEnum.SYSTEM)
        params.append(param1)
        return Tlv2Rest.execCmd(rest, cmd, params, False)

    @staticmethod
    def getHardwareRecords(rest, hardwareType):
        '''
        @summary: 根据硬件形态获取该硬件的所有记录
        @param rest: rest对象
        @param hardwareType: 硬件形态
        '''
        cmd = restData.TlvCmd.GET_BATCH_NEXT
        params = []
        param = (restData.PublicAttributes.TYPE, getattr(restData.Enum.ObjEnum, hardwareType))
        params.append(param)
        return Tlv2Rest.execCmd(rest, cmd, params)

    @staticmethod
    def isSingleModel(rest):
        '''
        @summary: 判读系统是否为单控模式
        @param productModel: rest连接
        @return: 
            True: 系统未单控模式
            False: 系统不为单控模式
        '''
        record = Tlv2Rest.getSysRecord(rest)
        configModel = Tlv2Rest.getRecordValue(record, restData.Sys.System.CONFIG_MODEL)
        if configModel == restData.Enum.CofigModelEnum.SINGLE_CTRL:
            return True

        return False

    '''
    Function：生成多条件值列表，供EQOR、NEAND和LIKEOR使用
    Params:
        allValue：筛选的所有条件值。如4,1,50 或"x","dfsd","234"
    Return: 条件值构成的字符串列表
    '''

    @staticmethod
    def getMultiConditionValueList(*allValue):
        valueList = []
        for value in allValue:
            # 此处将所有条件值均转为字符串，方便后续的比较
            valueList.append(str(value))
        return valueList

    '''
    Function：生成单个条件字典
    Params:
        argIndex：记录值项的索引。如parentID=4
        filterType：筛选类型。参见ConditionType字典的定义
        argValue：筛选的条件值
        paramIdx：兼容处理，匹配告警参数索引
    Return: 条件字典，格式为{"index":ArgIndex, "type":ConditionType, "value"：ArgValue}
    Example: 要求parentID=4的项为28，则入口参数为（4， "EQ", 28）
    '''

    @staticmethod
    def getCondition(argIndex, filterType, argValue, paramIdx=-1):
        # 参数校验，不正确时直接抛出异常
        if type(argIndex) != dict:
            raise Exception("argIndex is not a integer: " + str(type(argIndex)))
        filterType = str(filterType)
        if filterType not in vars(restData.Enum.ConditionTypeEnum).values():
            raise Exception("unsupported record filter type: " + str(filterType))

        # 返回结果
        return {"index": argIndex, "type": filterType, "value": argValue, "paramIdx": paramIdx}

    '''
    Function：生成多条件列表
    Params: allCondition：所有的条件，0或多个，由getCondition()获得
    Return: 条件列表
    '''

    @staticmethod
    def getConditionList(*allCondition):
        conditionList = []
        for condition in allCondition:
            conditionList.append(condition)
        return conditionList

    '''
    Function：生成批量查询命令中需要的条件字符串
    Params:
        queryList为查询条件列表，不存在则为None，当前仅支持一个查询列表，每个查询条件为一个三元字典，格式为：{"index":ArgIndex, "type":QueryType, "value"：ArgValue}
        sortinfo为二元字典，不存在则为None，格式为{"index":ArgIndex, "type":SortType}
        rangeinfo为二元字典，不存在则为None，格式为{"start":StartLineIndex, "end":EndLineIndex(not include)}
    Reuslt: queryCondition字段所需的字符串
    Attention: 
        1、受创建方法限制，传入的参数值中不能带有单引号；
        2、由于仅有少部分命令支持查询条件，故更多的查询结果筛选需使用Record子类中的filter方法
    '''

    @staticmethod
    def getQueryConditionString(queryDictList, sortDict, rangeDict):
        retCondition = {}
        # 构造查询字符串
        if None != queryDictList:
            queryDict = {}
            for cond in queryDictList:
                queryType = cond["type"]
                if restData.Enum.QueryType.LIKE == queryType:
                    queryDict[str(cond["index"])] = "LIKE" + str(cond["value"])
                elif restData.Enum.QueryType.EQUAL == queryType:
                    queryDict[str(cond["index"])] = str(cond["value"])
                elif restData.Enum.QueryType.NOT == queryType:
                    queryDict[str(cond["index"])] = "NOT" + str(cond["value"])
                elif restData.Enum.QueryType.RANGE == queryType:
                    rangeValue = cond["value"]
                    queryDict[str(cond["index"])] = "[" + str(rangeValue[0]) + "," + str(rangeValue[1]) + ")"
                else:
                    return "Invalid query condition: " + str(cond)
            retCondition["query"] = [queryDict]
        # 构造排序字符串
        if None != sortDict:
            strSort = str(sortDict["index"]) + str(sortDict["type"])
            retCondition["sortby"] = strSort
        # 构造区间字符串
        if None != rangeDict:
            strRange = str(rangeDict["start"]) + "-" + str(rangeDict["end"])
            retCondition["range"] = strRange
        # 采用Python的str将条件字典转换为字符串，并将其中的单引号替换为双引号
        return str(retCondition).replace("\'", "\"")

    '''
    Function：筛选出满足指定条件的记录
    Params:
        recordList为待筛选的记录列表
        conditionList为筛选条件列表，不存在则为None，当前仅支持一个条件列表，
                                        每个条件为一个三元字典，格式为：{"index":ArgIndex, "type":ConditionType, "value"：ArgValue}
    Return: 筛选结果 （python List） or 抛出异常（参数或运行错误时）
    '''

    @staticmethod
    def filter(recordList, conditionList):
        # 入口参数校验
        if None == recordList or len(recordList) == 0:
            return []  # 返回空List
        if None == conditionList:
            return recordList
        # 依次按recordList中的条件对传入的记录集进行筛选
        for cond in conditionList:
            # 初始化新的记录列表用于存放本轮筛选结果
            retRecList = []
            # 获取条件信息
            index = cond["index"]
            sType = str(cond["type"])
            value = cond["value"]
            if isinstance(value, list) or isinstance(value, tuple):
                # 如果匹配值为字典或元组，则将其元素转换为字符串，方便下面匹配
                tmpList = []
                for item in value:
                    tmpList.append(str(item))
                value = tmpList
            # 遍历所有的记录进行筛选
            recNum = len(recordList)
            for i in range(0, recNum):
                # 获取当前记录及相应的条件值（默认为字符串类型）
                rec = recordList[i]

                # 取得value值
                recValue = Tlv2Rest.getRecordValue(rec, index)

                # 按不同的条件类型进行筛选
                if restData.Enum.ConditionTypeEnum.LIKE == sType:
                    value = str(value)
                    if None != re.search(value, str(recValue)):  # , re.IGNORECASE): #此处需区分大小写
                        retRecList.append(rec)
                elif restData.Enum.ConditionTypeEnum.NLIKE == sType:
                    value = str(value)
                    if None == re.search(value, str(recValue)):  # , re.IGNORECASE): #此处需区分大小写
                        retRecList.append(rec)
                elif restData.Enum.ConditionTypeEnum.EQ == sType:
                    paramIdx = cond.get("paramIdx", -1)
                    # 告警、事件可以指定参数
                    if paramIdx != -1:
                        # 告警、事件指定的参数
                        vals = str(recValue).split(",")
                        if paramIdx < len(vals):
                            recValue = vals[paramIdx]
                    if str(recValue).strip() == str(value).strip():
                        retRecList.append(rec)
                elif restData.Enum.ConditionTypeEnum.NE == sType:
                    if str(recValue) != str(value):
                        retRecList.append(rec)
                elif restData.Enum.ConditionTypeEnum.GT == sType:
                    # 范围类型为数值（直接按Long取，避免溢出，范围为左闭右开）
                    if recValue > value:
                        retRecList.append(rec)
                elif restData.Enum.ConditionTypeEnum.GE == sType:
                    if recValue >= value:
                        retRecList.append(rec)
                elif restData.Enum.ConditionTypeEnum.LT == sType:
                    if recValue < value:
                        retRecList.append(rec)
                elif restData.Enum.ConditionTypeEnum.LE == sType:
                    if recValue <= value:
                        retRecList.append(rec)
                elif restData.Enum.ConditionTypeEnum.EQOR == sType:
                    if str(recValue) in value:  # 此时value为字符串列表
                        retRecList.append(rec)
                elif restData.Enum.ConditionTypeEnum.NEAND == sType:
                    if str(recValue) not in value:  # 此时value为字符串列表
                        retRecList.append(rec)
                elif restData.Enum.ConditionTypeEnum.LIKEOR == sType:
                    for item in value:  # 此时value为字符串列表
                        if None != re.search(item, str(recValue)):  # , re.IGNORECASE): #此处需区分大小写
                            retRecList.append(rec)
                            break
                else:
                    raise Exception("Invalid query condition type=" + sType + " in condition=" + str(cond))
            # 置待筛选集合为当前筛选结果，进入下轮筛选
            recordList = retRecList
        # 返回筛选结果
        return retRecList

    '''
    Function：依次按指定的条件对所有记录进行排序，满足条件的放在前部，不满足条件的放在尾部
    Params:
        recordList为待排序的记录列表
        conditionList为排序条件列表，不存在则为None，当前仅支持一个条件列表，
                                        每个条件为一个三元字典，格式为：{"index":ArgIndex, "type":ConditionType, "value"：ArgValue}
    Return: 排序结果 （python List）
    Method: 程序依次按条件的排列顺序对输入记录进行排序，一条记录仅参与其满足的第一个条件的排序，满足某个条件后，将不再参与后面条件的排序。
                                    满足第一个条件的记录顺序排在结果集首部，满足第二个条件的记录次之，并依次类推；对于不满足任何条件的记录，顺序排在结果集的尾部。
    '''

    @staticmethod
    def topSort(recordList, conditionList):  # context,
        # 入口参数校验
        if None == recordList or len(recordList) == 0:
            return []
        if None == conditionList:
            return recordList
        # 依次按recordList中的条件对传入的记录集进行排序
        # 初始化的时的记录列表
        retRecList = []
        for cond in conditionList:
            # 初始化临时记录列表，保存不满足本次排序条件的记录
            tmpRecList = []
            # 获取条件信息
            index = cond["index"]
            sType = str(cond["type"])
            value = cond["value"]

            if isinstance(value, list) or isinstance(value, tuple):
                # 如果匹配值为字典或元组，则将其元素转换为字符串，方便下面匹配
                tmpList = []
                for item in value:
                    tmpList.append(str(item))
                value = tmpList

            # 遍历所有的记录进行筛选
            recNum = len(recordList)
            for i in range(0, recNum):
                # 获取当前记录及相应的条件值（默认为字符串类型）
                rec = recordList[i]
                isSorted = False

                # 取得Value值
                recValue = Tlv2Rest.getRecordValue(rec, index)

                # 按不同的条件类型进行筛选
                if restData.Enum.ConditionTypeEnum.LIKE == sType:
                    value = str(value)
                    if None != re.search(value, str(recValue)):  # , re.IGNORECASE): #此处需区分大小写
                        isSorted = True
                elif restData.Enum.ConditionTypeEnum.NLIKE == sType:
                    value = str(value)
                    if None == re.search(value, str(recValue)):  # , re.IGNORECASE): #此处需区分大小写
                        isSorted = True
                elif restData.Enum.ConditionTypeEnum.EQ == sType:
                    if str(recValue) == str(value):
                        isSorted = True
                elif restData.Enum.ConditionTypeEnum.NE == sType:
                    if str(recValue) != str(value):
                        isSorted = True
                elif restData.Enum.ConditionTypeEnum.GT == sType:
                    # 范围类型为数值（直接按Long取，避免溢出，范围为左闭右开）
                    if recValue > value:
                        isSorted = True
                elif restData.Enum.ConditionTypeEnum.GE == sType:
                    if recValue >= value:
                        isSorted = True
                elif restData.Enum.ConditionTypeEnum.LT == sType:
                    if recValue < value:
                        isSorted = True
                elif restData.Enum.ConditionTypeEnum.LE == sType:
                    if recValue <= value:
                        isSorted = True
                elif restData.Enum.ConditionTypeEnum.EQOR == sType:
                    if str(recValue) in value:  # 此时value为字符串列表
                        isSorted = True
                elif restData.Enum.ConditionTypeEnum.NEAND == sType:
                    if str(recValue) not in value:  # 此时value为字符串列表
                        isSorted = True
                elif restData.Enum.ConditionTypeEnum.LIKEOR == sType:
                    for item in value:  # 此时value为字符串列表
                        if None != re.search(item, str(recValue)):  # , re.IGNORECASE): #此处需区分大小写
                            isSorted = True
                            break
                else:
                    raise Exception("Invalid query condition type=" + sType + " in condition=" + str(cond))
                # 处理排序结果
                if True == isSorted:
                    # 当前记录满足排序条件时，则将其存入返回记录集
                    retRecList.append(rec)
                else:
                    # 若当前记录不满足排序条件，则存入临时记录列表中，留待匹配后续的条件
                    tmpRecList.append(rec)
            # 置待筛选集合为当前筛选结果，进入下轮筛选
            if 0 == len(tmpRecList):
                break
            else:
                recordList = tmpRecList
        # 将不满足排序条件的记录放入retRecList尾部
        recNum = len(tmpRecList)
        for i in range(0, recNum):
            rec = tmpRecList[i]
            retRecList.append(rec)
        # 返回筛选结果
        return retRecList

    '''
    Function: 据数值获取TlvStr中的枚举字符串
    Params:
        lang: 语言类型，包括两种取值：zh,en
        enum: 枚举字典
        keyId: 待获取字符串的key
        defaultReturn: 如果keyId不存在时的返回值
    Return: 
        enum[keyId]=正常返回值
        "ERR: Invalid Input Parametors!"=异常返回值：参数错误
        defaultReturn=异常返回值：keyId在enum中找不到
    '''

    @staticmethod
    def getStrByLang(lang, enum, keyId, defaultReturn='--'):
        # 参数校验：若enum不为字典，或keyId不为数值，则直接返回
        if type(enum) != dict or type(keyId) != int:
            return "ERR: Invalid enum or keyId Parametor!"
        if "zh" != lang and "en" != lang:
            return "ERR: Invalid language Parametor!"
        # 初始化返回值为默认值
        retValue = defaultReturn
        # 若keyId存在，则按key获取字符串并分割为中英文
        langVals = []
        if keyId in enum:
            langVals = str(enum[keyId]).split('||')
            # 返回指定语言描述
            if "en" == lang:  # 英文
                retValue = langVals[0]
            else:  # 中文
                retValue = langVals[1]  # "xyz中文123"#
        # 编码转换并返回
        '''
        140303-Question-Python字符编码原理：
        1、python3以下的版本中，非英文（含中文）字符编码间的转换需要使用unicode作为中间过渡编码类型，
                             即若源编码为utf-8，目标编码为gbk，则需先使用decode函数将utf-8编码转换为unicode编码，再通过encode函数转为gbk编码。
        2、在本文件中，文件头部的注释内容指定了本文件内容的编码格式为utf-8，即下面的枚举定义也为utf-8，为了UI上能正确识别和显示，必须将其转换为unicode编码。 
        '''
        retValue = retValue.decode('utf-8')
        return retValue

    '''
    Function: 据数值获取TlvStr中的枚举字符串
    Params:
        enum: 枚举字典
        keyId: 待获取字符串的key
        lang: 语言类型，包括两种取值：zh,en
        defaultReturn: 如果keyId不存在时的返回值
    Return: 
        enum[keyId]=正常返回值
        "ERR: Invalid Input Parametors!"=异常返回值：参数错误
        defaultReturn=异常返回值：keyId在enum中找不到
    '''

    @staticmethod
    def getStr(context, enum, keyId, defaultReturn='--'):
        return Tlv2Rest.getStrByLang(contextUtil.getLang(context), enum, keyId, defaultReturn)

    @staticmethod
    def getStrEn(enum, keyId, defaultReturn='--'):
        return Tlv2Rest.getStrByLang("en", enum, keyId, defaultReturn)

    @staticmethod
    def getExpansionBayIdRecord(rest, bayId, isIpScaleOut):
        '''
        @summary: 获取当前集群的特定柜ID的记录
        @param tlv: tlv对象
        '''
        params = []
        param = (restData.PublicAttributes.TYPE, restData.Enum.ObjEnum.BAY)
        params.append(param)
        param = (restData.PublicAttributes.ID, bayId)
        params.append(param)
        if isIpScaleOut:
            param = (restData.Hardware.Bay.IS_IPSCALE_OUT, True)
            params.append(param)
        return Tlv2Rest.execCmd(rest, restData.TlvCmd.GET_EXPANSION_BAY_ID, params, False)

    '''
    Function: 判断Record中指定ID的元素是否存在
    Params: record=TLV执行后返回的记录对象; itemId=TLV参数项位置
    Return: 是否存在。True=存在； False=不存在
    '''

    @staticmethod
    def isItemExisted(record, itemId):
        # 尝试获取指定ID的元素，若能获取到，则说明存在；不存在时将抛出异常
        isExisted = False
        try:
            obj = Tlv2Rest.getRecordValue(record, itemId)
            if None != obj:
                isExisted = True
            else:
                isExisted = False
        except Exception, ex:
            isExisted = False

        return isExisted


class CommonRest():
    REST_CAN_NOT_EXECUTE = "1077936891"  # 命令执行失败

    @staticmethod
    def convertRecord2Dict(jsonRecord):
        """
        @summary: 将Json格式的回显转换成字典
        @param jsonRecord: REST返回的数据. 
        """
        return jsonUtil.jsonStr2Dict(jsonRecord)

    @staticmethod
    def getParamsJsonStr(paramDict):
        """
        @summary: 将参数字典转化为json字符串
        @param paramDict: 参数字典. 
        """
        if paramDict is None or len(paramDict) == 0:
            return ""
        return jsonUtil.dict2JsonStr(paramDict)

    @staticmethod
    def getRecordValue(data, field):
        """
        @summary: 获取指定属性Key值转化后成指定类型并返回
        @param data: REST返回的单个DATA数据（类型为字典）. 
        @param field: 包含 key, type。其中key是DATA中的属性。TYPE是该属性的值。 
        """
        fieldKey = field.get("key")
        fieldType = field.get("type")
        paramValue = data.get(fieldKey, "")

        if fieldType == restDataType.DataType.DATA_TYPE_UNSIGN_INT or \
                        fieldType == restDataType.DataType.DATA_TYPE_SIGN_INT or \
                        fieldType == restDataType.DataType.DATA_TYPE_BOOL or \
                        fieldType == restDataType.DataType.DATA_TYPE_ENUM or \
                        fieldType == restDataType.DataType.DATA_TYPE_TIME:
            return int(paramValue)

        elif fieldType == restDataType.DataType.DATA_TYPE_U64LONG_TYPE:
            return UnsignedInt64(paramValue)
        else:
            return unicode(paramValue)

    @staticmethod
    def getData(record):
        """
        @summary: 获取回显中的data数据
        @param record: rest回显
        @return : 返回data数据
        """
        if record is None or len(record) == 0:
            return None

        return record.get("data", None)

    @staticmethod
    def getErrInfo(record):
        """
        @summary: 获取回显中的error数据
        @param record: rest回显
        @return : 返回error数据
        """
        if record is None or len(record) == 0:
            return None

        return record.get("error", None)

    @staticmethod
    def getParamDict(objUri, uriParams=[]):
        """
        @summary: 获取指定属性Key值转化后成指定类型并返回
        @param data: REST返回的单个DATA数据（类型为字典）. 
        @param field: 包含 key, type。其中key是DATA中的属性。TYPE是该属性的值。 
        """
        uriParamDict = {}
        uriParamDict["objUri"] = objUri
        uriParamDict["paramList"] = uriParams
        return uriParamDict

    @staticmethod
    def getUriParamDict(objUri, uriParams=[]):
        """
        @summary: 获取指定属性Key值转化后成指定类型并返回
        @param data: REST返回的单个DATA数据（类型为字典）. 
        @param field: 包含 key, type。其中key是DATA中的属性。TYPE是该属性的值。 
        """
        uriParamDict = {}
        uriParamDict["objUri"] = objUri
        uriParamDict["paramList"] = uriParams
        return uriParamDict

    @staticmethod
    def getRestUri(baseUri, objUri, uriParams=[]):
        """
        @summary: 获取REST接口使用的uri连接
        @param objUri: objUri为各对象差异的URI
        """
        if len(uriParams) > 0:
            objUri = objUri + '/' + '/'.join(uriParams)
        return os.path.join(baseUri, objUri)

    @staticmethod
    def execCmd(rest, uriParamDict, paramDict, restMethod, timeOut=restData.RestCfg.REST_CMD_TIMEOUT):
        '''
        @summary: 执行rest命令
        @param rest: rest对象
        @param cmd: 命令字
        @param paramDict: 参数字典
        @param restMethod: rest接口方法
        @param timeOut: 超时时间
        '''
        records = None
        restConn = rest.getRest()
        baseUri = rest.getBaseUri()
        uri = CommonRest.getRestUri(baseUri, uriParamDict["objUri"], uriParamDict.get("paramList"))
        paramsStr = CommonRest.getParamsJsonStr(paramDict)
        for i in range(config.REST_DEFAULT_RETRY_TIMES + 1):
            try:
                if restMethod == restData.RestCfg.RestMethod.GET:
                    records = CommonRest.convertRecord2Dict(restConn.execGet(uri, paramsStr).getContent())

                elif restMethod == restData.RestCfg.RestMethod.POST:
                    records = CommonRest.convertRecord2Dict(restConn.execPost(uri, paramsStr).getContent())

                elif restMethod == restData.RestCfg.RestMethod.PUT:
                    records = CommonRest.convertRecord2Dict(restConn.execPut(uri, paramsStr).getContent())

                elif restMethod == restData.RestCfg.RestMethod.DELETE:
                    records = CommonRest.convertRecord2Dict(restConn.execDelete(uri, paramsStr).getContent())

                else:
                    # 无此方法返回空
                    return None

            except Exception, e:
                raise Exception(CommonRest.REST_CAN_NOT_EXECUTE, e)

            errInfo = CommonRest.getErrInfo(records)
            if errInfo is not None:
                errCode = CommonRest.getRecordValue(errInfo, restData.RestCfg.ErrorInfo.CODE)

                # 为默认错误码则为正常情况
                if errCode == config.ERR_CODE_NORMAL:
                    return records

                # 往从节点下发消息失败，可以忽略。
                if errCode == config.ERR_CODE_SLAVE_NODE_HANDLE_MSG_IGNORE:
                    return None

                # 通信异常进行重试
                if i < config.REST_DEFAULT_RETRY_TIMES and errCode == config.ERR_CODE_FAIL_COMMUNICATION:
                    time.sleep(30)  # 间隔30s重新下发
                    continue

                # 无法处理的错误码，以异常形式抛出。
                description = CommonRest.getRecordValue(errInfo, restData.RestCfg.ErrorInfo.DESCRIPTION)
                raise Exception(errCode, description)

            return records

        return records

    @staticmethod
    def getSysRecord(rest):
        '''
        @summary: 查询系统信息
        @param rest: rest连接
        @param baseUri: rest基础链接
        @return: 系统回显
        '''
        paramDict = {}
        uriParamDict = CommonRest.getUriParamDict(restData.RestCfg.OBJ.SYSTEM)
        return CommonRest.execCmd(rest, uriParamDict, paramDict, restData.RestCfg.RestMethod.GET)

    @staticmethod
    def getIBCRecord(rest):
        '''
        @summary: 检查ibc密码是否为默认密码
        @param rest: rest连接
        @param baseUri: rest基础链接
        @return: ibc回显
        '''
        params = {}
        uriParam = CommonRest.getUriParamDict(restData.RestCfg.OBJ.IBC)
        return CommonRest.execCmd(rest, uriParam, params, restData.RestCfg.RestMethod.GET)

    @staticmethod
    def getCtrlRecords(rest):
        '''
        @summary: 批量查询控制器
        @param rest: rest连接
        @param baseUri: rest基础链接
        @return: rest回显
        '''
        params = {}
        uriParam = CommonRest.getUriParamDict(restData.RestCfg.OBJ.CONTROLLER)
        return CommonRest.execCmd(rest, uriParam, params, restData.RestCfg.RestMethod.GET)

    @staticmethod
    def getCtrlIds(ctrlRecords):
        '''
        @summary: 获取控制器id与父id的映射字典{id:parentId}
        @param ctrlRecords: 控制器rest回显
        @return: 控制器id字典
        '''
        ctrlIds = []
        datas = CommonRest.getData(ctrlRecords)
        for ctrlData in datas:
            ctrlId = CommonRest.getRecordValue(ctrlData, restData.RestCfg.Controller.ID)
            ctrlIds.append(ctrlId)
        return ctrlIds

    @staticmethod
    def getCtrlIdDict(ctrlRecords):
        '''
        @summary: 获取控制器id与父id的映射字典{id:parentId}
        @param ctrlRecords: 控制器rest回显
        @return: 控制器id字典
        '''
        ctrlIdDict = {}
        datas = CommonRest.getData(ctrlRecords)
        for ctrlData in datas:
            ctrlId = CommonRest.getRecordValue(ctrlData, restData.RestCfg.Controller.ID)
            parentId = CommonRest.getRecordValue(ctrlData, restData.RestCfg.Controller.PARENT_ID)
            ctrlIdDict[ctrlId] = parentId
        return ctrlIdDict

    @staticmethod
    def getCtrlRecord(rest, ctrlId):
        '''
        @summary: 单个查询控制器
        @param rest: rest连接
        @param ctrlId: 控制器ID
        @return: rest回显
        '''
        params = {}
        uriParamDict = CommonRest.getUriParamDict(restData.RestCfg.OBJ.CONTROLLER, [ctrlId])
        return CommonRest.execCmd(rest, uriParamDict, params, restData.RestCfg.RestMethod.GET)

    @staticmethod
    def getCtrlIoRecord(rest, ctrlId):
        '''
        @summary: 查询控制器Io
        @param rest: rest连接
        @param baseUri: rest基础链接
        @return: rest回显
        '''
        params = {restData.RestCfg.ControllerIo.ID["key"]: ctrlId}
        uriParam = CommonRest.getUriParamDict(restData.RestCfg.OBJ.CONTROLLER_IO)
        return CommonRest.execCmd(rest, uriParam, params, restData.RestCfg.RestMethod.GET)

    @staticmethod
    def getEthPortRecords(rest):
        '''
        @summary: 批量查询eth端口
        @param rest: rest连接
        @return: rest回显
        '''
        paramDict = {}
        uriParamDict = CommonRest.getUriParamDict(restData.RestCfg.OBJ.ETH_PORT)
        return CommonRest.execCmd(rest, uriParamDict, paramDict, restData.RestCfg.RestMethod.GET)

    @staticmethod
    def getCtrlEnclosureIds(rest):
        '''
        @summary: 获取控制框的id列表
        @param rest: rest连接
        @return: rest回显
        '''
        encIds = []
        paramDict = {}
        uriParamDict = CommonRest.getUriParamDict(restData.RestCfg.OBJ.ENCLOSURE)
        record = CommonRest.execCmd(rest, uriParamDict, paramDict, restData.RestCfg.RestMethod.GET)
        datas = CommonRest.getData(record)
        for data in datas:
            encId = CommonRest.getRecordValue(data, restData.RestCfg.Enclosure.ID)
            logicType = CommonRest.getRecordValue(data, restData.RestCfg.Enclosure.LOGIC_TYPE)
            if logicType == restData.RestCfg.EncLogicTypeEnum.CTRL:
                encIds.append(encId)

        return encIds

    @staticmethod
    def getBackupPowerRecords(rest):
        '''
        @summary: 获取备电模块信息
        @param rest: rest连接
        @return: rest回显
        '''
        paramDict = {}
        uriParamDict = CommonRest.getUriParamDict(restData.RestCfg.OBJ.BACKUP_POWER)
        record = CommonRest.execCmd(rest, uriParamDict, paramDict, restData.RestCfg.RestMethod.GET)
        return record

    @staticmethod
    def backUpConfigData(rest):
        """
        @summary:下载后删除阵列上的临时DB信息
        @param rest: rest对象
        """

        uriParams = CommonRest.getUriParamDict(restData.RestCfg.OBJ.EXPORT_DB_CONFIG)
        params = {restData.RestCfg.ExportData.CMO_EXPORT_DB_TYPE["key"]: restData.RestCfg.ExportTypeEnum.MEMORY}
        return CommonRest.execCmd(rest, uriParams, params, restData.RestCfg.RestMethod.PUT)

    @staticmethod
    def clearDb(rest, ctrlId):
        """
        @summary:下载后删除阵列上的临时DB信息
        @param rest: rest对象
        """

        uriParam = CommonRest.getUriParamDict(restData.RestCfg.OBJ.CLEAR_EXPORT_DATA)
        paramDict = {restData.RestCfg.ExportData.EXPORT_DATA_TYPE["key"]: restData.RestCfg.DataTypeEnum.CONFIG_DATA
            , restData.RestCfg.ExportData.NODE_ID["key"]: ctrlId}
        return CommonRest.execCmd(rest, uriParam, paramDict, restData.RestCfg.RestMethod.PUT)

    @staticmethod
    def offLineCtrl(rest, ctrlId):
        """
        @summary:离线控制器（多控模式）
        @param rest: rest对象
        @param ctrlId: 控制器id
        """
        paramDict = {restData.RestCfg.Controller.ID["key"]: ctrlId}
        uriParamDict = CommonRest.getUriParamDict(restData.RestCfg.SpecialUri.OFFLINE_CONTROLLER)
        return CommonRest.execCmd(rest, uriParamDict, paramDict, restData.RestCfg.RestMethod.PUT)

    @staticmethod
    def offLineCtrlForSingle(rest, ctrlId):
        """
        @summary:离线控制器（单控模式）
        @param rest: rest对象
        @param cntrId: 控制器id
        """
        paramDict = {restData.RestCfg.Controller.ID["key"]: ctrlId}
        uriParamDict = CommonRest.getUriParamDict(restData.RestCfg.SpecialUri.OFFLINE_CONTROLLER_SIGNLE)
        return CommonRest.execCmd(rest, uriParamDict, paramDict, restData.RestCfg.RestMethod.PUT)

    @staticmethod
    def lightCtrl(rest, ctrlId):
        """
        @summary:点灯控制器
        @param rest: rest对象
        @param cntrId: 控制器id
        """
        paramDict = {restData.RestCfg.Controller.ID["key"]: ctrlId,
                     restData.RestCfg.Controller.LIGHT_STATUS["key"]: restData.RestCfg.LightStatusEnum.ON,
                     }
        uriParamDict = CommonRest.getUriParamDict(restData.RestCfg.OBJ.CONTROLLER)
        try:
            CommonRest.execCmd(rest, uriParamDict, paramDict, restData.RestCfg.RestMethod.PUT)
            return (True, "")
        except Exception, ex:
            return (False, ex)

    @staticmethod
    def getMgmtPortRecords(rest):
        """
        @summary:查询管理eth端口
        @param rest: rest对象
        """
        paramDict = {restData.RestCfg.EthPort.LOGIC_TYPE["key"]: restData.RestCfg.EthPortLogicTypeEnum.MNGT}
        uriParamDict = CommonRest.getUriParamDict(restData.RestCfg.OBJ.ETH_PORT)
        return CommonRest.execCmd(rest, uriParamDict, paramDict, restData.RestCfg.RestMethod.GET)

    @staticmethod
    def getAllSasPorts(rest):
        """
        @summary: 获取所有的SAS端口
        @param rest: rest对象
        """
        paramDict = CommonRest.getUriParamDict(restData.RestCfg.OBJ.SAS_PORT)
        data = CommonRest.execCmd(rest, paramDict, None, restData.RestCfg.RestMethod.GET)
        return CommonRest.getData(data)

    @staticmethod
    def getInterfaceById(rest, intfId):
        """
        @summary: 根据接口卡ID，查询接口卡
        """
        paramDict = CommonRest.getUriParamDict(restData.RestCfg.OBJ.INTERFACE_MODULE, [intfId])
        data = CommonRest.execCmd(rest, paramDict, None, restData.RestCfg.RestMethod.GET)
        return CommonRest.getData(data)

    @staticmethod
    def getCurrentAlarm(rest):
        '''
        @summary: 获取系统中当前告警
        @param rest: rest连接
        '''
        paramDict = CommonRest.getUriParamDict(restData.RestCfg.SpecialUri.QUERY_CURRENT_ALARM_INFO)
        data = CommonRest.execCmd(rest, paramDict, None, restData.RestCfg.RestMethod.GET)
        return CommonRest.getData(data)

    @staticmethod
    def getReplicationPair(rest):
        '''
        @summary: 获取远程复制pair， 出现异常认为不存在远程复制项目License。
        @param rest: rest连接
        '''
        paramDict = CommonRest.getUriParamDict(restData.RestCfg.OBJ.REPLICATIONPAIR)
        data = CommonRest.execCmd(rest, paramDict, None, restData.RestCfg.RestMethod.GET)
        return CommonRest.getData(data)

    @staticmethod
    def getFcPortRecords(rest):
        '''
        @summary: 批量查询eth端口
        @param rest: rest连接
        @return: rest回显
        '''
        paramDict = {}
        uriParamDict = CommonRest.getUriParamDict(restData.RestCfg.OBJ.FC_PORT)
        data = CommonRest.execCmd(rest, uriParamDict, paramDict, restData.RestCfg.RestMethod.GET)
        return CommonRest.getData(data)

    @staticmethod
    def getFcoePortRecords(rest):
        '''
        @summary: 批量查询eth端口
        @param rest: rest连接
        @return: rest回显
        '''
        paramDict = {}
        uriParamDict = CommonRest.getUriParamDict(restData.RestCfg.OBJ.FCoE_PORT)
        data = CommonRest.execCmd(rest, uriParamDict, paramDict, restData.RestCfg.RestMethod.GET)
        return CommonRest.getData(data)

    @staticmethod
    def getEthPortsRecord(rest):
        '''
        @summary: 批量查询eth端口
        @param rest: rest连接
        @return: rest回显
        '''
        paramDict = {}
        uriParamDict = CommonRest.getUriParamDict(restData.RestCfg.OBJ.FCoE_PORT)
        data = CommonRest.execCmd(rest, uriParamDict, paramDict, restData.RestCfg.RestMethod.GET)
        return CommonRest.getData(data)

    @staticmethod
    def getRelicationPairById(rest, portId):
        """
        @summary: 根据类型和ID获取端口
        """
        paramDict = CommonRest.getUriParamDict(restData.RestCfg.OBJ.REPLICATIONPAIR, [portId])
        data = CommonRest.execCmd(rest, paramDict, None, restData.RestCfg.RestMethod.GET)
        return CommonRest.getData(data)

    @staticmethod
    def getEnclosureRecords(rest):
        '''
        @summary: 批量查询框信息
        @param rest: rest连接
        @return: rest回显
        '''
        paramDict = {}
        uriParamDict = CommonRest.getUriParamDict(restData.RestCfg.OBJ.ENCLOSURE)
        record = CommonRest.execCmd(rest, uriParamDict, paramDict, restData.RestCfg.RestMethod.GET)
        return CommonRest.getData(record)

    @staticmethod
    def isSingleModel(rest):
        '''
        @summary: 判读系统是否为单控模式
        @param productModel: rest连接
        @return: 
            True: 系统未单控模式
            False: 系统不为单控模式
        '''
        record = CommonRest.getSysRecord(rest)
        data = CommonRest.getData(record)
        configModel = CommonRest.getRecordValue(data, restData.RestCfg.System.CONFIG_MODEL)
        if configModel == restData.RestCfg.CofigModelEnum.SINGLE_CTRL:
            return True

        return False

    @staticmethod
    def checkIBCPassword(record):
        '''
        @summary: 检查ibc密码是否为默认密码
        @param record: rest回显
        @return: 
            True: ibc密码为默认密码
            False: ibc密码不为默认密码
        '''
        data = CommonRest.getData(record)
        pwdChanged = CommonRest.getRecordValue(data, restData.RestCfg.IBC.PWD_CHANGED)
        if pwdChanged == restData.RestCfg.IbcEnum.PWD_CHANGED:
            return False
        return True

    @staticmethod
    def checkIBCKeyPiar(record):
        '''
        @summary: 检查ibc公私钥对是否为默认
        @param record: rest回显
        @return:  True 公私钥对变更
                  False 公私钥对未变更
        '''
        data = CommonRest.getData(record)
        pwdChanged = CommonRest.getRecordValue(data, restData.RestCfg.IBC.KEYPAIR_CHANGED)
        if pwdChanged == restData.RestCfg.IbcEnum.KEYPAIR_CHANGED:
            return False
        return True

    @staticmethod
    def precheckCBUForCtrl(rest, cntrlId):
        """
        @summary: 更换前检查CBU（只适用于ARM设备）
        @param record: rest回显
        @return: True:检查通过
                False:检查不通过
        """
        cbuCount = 0
        abnormalCBUs = []
        record = CommonRest.getBackupPowerRecords(rest)
        datas = CommonRest.getData(record)
        for data in datas:
            parentId = CommonRest.getRecordValue(data, restData.RestCfg.BackupPower.PARENT_ID)
            if parentId != cntrlId:
                continue

            cbuCount += 1
            location = CommonRest.getRecordValue(data, restData.RestCfg.BackupPower.LOCATION)
            healthStatus = CommonRest.getRecordValue(data, restData.RestCfg.BackupPower.HEALTH_STATUS)
            runningStatus = CommonRest.getRecordValue(data, restData.RestCfg.BackupPower.RUNNING_STATUS)
            if healthStatus != restData.RestCfg.HealthStatusEnum.NORMAL or runningStatus != restData.RestCfg.RunningStatusEnum.ONLINE:
                abnormalCBUs.append(data)

        if len(abnormalCBUs) > 0 or cbuCount == 0:
            return (False, abnormalCBUs)

        return (True, [])

    @staticmethod
    def postcheckCBUForCtrl(rest, cntrlId):
        """
        @summary: 更换后检查CBU（只适用于ARM设备）
        @param record: rest回显
        @return: True:检查通过
                False:检查不通过
        """
        cbuCount = 0
        abnormalCBUs = []
        record = CommonRest.getBackupPowerRecords(rest)
        datas = CommonRest.getData(record)
        for data in datas:
            parentId = CommonRest.getRecordValue(data, restData.RestCfg.BackupPower.PARENT_ID)
            if parentId != cntrlId:
                continue

            cbuCount += 1
            location = CommonRest.getRecordValue(data, restData.RestCfg.BackupPower.LOCATION)
            healthStatus = CommonRest.getRecordValue(data, restData.RestCfg.BackupPower.HEALTH_STATUS)
            runningStatus = CommonRest.getRecordValue(data, restData.RestCfg.BackupPower.RUNNING_STATUS)
            if healthStatus in [restData.RestCfg.HealthStatusEnum.FAULT,
                                restData.RestCfg.HealthStatusEnum.PRE_FAIL] or runningStatus == restData.RestCfg.RunningStatusEnum.OFFLINE:
                abnormalCBUs.append(location)

        if len(abnormalCBUs) > 0 or cbuCount == 0:
            return (False, abnormalCBUs)

        return (True, [])

    @staticmethod
    def checkCtrlRunningStatus(rest, ctrlId):
        '''
        @summary: 检查控制器运行状态
        @param ctrlId:
        '''
        errInfo = ""
        try:
            ctrlIds = CommonRest.getCtrlIds(CommonRest.getCtrlRecords(rest))
            # 处理冗余控制器查询不到的场景
            if ctrlId not in ctrlIds:
                return (False, ctrlIds)

            # 获取冗余控制器信息
            record = CommonRest.getCtrlRecord(rest, ctrlId)
            data = CommonRest.getData(record)
            runningStatus = CommonRest.getRecordValue(data, restData.RestCfg.Controller.RUNNING_STATUS)
            if CommonRest.checkRunningStatus(runningStatus):
                return (True, "")

            errInfo = record
            return (False, errInfo)
        except Exception, e:
            raise e

    @staticmethod
    def checkHealthStatus(healthStatus, normalStatusList=[restData.RestCfg.HealthStatusEnum.NORMAL]):
        '''
        @summary: 检查健康状态
        @param healthStatus: 当前健康状态
        @param normalStatusList: 为正常状态的列表
        @return: 
            True: 检查通过, 
            False: 检查不通过
        '''
        return healthStatus in normalStatusList

    @staticmethod
    def checkRunningStatus(runningStatus, normalStatusList=[restData.RestCfg.RunningStatusEnum.ONLINE]):
        '''
        @summary: 检查运行状态
        @param healthStatus: 当前运行状态
        @param normalStatusList: 为正常状态的列表
        @return: 
            True: 检查通过, 
            False: 检查不通过
        '''
        return runningStatus in normalStatusList

    @staticmethod
    def checkAlarm(rest, alarmRuleItems, filterList=[]):
        """
        @summary: 过滤告警，通过正则匹配符合规则的记录。
        @param rest: rest连接
               alarmRuleItems：配置的需要检查的告警id、检查那个字段，检查该字段中哪些值。
                                                       如64438009898: {"obj":restData.RestCfg.AlCommonRest.EVENTPARAM ,"checkIndex": [1,2]},
               filterList: 用于过滤的值，其中该列表中的每条记录的每个元素都会和checkIndex中对应位置上的值做匹配。
                                                   如 [[DAE000,A],[DAE001,A], [DAE001,A]]
        @return True: 匹配成功。
                False: 未匹配出告警
        """
        # 查询出所有的告警
        alarmRecords = CommonRest.getCurrentAlarm(rest)
        checkRes = []
        for alarm in alarmRecords:
            if CommonRest.checkSingleAlarm(alarm, alarmRuleItems, filterList):
                alarmId = CommonRest.getRecordValue(alarm, restData.RestCfg.Alarm.STREVENTID)
                checkRes.append(alarmId)
        if len(checkRes) > 0:
            return (True, checkRes)
        else:
            return (False, checkRes)

    @staticmethod
    def checkSingleAlarm(alarm, alarmRuleItems, valueFilterList=[]):
        """
        @summary: 单个告警匹配过滤, 如果不需要过滤字段，直接返回匹配上的ID
        @param alarmRuleItems：配置的需要检查的告警id、检查那个字段，检查该字段中哪些值。
                                                       如64438009898: {"obj":restData.RestCfg.AlCommonRest.EVENTPARAM ,"checkIndex": [1,2]},
               valuefilter: 用于过滤的值，其中该列表中的每条记录的每个元素都会和checkIndex中对应位置上的值做匹配。
                                                   如 [DAE001,A]
        @return: True 存在告警，False 不存在告警
        """

        # 需要配置的告警id
        alarmIds = alarmRuleItems.keys()
        alarmId = CommonRest.getRecordValue(alarm, restData.RestCfg.Alarm.EVENTID)

        if alarmId not in alarmIds:
            return False

        rule = alarmRuleItems[alarmId]
        if len(rule) == 0:
            return False

        for sFilter in valueFilterList:
            regPatternList = CommonRest.getRegPatternList(sFilter)

            # 获取出告警的字段值。
            checkValue = CommonRest.getRecordValue(alarm, rule["obj"])
            indexList = rule["checkIndex"]

            if CommonRest.checkRegValue(checkValue, regPatternList, indexList):
                return True
        return False

    @staticmethod
    def checkRegValue(checkValue, regList, indexList):
        """
        @summary: 单个告警匹配过滤，过滤字段
        @param alarm: 待过滤的告警
               alarmRuleItems：配置的需要检查的告警id、检查那个字段，检查该字段中哪些值。
                                                       如64438009898: {"obj":restData.Alarm.EVENTPARAM ,"checkIndex": [1,2]},
               regList: 用于过滤的正则，其中该列表中的每条记录的每个元素都会和checkIndex中对应位置上的值做匹配。
        @return: True 存在告警， False 不存在告警
        """
        checkValue = checkValue.split(",")
        valueList = []
        # 取出对应位置上的值，存入valueList
        for index in indexList:
            if index >= len(checkValue):
                return False
            else:
                valueList.append(checkValue[index])

        for i in range(len(regList)):
            # 只要有一个找不到就退出当前备件进入下一个记录，如从级联板DAE000.A记录的检查跳到DAE001.A
            if regList[i].search(valueList[i]) is None:
                return False

        # 如果True表示该告警匹配目标备件所有规则都成功。
        return True

    @staticmethod
    def getRegPatternList(patternList):
        """
        @summary: 获取正则匹配项
        """
        return [re.compile(pattern) for pattern in patternList]

    @staticmethod
    def checkIpInReplacedCtrl(context, ctrlId):
        """
        @summary: 检查工具连接ip是否在连接待更换控制器（只适用2U设备）
        @param context: 上下文
        @param ctrlId: 待更换控制器id
        @return: 
            False:非待更换控制器ip
            True:为当前更换控制器ip
        """

        isExist = False
        location = ""
        ip = contextUtil.getIp(context)
        rest = contextUtil.getRest(context)
        record = CommonRest.getEthPortRecords(rest)
        datas = CommonRest.getData(record)
        for data in datas:
            location = CommonRest.getRecordValue(data, restData.RestCfg.EthPort.LOCATION)
            ipv4 = CommonRest.getRecordValue(data, restData.RestCfg.EthPort.IPV4_ADDR)
            ipv6 = CommonRest.getRecordValue(data, restData.RestCfg.EthPort.IPV6_ADDR)
            if ip in [ipv4, ipv6]:
                isExist = True
                break

        if isExist:
            ctrlName = ctrlId[-1].upper()
            pattern = ".%s" % ctrlName
            if re.search(location, pattern) is not None:
                return True

        return False

    @staticmethod
    def getIpList(ipInfo, ipType):
        """
        @summary: 根据ip类型获取对应的ip地址列表
        @param ipInfo: ip信息，信息格式：{location:{'ipv4':ipv4,'ipv6':ipv6}}
        @param ipType: ip类型，为枚举值（ipv4或ipv6）
        @return: ip信息列表，
        """
        ipList = []
        ipKey = ""
        if ipType == restData.RestCfg.IpTypeEnum.IPV4:
            ipKey = "ipv4"
        elif ipType == restData.RestCfg.IpTypeEnum.IPV6:
            ipKey = "ipv6"

        for itemKey in ipInfo.keys():
            item = ipInfo[itemKey]
            ipList.append(item[ipKey])

        return ipList

    @staticmethod
    def getUsableIpList(context, ipList):
        """
        @summary: 获取可用ip列表
        @param context: 上下文
        @param ipList: ip全集
        @return: 可用的ip列表 
        """

        usableIpList = []
        for ip in ipList:
            flag, restConn = contextUtil.createRest(context, ip)
            if flag:
                usableIpList.append(ip)

            restConn.close()

        return usableIpList

    @staticmethod
    def getEnclosureListByWwnList(sasPorts, allSasPorts):
        """
        @summary: 根据控制器上的SAS端口列表过滤出当前环路的硬盘框
        @param sasPorts: 当前控制器上的SAS端口
                allSasPorts: 系统中所有的SAS端口
        @return: 当前环路的硬盘框
        """
        encList = []
        wwnList = CommonRest.getSasPortWwnList(sasPorts)
        for sasPortRecord in allSasPorts:
            parentType = CommonRest.getRecordValue(sasPortRecord, restData.RestCfg.SasPort.PARENT_TYPE)
            if parentType != restData.RestCfg.ParentTypeEnum.EXPBOARD:
                continue

            wwn = CommonRest.getRecordValue(sasPortRecord, restData.RestCfg.SasPort.WWN)
            location = CommonRest.getRecordValue(sasPortRecord, restData.RestCfg.SasPort.LOCATION)
            encName = location.split(".")[0]
            # 数据支持，WWN 和 EnclosureWwnLis是后半部分一致
            # 如：'ENCLOSURE_WWN_LIST': '0x5fce33caad2d003f,0x5fce33caacfe803f,',
            # 而WWN 为：'WWN': '5fce33caad2d003f'
            for hexWwn in wwnList:
                if wwn != '' and wwn in hexWwn and encName not in encList:
                    # 记录硬盘框号和级联板前两位，方便待会做对比
                    encList.append(encName)
        return encList

    @staticmethod
    def getExpBoardByPeerPortId(sasPorts, allSasPorts):
        '''
        @summary: 根据控制器上的SAS端口的对端端口ID所在的SAS硬盘框wwnList环路计算出环路的所有级联板
        @param sasPorts:当前控制器连接的端口
        @param allSasPorts:所有SAS端口，包含控制器上的和级联板上的。
        '''

        expList = []
        for sasPort in sasPorts:
            peerPortId = CommonRest.getRecordValue(sasPort, restData.RestCfg.SasPort.CURRENT_PEER_PORT_ID)
            encOnCurrentSasPortList = CommonRest.getEnclosureListByWwnList([sasPort], allSasPorts)
            expId = ''
            for sasPortRecord in allSasPorts:
                parentType = CommonRest.getRecordValue(sasPortRecord, restData.RestCfg.SasPort.PARENT_TYPE)
                portId = CommonRest.getRecordValue(sasPortRecord, restData.RestCfg.SasPort.ID)
                location = CommonRest.getRecordValue(sasPortRecord, restData.RestCfg.SasPort.LOCATION)
                locationList = location.split(".")
                if parentType != restData.RestCfg.ParentTypeEnum.EXPBOARD:
                    continue
                if peerPortId == portId:
                    expId = locationList[1]

            if expId == '':
                continue

            for currentEnc in encOnCurrentSasPortList:
                expBoardLocList = [currentEnc, expId]
                if expBoardLocList not in expList:
                    expList.append(expBoardLocList)

        return expList

    @staticmethod
    def getSasPortWwnList(sasPorts):
        """
        @summary: 获取端口的wwnList
        """
        resList = []
        for sasPort in sasPorts:
            wwnListStr = CommonRest.getRecordValue(sasPort, restData.RestCfg.SasPort.enclosureWWNList)
            if wwnListStr == '':
                continue

            wwnList = wwnListStr.split(",")
            # wwnList以,结尾，需要去除最后的''元素
            if wwnList[-1] == '':
                wwnList = wwnList[:-1]
            resList = resList + wwnList
        return list(set(resList))

    @staticmethod
    def offLineController(rest, ctrlId, logger):
        """
        @summary: 离线控制器
        """
        # 参数合法性检查
        if None == ctrlId or "" == ctrlId:
            logger.logInfo('[offline_controller] the controller id(%s) is invalid.' % ctrlId)
            return False

        # 根据系统模式下发离线命令
        singleFlag = CommonRest.isSingleModel(rest)
        logger.logInfo("[offline_controller] check single model result: %s" % str(singleFlag))
        if not singleFlag:
            # 多控模式下离线控制器
            CommonRest.offLineCtrl(rest, ctrlId)
            totalTime = config.OFFLINE_CTRL_TIMEOUT
        else:
            # 单控模式下离线控制器
            CommonRest.offLineCtrlForSingle(rest, ctrlId)
            return True

        # 检查离线状态,通过条件：运行状态为offline或空，或查不到控制器信息
        beginTime = time.time()
        while time.time() - beginTime <= totalTime:
            try:
                runningStatus = CommonRest.getFruRunningStatus(rest, restData.RestCfg.OBJ.CONTROLLER, ctrlId)
                logger.logInfo("[offline_controller] the controller runningStatus:%s" % runningStatus)
                if runningStatus == "" or CommonRest.checkRunningStatus(runningStatus,
                                                                        [restData.RestCfg.RunningStatusEnum.OFFLINE]):
                    return True

            except Exception, ex:
                logger.logInfo("[offline_controller] get controller error, the exception:%s" % unicode(ex))
                errCode = ex.args[0]
                if errCode == config.ERR_CODE_NOT_EXIST:
                    # 容错：兼容查询不到控制器信息
                    return True

            CommonRest.safeSleep(config.OFFLINE_CTRL_INTERVAL)

        # 超时报错
        logger.logInfo("[offline_controller] check controller timeout")
        return False

    @staticmethod
    def getFruRunningStatus(rest, Obj, FruId):
        '''
        @summary: 获取指定FRU运行状态
        @param Obj: FRU类型
        @param FruId: 指定FRU的ID
        @return: runningStatus：巡行状态
        '''

        uriParams = [FruId]
        uriParamDict = CommonRest.getUriParamDict(Obj, uriParams)
        record = CommonRest.execCmd(rest, uriParamDict, None, restData.RestCfg.RestMethod.GET)
        data = CommonRest.getData(record)
        runningStatus = CommonRest.getRecordValue(data, restData.RestCfg.PublicAttributes.RUNNING_STATUS)

        return runningStatus

    @staticmethod
    def checkSingleEngine(rest):
        """
        @summary: 检查系统单引擎
        @param rest: rest对象
        @return: 
            False:多引擎
            True:单引擎
        """
        ctrlIds = CommonRest.getCtrlEnclosureIds(rest)
        if len(ctrlIds) > 1:
            return False

        return True

    @staticmethod
    def checkCpuUsage(rest, baseUri, logger):
        '''
        @summary: 检查cpu利用率
        @param record: 控制器信息回显
        @return: 
            result: True: 检查通过, False: 检查不通过
            failedList: cpu利用率过高的控制器信息
        '''

        result = True
        failedList = []

        cpuUsageDict = {}
        maxCpuUsageDict = {}
        ctrlRecords = CommonRest.getCtrlRecords(rest)
        logger.logInfo('[check_cpu_usage] controller records:%s' % unicode(ctrlRecords))
        datas = CommonRest.getData(ctrlRecords)
        for data in datas:
            encId = CommonRest.getRecordValue(data, restData.RestCfg.Controller.PARENT_ID)
            cpuUsage = CommonRest.getRecordValue(data, restData.RestCfg.Controller.CPU_USAGE)

            oldCpuUsage = cpuUsageDict.get(encId, 0)
            cpuUsageDict[encId] = oldCpuUsage + cpuUsage

            oldMaxCpuUsage = maxCpuUsageDict.get(encId, 0)
            maxCpuUsageDict[encId] = oldMaxCpuUsage + config.MAX_CPUUSAGE_PER_CTRL

        for enclosureId in cpuUsageDict.keys():
            curCpuUsage = cpuUsageDict[enclosureId]
            curMaxCpuUsage = maxCpuUsageDict[enclosureId]
            if curCpuUsage >= curMaxCpuUsage:
                result = False
                failedList.append({enclosureId: curCpuUsage})

        return (result, failedList)

    @staticmethod
    def checkBackIoCurrent(rest, baseUri, logger):
        '''
        @summary: 检查后端IO压力
        @param rest: rest连接
        @param baseUri: rest基础链接
        @param logger: 日志对象
        @return: 
           result: True: 检查通过, False: 检查不通过
           failedList: 压力过大的框列表
        '''
        result = True
        failedList = []

        ctrlRecords = CommonRest.getCtrlRecords(rest)
        logger.logInfo('[check_backend_Io] controller records:%s' % unicode(ctrlRecords))

        # 统计引擎下的总后端IO
        backEndIoDict = {}
        backEndIoLimitDict = {}
        ctrlIdDict = CommonRest.getCtrlIdDict(ctrlRecords)
        for ctrlId in ctrlIdDict.keys():
            parentId = ctrlIdDict[ctrlId]
            record = CommonRest.getCtrlIoRecord(rest, ctrlId)
            data = CommonRest.getData(record)
            logger.logInfo('[check_backend_Io] controller records:%s' % unicode(record))

            # 获取当前后端并发
            backEndIo = CommonRest.getRecordValue(data, restData.RestCfg.ControllerIo.BACKEND_IO)
            oldBackIo = backEndIoDict.get(parentId, 0)
            backEndIoDict[parentId] = oldBackIo + backEndIo

            # 获取后端并发阈值
            backEndIoLimit = CommonRest.getRecordValue(data, restData.RestCfg.ControllerIo.BACKEND_LIMIT)
            oldBackEndIoLimit = backEndIoLimitDict.get(parentId, 0)
            backEndIoLimitDict[parentId] = oldBackEndIoLimit + backEndIoLimit

            logger.logInfo(
                '[check_backend_Io] controller [%s] backend IO information [backend IO: %s, backend IO limit: %s.' \
                % (ctrlId, unicode(backEndIo), unicode(backEndIoLimit)))

        # 检查后端IO是否在限制标准以内
        for encId in backEndIoDict.keys():
            totalBackEndIo = backEndIoDict[encId]
            totalBackEndIoLimit = backEndIoLimitDict[encId]
            if totalBackEndIo > 0 and totalBackEndIo >= totalBackEndIoLimit:
                result = False
                failedList.append(encId)

        return (result, failedList)

    @staticmethod
    def checkSysPressure(rest, baseUri, logger):
        '''
        @summary: 检查系统业务压力（包括CPU利用率和后端业务压力）
        @param rest: rest连接
        @param baseUri: rest基础链接
        @param logger: 日志对象
        @return: 
            True: 检查通过
            False: 检查不通过
        '''

        succCpuUsageNum = 0
        succBackIoNum = 0
        permitSuccNum = config.CHECK_SYS_PRESSURE_TIMES * config.CHECK_SUCCESS_RATE
        for i in range(0, config.CHECK_SYS_PRESSURE_TIMES):
            logger.logInfo('[sys_pressure] check process [%s/%s]' % (str(i), str(config.CHECK_SYS_PRESSURE_TIMES)))

            # 检查CPU利用率
            checkCpuFlag, checkCpuInfo = CommonRest.checkCpuUsage(rest, baseUri, logger)
            if not checkCpuFlag:
                logger.logInfo('[sys_pressure] check cpu usage failed:[%s]' % str(checkCpuInfo))
            else:
                succCpuUsageNum += 1

            # 检查后端并发
            checkBackEndIoFlag, checkBackEndIoInfo = CommonRest.checkBackIoCurrent(rest, baseUri, logger)
            if not checkBackEndIoFlag:
                logger.logInfo('[sys_pressure] check backend IO failed:[%s]' % (','.join(checkBackEndIoInfo)))
            else:
                succBackIoNum += 1

            if (succCpuUsageNum >= permitSuccNum) and (succBackIoNum >= permitSuccNum):
                return True

            CommonRest.safeSleep(config.CHECK_SYS_PRESSURE_PERIOD)

        logger.logInfo('[sys_pressure] check failed')
        return False

    @staticmethod
    def safeSleep(seconds):
        '''
        @summary: 睡眠
        @param seconds: 睡眠时间
        '''
        try:
            time.sleep(seconds)
        except:
            pass

    @staticmethod
    def checkMgmtPath(context, records, ipType):
        """
        @summary: 检查冗余管理ip
        @param context: 上下文
        @param record: 所有eth端口信息
        @return: 
            True: 存在冗余
            False: 不存在冗余
        """
        defaultUsableIpNum = 2
        ipInfo = CommonRest.getEthPortIpInfo(records)
        ipList = CommonRest.getIpList(ipInfo, ipType)
        usableIpList = CommonRest.getUsableIpList(context, ipList)
        if usableIpList is None or len(usableIpList) < defaultUsableIpNum:
            return False

        return True

    @staticmethod
    def getEthPortIpInfo(records, portTypeList=[restData.RestCfg.EthPortLogicTypeEnum.MNGT,
                                                restData.RestCfg.EthPortLogicTypeEnum.MNGT_SRV]):
        """
        @summary: 获取ETH端口的ip信息
        @param record: rest回显
        @return: ip信息
                                        信息格式：{location:{'ipv4':ipv4,'ipv6':ipv6}}
        """
        ipInfo = {}
        datas = CommonRest.getData(records)
        for data in datas:
            logicType = CommonRest.getRecordValue(data, restData.RestCfg.EthPort.LOGIC_TYPE)
            if logicType not in portTypeList:
                continue

            location = CommonRest.getRecordValue(data, restData.RestCfg.EthPort.LOCATION)
            ipv4 = CommonRest.getRecordValue(data, restData.RestCfg.EthPort.IPV4_ADDR)
            ipv6 = CommonRest.getRecordValue(data, restData.RestCfg.EthPort.IPV6_ADDR)
            ipInfo[location] = {"ipv4": ipv4, "ipv6": ipv6}

        return ipInfo

    @staticmethod
    def getLinkedPorts(rest, objType):
        """
        @summary: 根据type获取连接的端口
        """
        paramDict = CommonRest.getUriParamDict(objType)
        data = CommonRest.execCmd(rest, paramDict, None, restData.RestCfg.RestMethod.GET)
        records = CommonRest.getData(data)
        resList = []
        if records is None:
            return resList

        for record in records:
            healthStatus = CommonRest.getRecordValue(record, restData.RestCfg.PublicAttributes.HEALTH_STATUS)
            runningStatus = CommonRest.getRecordValue(record, restData.RestCfg.PublicAttributes.RUNNING_STATUS)
            if healthStatus == restData.RestCfg.HealthStatusEnum.NORMAL and runningStatus == restData.RestCfg.RunningStatusEnum.LINK_UP:
                resList.append(record)
        return resList

    @staticmethod
    def getLocation(portList):
        """
        @summary: 根据端口集合取得一个location的字典。
        @param ports: 传入的端口列表
        @return: 返回以location为key，以record为值的字典。
        """
        locationDict = {}
        if portList is None:
            return locationDict

        for record in portList:
            location = CommonRest.getRecordValue(record, restData.RestCfg.PublicAttributes.LOCATION)
            locationDict[location] = record
        return locationDict

    @staticmethod
    def getUnLinkedSasPortsLocation(beforePorts, afterPorts):
        """
        @summary: 检查端口一致性。
        @param beforePorts: 更换之前记录的端口
                afterPorts: 更换之后的端口
        @return: 未连接的所有端口的location列表
        """
        unlinkedSasPorts = set()
        locDictBefore = CommonRest.getLocation(beforePorts)
        locDictAfter = CommonRest.getLocation(afterPorts)
        locsAfter = locDictAfter.keys()
        for beforeLoc in locDictBefore.keys():
            if beforeLoc not in locsAfter:
                unlinkedSasPorts.add(beforeLoc)

        return list(unlinkedSasPorts)

    @staticmethod
    def getCtrlEnclosureHeight(rest, logger):
        '''
        @summary: 获取当前集群框高
        @param rest: rest对象
        '''
        records = CommonRest.getEnclosureRecords(rest)
        for record in records:
            logicType = CommonRest.getRecordValue(record, restData.RestCfg.Enclosure.LOGIC_TYPE)
            logger.logInfo("====get getCtrlEnclosureHeight====record=[%s]." % str(record))
            logger.logInfo("====get getCtrlEnclosureHeight====logicType=[%s]." % str(logicType))
            if logicType == restData.Enum.EncLogicTypeEnum.CTRL:
                height = CommonRest.getRecordValue(record, restData.RestCfg.Enclosure.HEIGHT)
                logger.logInfo(
                    "====get getCtrlEnclosureHeight====logicType=[%s]====height=[%s]." % (str(logicType), str(height)))
                return height

    @staticmethod
    def getHardwareRecords(rest, hardwareType):
        '''
        @summary: 根据硬件形态获取该硬件的所有记录
        @param rest: rest对象
        @param hardwareType: 硬件形态
        '''
        paramDict = {}
        uriParamDict = CommonRest.getUriParamDict(hardwareType)
        record = CommonRest.execCmd(rest, uriParamDict, paramDict, restData.RestCfg.RestMethod.GET)
        return CommonRest.getData(record)


class Graph2Rest():
    @staticmethod
    def getRecordValue(data, field):
        """
        @summary: 获取指定属性Key值转化后成指定类型并返回
        @param data: REST返回的单个DATA数据（类型为字典）. 
            {   propertyIndex0:{ “type”:“int”，“value”：propertyValue }，
                propertyIndex1: { “type”:“int”，“value”：propertyValue }，
                …
                }
        @param field: 包含 index, type。其中key是DATA中的属性。TYPE是该属性的值。 
        """
        try:
            fieldIndex = field.get("index")
            fieldType = field.get("type")
            paramInfo = data.get(str(fieldIndex), {})
            paramValue = paramInfo.get("value")

            if fieldType == restDataType.DATA_TYPE_UNSIGN_INT or \
                            fieldType == restDataType.DATA_TYPE_SIGN_INT or \
                            fieldType == restDataType.DATA_TYPE_BOOL or \
                            fieldType == restDataType.DATA_TYPE_ENUM or \
                            fieldType == restDataType.DATA_TYPE_TIME:
                if paramValue == None:
                    return 0
                else:
                    return int(paramValue)
            elif fieldType == restDataType.DATA_TYPE_ARRAY:
                if paramValue == None:
                    return ""
                else:
                    return ast.literal_eval(paramValue)
            elif fieldType == restDataType.DATA_TYPE_U64LONG_TYPE:
                if paramValue == None:
                    return 0
                else:
                    return long(paramValue)
            elif fieldType == restDataType.DATA_TYPE_STRING or \
                            fieldType == restDataType.DATA_TYPE_JSON:
                if paramValue == None:
                    return ""
                else:
                    return str(paramValue)
            else:
                return unicode(paramValue)
        except Exception, e:
            raise Exception(config.ERR_CODE_CAN_NOT_EXECUTE, "exec cmd return data format error: %s" % str(e))


class CommonRestService():

    @staticmethod
    def get(rest, uri):
        restConn = rest.getRest()
        url = rest.getBaseUri() + uri

        try:
            responseStr = restConn.execGet(url, "").getContent()
            responseDict = Tlv2Rest.convertRecord2Dict(responseStr)
            errInfo = Tlv2Rest.getErrInfo(responseDict)
            errCode = Tlv2Rest.getErrorTetail(errInfo, restData.ErrorInfo.CODE)
            if errCode == "0":
                data = Tlv2Rest.getData(responseDict)
                if data is None:
                    data = []
                return data
            else:
                description = Tlv2Rest.getErrorTetail(errInfo, restData.ErrorInfo.DESCRIPTION)
                raise IsmException(int(errCode), description)
        except IsmException,e:
            raise e
        except (Exception, JException), e:
            raise IsmException(int(REST_CAN_NOT_EXECUTE), e)

    @staticmethod
    def get4Page(rest, uri):
        allDatas = []
        index = 0
        while True:
            range = "range=[%s-%s]" % (index, index+100)
            rangeUri = CommonRestService.addRangeUri(uri, range)
            datas = CommonRestService.get4Big(rest, rangeUri)
            allDatas.extend(datas)
            if len(datas) < 100:
                break
            index = index + 100
        return allDatas


    @staticmethod
    def addRangeUri(uri, str):
        if "?" in uri:
            countUri = uri.replace("?", "?" + str + "&")
        else:
            countUri = uri + "?" + str
        return countUri

    @staticmethod
    def get4Big(rest, uri):
        restConn = rest.getRest()
        url = rest.getBaseUri() + uri

        try:
            responseDict = restConn.execGetForBig(url, "")
            errInfo = Tlv2Rest.getErrInfo(responseDict)
            errCode = Tlv2Rest.getErrorTetail(errInfo, restData.ErrorInfo.CODE)
            if errCode == "0":
                data = Tlv2Rest.getData(responseDict)
                if data is None:
                    data = []
                return data
            else:
                description = Tlv2Rest.getErrorTetail(errInfo, restData.ErrorInfo.DESCRIPTION)
                raise IsmException(int(errCode), description)
        except IsmException,e:
            raise e
        except JException, e:
            raise IsmException(int(REST_CAN_NOT_EXECUTE), e)

    @staticmethod
    def put(rest, uri, paramDict):
        restConn = rest.getRest()
        url = rest.getBaseUri() + uri

        paramJson = jsonUtil.dict2JsonStr(paramDict)
        responseStr = restConn.execPut(url, paramJson).getContent()
        responseDict = Tlv2Rest.convertRecord2Dict(responseStr)
        errInfo = Tlv2Rest.getErrInfo(responseDict)
        errCode = Tlv2Rest.getErrorTetail(errInfo, restData.ErrorInfo.CODE)
        if errCode == "0":
            data = Tlv2Rest.getData(responseDict)
            return data
        else:
            description = Tlv2Rest.getErrorTetail(errInfo, restData.ErrorInfo.DESCRIPTION)
            raise IsmException(int(errCode), description)

