# -*- coding: UTF-8 -*-
from frame.cli import cliUtil
from frame.common import common
from frame.context import contextUtil
import traceback

def execute(context):
    '''
    @summary: 系统文件共享服务检查
    '''
    status, cliRet, errMsg = NasProtocolCheck(context).check()
    return common.getUpgEvaluationRs(status, cliRet, errMsg)

class NasProtocolCheck():
    def __init__(self, context):
        self.lang = contextUtil.getLang(context)
        self.cli = contextUtil.getSSH(context)
        self.logger = common.getLogger(contextUtil.getLogger(context), __file__)
        self.context = context
        self.allCliRet = ""

    def check(self):
        try:
            cmd = "show service cifs"
            checkRet = cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
            self.allCliRet += checkRet[1]
            if not cliUtil.hasCliExecPrivilege(checkRet[1]):
                contextUtil.setItem(self.context, 'supportnas', False)
                return CheckStatus.PASS, self.allCliRet, ""

            checkRet = cliUtil.getControllerIdList(self.cli, self.lang)
            if checkRet[0] != True:
                contextUtil.setItem(self.context, 'supportnas', "NOCHECK")
                errMsg = common.getMsg(self.lang, "excute.cmd.error")
                return CheckStatus.NOCHECK, checkRet[1], errMsg
            controlList = checkRet[1]
            ftpCheckResult = FtpCheck(self.context).check()
            ftpCheckResult.protocol = "FTP"
            self.allCliRet += ftpCheckResult.cliRet

            cifsCheckResult = CifsCheck(self.context).check(controlList)
            cifsCheckResult.protocol = "CIFS"
            self.allCliRet += cifsCheckResult.cliRet

            nfsCheckResult = NfsCheck(self.context).check(controlList)
            nfsCheckResult.protocol = "NFS"
            self.allCliRet += nfsCheckResult.cliRet

            ndmpCheckResult = NdmpCheck(self.context).check()
            ndmpCheckResult.protocol = "NDMP"
            self.allCliRet += ndmpCheckResult.cliRet

            allCheckResult = [ftpCheckResult, cifsCheckResult, nfsCheckResult, ndmpCheckResult]
            allStatus = map(lambda x : x.status, allCheckResult)
            noPassProtocol = []
            warningProtocol = []
            for tmpResult in allCheckResult:
                if tmpResult.status == CheckStatus.NOPASS:
                    noPassProtocol.append(tmpResult.protocol)
                elif tmpResult.status == CheckStatus.WARNING:
                    warningProtocol.append(tmpResult.protocol)
            status = CheckStatus.mergeStatus(allStatus)
            errMsg = ""
            if len(noPassProtocol) > 0:
                errMsg += common.getMsg(self.lang, "nasProtocol.cannotUpgrade", ",".join(noPassProtocol))
            if len(warningProtocol) > 0:
                errMsg += common.getMsg(self.lang, "nasProtocol.exsitRisk", ",".join(warningProtocol))

            noPassProtocol = []
            warningProtocol = []
            nocheckProtocol = []
            for tmpResult in allCheckResult:
                if tmpResult.status == CheckStatus.NOPASS:
                    noPassProtocol.append(tmpResult.protocol)
                elif tmpResult.status == CheckStatus.WARNING:
                    warningProtocol.append(tmpResult.protocol)
                elif tmpResult.status == CheckStatus.NOCHECK:
                    nocheckProtocol.append(tmpResult.protocol)
            status = CheckStatus.mergeStatus(allStatus)
            errMsg = ""
            if len(noPassProtocol) > 0:
                errMsg += common.getMsg(self.lang, "checkProtocol.cannotUpgrade", ",".join(noPassProtocol))
            if len(nocheckProtocol) > 0:
                errMsg += common.getMsg(self.lang, "excute.cmd.error")
            if len(warningProtocol) > 0:
                errMsg += common.getMsg(self.lang, "checkProtocol.exsitRisk", ",".join(warningProtocol))

            #若没有成功设置是否存在NAS业务，且命令执行失败，设置为“nocheck”
            allNaveNasDatas = map(lambda x: x.data, allCheckResult)
            haveNas = mergeHaveNas(allNaveNasDatas)
            contextUtil.setItem(self.context, 'supportnas', haveNas)

            return status, self.allCliRet, errMsg
        except:
            self.logger.logInfo("NasProtocolCheck exception[%s]" % traceback.format_exc())
            errMsg = common.getMsg(self.lang, "excute.cmd.error")
            contextUtil.setItem(self.context, 'supportnas', "NOCHECK")
            return CheckStatus.NOCHECK, self.allCliRet, errMsg

class CheckStatus():
    NOCHECK = cliUtil.RESULT_NOCHECK
    PASS = True
    NOPASS = False
    WARNING = cliUtil.RESULT_WARNING

    @staticmethod
    def mergeStatus(allStatus):
        if CheckStatus.NOPASS in allStatus:
            return CheckStatus.NOPASS
        if CheckStatus.NOCHECK in allStatus:
            return CheckStatus.NOCHECK
        if CheckStatus.WARNING in allStatus:
            return CheckStatus.WARNING
        return CheckStatus.PASS

class CheckResult():
    def __init__(self, status, cliRet, data=None):
        self.status = status
        self.cliRet = cliRet
        self.data = data
        self.protocol = ""

class CifsCheck():
    def __init__(self, context):
        self.lang = contextUtil.getLang(context)
        self.cli = contextUtil.getSSH(context)
        self.logger = common.getLogger(contextUtil.getLogger(context), __file__)

    def check(self, controllers):
        allCliRet = ""
        allCheckStatus = []
        haveNas = False
        for ctrlId in controllers:
            cmd = "show cifs server_info controller=%s" % ctrlId
            checkRet = cliUtil.excuteCmdInDeveloperMode(self.cli, cmd, True, self.lang)
            allCliRet += checkRet[1]
            if not cliUtil.hasCliExecPrivilege(checkRet[1]):
                allCheckStatus.append(CheckStatus.PASS)
                break
            if checkRet[0] != True:
                haveNas = setHaveNasNocheck(haveNas)
                allCheckStatus.append(CheckStatus.NOCHECK)
                continue

            if cliUtil.queryResultWithNoRecord(checkRet[1]):
                allCheckStatus.append(CheckStatus.PASS)
                continue

            cliRetLinesList = cliUtil.getVerticalCliRet(checkRet[1])
            if len(cliRetLinesList) == 0:
                haveNas = setHaveNasNocheck(haveNas)
                allCheckStatus.append(CheckStatus.NOCHECK)
                continue
            # Protocol Detail 的值大于0，则表示存在CIFS业务
            protocolDetail = cliRetLinesList[0].get("Protocol Detail", "")
            if '/' in protocolDetail:
                for num in protocolDetail.split('/'):
                    if num != '0':
                        haveNas = True

            protocolWithoutCA = cliRetLinesList[0].get("Protocol Without CA")
            connectionCountWithoutCA = '0'
            if protocolWithoutCA:
                connectionCountWithoutCA = protocolWithoutCA.split('/')[-1]

            totalConnectionCount = cliRetLinesList[0].get("Connection Count", '0')
            if totalConnectionCount == '0' and connectionCountWithoutCA == '0':
                allCheckStatus.append(CheckStatus.PASS)
            elif connectionCountWithoutCA != '0':
                allCheckStatus.append(CheckStatus.NOPASS)
            else:
                allCheckStatus.append(CheckStatus.WARNING)
        status = CheckStatus.mergeStatus(allCheckStatus)
        return CheckResult(status, allCliRet, haveNas)

class NfsCheck():
    def __init__(self, context):
        self.lang = contextUtil.getLang(context)
        self.cli = contextUtil.getSSH(context)
        self.logger = common.getLogger(contextUtil.getLogger(context), __file__)
        self.currentVersion = contextUtil.getCurVersion(context)

    def check(self, controllers):
        allCheckStatus = []
        allCliRet = ""

        haveNas = False
        for ctrlId in controllers:
            cmd = "show nfs server_info controller=%s" % ctrlId
            checkRet = cliUtil.excuteCmdInDeveloperMode(self.cli, cmd, True, self.lang)
            allCliRet += checkRet[1]
            if not cliUtil.hasCliExecPrivilege(checkRet[1]):
                allCheckStatus.append(CheckStatus.PASS)
                break
            if checkRet[0] != True:
                haveNas = setHaveNasNocheck(haveNas)
                allCheckStatus.append(CheckStatus.NOCHECK)
                continue

            if cliUtil.queryResultWithNoRecord(checkRet[1]):
                allCheckStatus.append(CheckStatus.PASS)
                continue

            cliRetLinesList = cliUtil.getVerticalCliRet(checkRet[1])
            if len(cliRetLinesList) == 0:
                haveNas = setHaveNasNocheck(haveNas)
                allCheckStatus.append(CheckStatus.NOCHECK)
                continue

            nfs4ConnectionCount = cliRetLinesList[0].get("NFS4 Connection Count", '0')
            nfs3ConnectionCount = cliRetLinesList[0].get("NFS3 Connection Count", '0')

            if nfs3ConnectionCount == '0' and nfs4ConnectionCount == '0':
                allCheckStatus.append(CheckStatus.PASS)
            elif nfs4ConnectionCount != '0' and self.currentVersion < "V300R006C00":
                # NFS3 Connection Count和NFS4 Connection Count的值大于0，则表示存在NFS业务
                haveNas = True
                allCheckStatus.append(CheckStatus.NOPASS)
            else:
                # NFS3 Connection Count和NFS4 Connection Count的值大于0，则表示存在NFS业务
                haveNas = True
                allCheckStatus.append(CheckStatus.WARNING)
        status = CheckStatus.mergeStatus(allCheckStatus)
        return CheckResult(status, allCliRet, haveNas)

class NdmpCheck():
    def __init__(self, context):
        self.lang = contextUtil.getLang(context)
        self.cli = contextUtil.getSSH(context)
        self.logger = common.getLogger(contextUtil.getLogger(context), __file__)

    def check(self):
        cmd = "show service ndmp_connection"
        checkRet = cliUtil.excuteCmdInDeveloperMode(self.cli, cmd, True, self.lang)
        if not cliUtil.hasCliExecPrivilege(checkRet[1]):
            return CheckResult(CheckStatus.PASS, checkRet[1], False)

        if checkRet[0] != True:
            return CheckResult(CheckStatus.NOCHECK, checkRet[1], "NOCHECK")

        cliRetLinesList = cliUtil.getVerticalCliRet(checkRet[1])
        if len(cliRetLinesList) == 0:
            return CheckResult(CheckStatus.NOCHECK, checkRet[1], "NOCHECK")

        if cliRetLinesList[0].get("Client Connection") != '0':
            #Client Connection的值大于0，则表示存在NDMP业务
            return CheckResult(CheckStatus.NOPASS, checkRet[1], True)
        return CheckResult(CheckStatus.PASS, checkRet[1], False)

class FtpCheck():
    def __init__(self, context):
        self.lang = contextUtil.getLang(context)
        self.cli = contextUtil.getSSH(context)
        self.logger = common.getLogger(contextUtil.getLogger(context), __file__)
        self.allCliRet = ""

    def check(self):
        cmd = "show service ftp"
        checkRet = cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        self.allCliRet += checkRet[1]
        if not cliUtil.hasCliExecPrivilege(checkRet[1]):
            return CheckResult(CheckStatus.PASS, self.allCliRet, False)
        if checkRet[0] != True:
            return CheckResult(CheckStatus.NOCHECK, self.allCliRet, "NOCHECK")
        cliRetLinesList = cliUtil.getVerticalCliRet(checkRet[1])
        if len(cliRetLinesList) == 0:
            return CheckResult(CheckStatus.NOCHECK, self.allCliRet, "NOCHECK")
        runningStatus = cliRetLinesList[0].get("Running Status", "")
        if runningStatus.lower() == "stop":
            return CheckResult(CheckStatus.PASS, self.allCliRet, False)

        acceesSwitch = cliRetLinesList[0].get("Anonymous Access Switch", "")
        if acceesSwitch.lower() == "on":
            #Running Status的值是Running，且 Anonymous Access Switch的值是on，存在NAS业务
            return CheckResult(CheckStatus.NOPASS, self.allCliRet, True)

        return self.checkPermission()


    def checkPermission(self):
        cmd = "show share_permission ftp"
        checkRet = cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        self.allCliRet += checkRet[1]
        if not cliUtil.hasCliExecPrivilege(checkRet[1]):
            return CheckResult(CheckStatus.PASS, self.allCliRet, False)
        if checkRet[0] != True:
            return CheckResult(CheckStatus.NOCHECK, self.allCliRet, "NOCHECK")

        if cliUtil.queryResultWithNoRecord(checkRet[1]):
            return CheckResult(CheckStatus.PASS, self.allCliRet, False)
        # 不为No matching records，则确认存在ftp共享，存在FTP业务
        cliRetLinesList = cliUtil.getHorizontalCliRet(checkRet[1])
        if len(cliRetLinesList) == 0:
            return CheckResult(CheckStatus.NOCHECK, self.allCliRet, "NOCHECK")
        return CheckResult(CheckStatus.NOPASS, self.allCliRet, True)

def setHaveNasNocheck(haveNas):
    if haveNas != True:
        haveNas = "NOCHECK"
    return haveNas

def mergeHaveNas(allHaveNasData):
    if True in allHaveNasData:
        return True
    if "NOCHECK" in allHaveNasData:
        return "NOCHECK"
    return False