#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.
import traceback

from common.contentParse import setCmdRet4EvalAndReport
from common.migration_constants import DEVICE_NAME
from common.util import error_msg_handler, updateItemProgress
from hosts.common.migration_parse_utils import remove_blank_lines
from hosts.sudo_utils import get_sudo_cmd
from storages.storage_common import STORAGE_DEV_MODEL, STORAGE_EMC_ISILON

"""
SSH连接命令执行的基础类
  1、关联查询的命令，需要子类处理好命令参数，调用generate_relative_cmd_list方法组装命令列表
"""
DES = "description"
CMD = "cmd"
EVAL = "Eval"


class BaseCmdExecute(object):
    def __init__(self, context, cmd_list=None):
        if cmd_list is None:
            cmd_list = []
        self.context = context
        self.cli = context.get("SSH")
        self.language = context.get("lang")
        self.display = context.get("ret_map")
        self.err_msg = ""
        self.log = context.get("Logger")
        self.cmd_list = cmd_list
        self.execute_cmd_method = None

    def execute_pure_cmd_list(self, ignore_bad_start_line=False, commands=None, cur_step=5, progress_width=85.0,
                              timeout=60):
        """
        命令执行并刷新进度
        :param ignore_bad_start_line: 是否忽略回文前面的空白行（目前Isilon会有）
        :param commands: 传入的命令列表
        :param cur_step: 当前步骤起始进度
        :param progress_width: 当前阶段进度宽度
        :param timeout: 执行命令超过这个时间抛出异常超时, 默认60s
        """
        cmds = commands if commands else self.cmd_list
        prg_step = progress_width / len(cmds) if len(cmds) != 0 else progress_width
        self.update_progress(cur_step, 0)
        try:
            for cmd_dic in cmds:
                cmd = cmd_dic.get(CMD)
                cmd_display_temp = self.exec_cmd(cmd, ignore_bad_start_line, timeout=timeout)
                # 原始回显信息中有控制文本颜色的ANSI转义序列时会乱码，需要对ANSI转义序列进行替换
                if hasattr(self.cli, 'eliminatesResultColorCode'):
                    cmd_display_temp = self.cli.eliminatesResultColorCode(cmd_display_temp)
                # 如果返回信息中第一行仅有 ^ 那说明该语句在华为存储CLI命令中不支持，使用空行跳过错误语句，避免错误语句影响下一条语句
                if cmd_display_temp.splitlines()[0].strip() == '^':
                    self.exec_cmd("")
                cur_step = self.update_progress(cur_step, prg_step)
                self.display.put(cmd_dic.get(DES), cmd_display_temp)
                self.check_result(cmd_display_temp, cmd)
                if cmd_dic.get(EVAL, False):
                    setCmdRet4EvalAndReport(self.display, cmd_display_temp, cmd_dic.get(DES))
            self.display.put("err_msg", self.display.get("err_msg", "") + self.err_msg)
        except Exception:
            self.log.error("the except info is:" + traceback.format_exc())

    def generate_relative_cmd_list(self, params, cmd_dicts):
        """
        关联命令的拼接
        :param params: 命令参数，元组的格式，即便单个参数也需要是元组的格式（"param1",）
        :param cmd_dicts: 关联命令的列表
        """
        if not params or not cmd_dicts:
            return
        for cmd_dic in cmd_dicts:
            for param in params:
                self.cmd_list.append({DES: cmd_dic.get(DES) + "_".join(param), CMD: cmd_dic.get(CMD) % param,
                                      EVAL: cmd_dic.get(EVAL, False)})

    def check_result(self, cmd_display_temp, cmd):
        """
        命令执行结果的检查
        :param cmd_display_temp: 回文
        :param cmd: 命令
        """
        _, result = error_msg_handler(cmd_display_temp, cmd, self.language)
        self.err_msg = self.err_msg + result

    def update_progress(self, cur_step, prg_step):
        """
        进度刷新
        :param cur_step: 进度
        :param prg_step: 进度增长值
        :return: 当前进度值
        """
        tmp_cur_step = cur_step
        cur_step = cur_step + prg_step
        if int(cur_step) > int(tmp_cur_step):
            updateItemProgress(self.context, int(cur_step))
        return cur_step

    def exec_cmd(self, cmd, ignore_bad_start_line=False, timeout=60):
        """
        命令执行
        :param cmd: 命令
        :param ignore_bad_start_line: 是否忽略回文前面的空白行
        :param timeout: 执行命令超过这个时间抛出异常超时, 默认60s
        :return: 回文
        """
        exec_cmd_method = self.cli.execCmdNoLogTimout if not self.execute_cmd_method else self.execute_cmd_method
        cmd_result = exec_cmd_method(self.handle_cmd(cmd), timeout)
        if self.context.get(STORAGE_DEV_MODEL, "") == STORAGE_EMC_ISILON:
            cmd_result = self.check_and_get_cmd_result(cmd, cmd_result)
        if not ignore_bad_start_line:
            return cmd_result
        return get_ignore_bad_start_line(cmd, cmd_result)

    def handle_cmd(self, cmd):
        """
        命令处理：sudo场景
        :param cmd: 命令
        :return: 命令
        """
        return get_sudo_cmd(cmd, self.context.get(DEVICE_NAME, "").lower(), self.context)

    def check_and_get_cmd_result(self, cmd, content):
        """
        执行isilon的命令，可能会遇见连接异常，导致回文错误（多行长的空白行，没有命令的回文），此时需要重新连接
        :param cmd: cmd
        :param content: 第一次的回文
        :return: 正确回文
        """
        for line in content.splitlines():
            if line.strip().endswith(cmd):
                return content
        self.cli.reConnect()
        return self.cli.execCmdNoLogTimout(cmd, 60)

    def get_param_from_cmd(self, cmd, start_index):
        """
        直接将命令结果作为关联命令的参数
        :param cmd: 命令
        :param start_index: 参数的起始行索引
        :return: 数据
        """
        return change_list_to_tuple(remove_blank_lines(self.exec_cmd(cmd).splitlines()[start_index:]))

    def change_dir_to_user_home(self):
        """
        切换当前目录到登录用户的家目录
        """
        login_user = self.context.get("LoginUsername")
        if login_user == "root":
            self.exec_cmd("cd /root")
        else:
            self.exec_cmd("cd /home/" + login_user)


def get_ignore_bad_start_line(cmd, cmd_result):
    """
    忽略命令前的冗余行
    :param cmd: 命令
    :param cmd_result: 回文
    :return: 处理后的回文
    """
    cmd_index = -1
    lines = cmd_result.splitlines()
    for index, line in enumerate(lines):
        if line.strip().endswith(cmd):
            cmd_index = index
            break
    if cmd_index != -1:
        return "\n".join(lines[cmd_index:])
    return cmd_result


def change_list_to_tuple(param_list):
    """
    将单参数的列表转为tuple
    :param param_list: 参数列表
    :return: 参数元组列表
    """
    tuple_list = []
    for param in param_list:
        tuple_list.append((param,))
    return tuple_list
