#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
import re
from common.contentParse import getStringRet, check_lst
from common.util import log, check_result
from common.cmd_execute import DES, CMD, BaseCmdExecute, EVAL

CMD_LIST = [
    {DES: "cmd_display_hba_card_lscfg", CMD: "lscfg -vpl fcs*", EVAL: True},
    {DES: "cmd_display_hab_card_fc_check", CMD: "lsdev -Cc adapter |grep FC"},
    {DES: "cmd_display_hba_card_prot_cfg", CMD: "lspath |grep -i fscsi|grep -i enabled"},
    {DES: "cmd_display_hba_card_lsdev", CMD: "lsdev | grep iscsi"},
    {DES: "cmd_display_hba_list", CMD: "lscfg -vpl fcs*|grep fcs|awk '{print $1}'"},
    {DES: "cmd_display_hba_iscsi_list", CMD: "lsdev | grep iscsi | awk '{print $1}'"},

]

HBA_CMD_DICTS = [
    {DES: "cmd_display_hba_card_lsattr_", CMD: "lsattr -El %s"},
    {DES: "cmd_display_hba_card_status_", CMD: "fcstat %s | grep Attention"},
    {DES: "cmd_display_hba_card_status_D_", CMD: "fcstat -D %s | grep Attention"},
    {DES: "cmd_display_hba_card_prot_info_", CMD: "lsdev -p %s"},
    {DES: "cmd_display_hba_card_info_fc_", CMD: "lsmcode -A | grep %s", EVAL: True},
]

HBA_ISCSI_CMD_DICTS = [
    {DES: "cmd_display_hba_card_iscsi_info_", CMD: "lsattr -El %s"},
]

HBA_FC_CMD_DICTS = [
    {DES: "cmd_display_hba_card_info_fc_", CMD: "lsmcode -A | grep %s"},
]


def execute(context):
    """
    采集Aix主机HBA相关信息
    :param context: 上下文
    :return: 执行结果
    """
    HbaInfo(context).execute()


class HbaInfo(BaseCmdExecute):
    def __init__(self, context):
        BaseCmdExecute.__init__(self, context, CMD_LIST)

    def execute(self):
        self.generate_relative_cmd_list(self.get_param_from_cmd("lscfg -vpl fcs*|grep fcs|awk '{print $1}'", 1),
                                        HBA_CMD_DICTS)
        self.generate_relative_cmd_list(self.get_param_from_cmd("lsdev | grep iscsi | awk '{print $1}'", 1),
                                        HBA_ISCSI_CMD_DICTS)
        self.generate_relative_cmd_list(self.get_hba_fc_list(), HBA_FC_CMD_DICTS)
        self.execute_pure_cmd_list()
        self.special_cmd_execute()

    def get_hba_fc_list(self):
        """
        获取hba fc结果
        :return: 数据
        """
        result = []
        content_lines = self.exec_cmd("lsdev -Cc adapter |grep FC").splitlines()
        content_lines = check_lst(content_lines)
        for line in content_lines[1:]:
            if ("FC" in line) and ("FCoE" not in line):
                lst = re.split('\\s+', line.strip())
                result.append((lst[0].strip(),))
        return result

    def special_cmd_execute(self):
        """
        特殊执行，非单纯的命令执行
        """
        content = self.exec_cmd("lspath |grep -i fscsi|grep -i enabled")
        if not self.check_fc_hba(content):
            return
        self.handle_hba_info(content.splitlines())

    def get_hba_port_list(self, fc_hba_port_list):
        """
        获取HBA端口
        :return: 数据
        """
        hba_port_list = []
        for hba_port in fc_hba_port_list[1:]:
            if "fscsi" not in hba_port:
                log.warn(self.context, "not fc hba card:" + hba_port)
                continue
            hba_port_list.append(hba_port.split()[-1])
        return hba_port_list

    def check_fc_hba(self, fc_hba_port_content):
        """
        hba fc数据检查
        :param fc_hba_port_content: fc hba回文
        :return: 检查结果
        """
        hba_card_info_list = []
        if "command not found" in fc_hba_port_content:
            hba_card_info_list.append({"HBATimeout": "NA"})
            getStringRet(self.context, str(hba_card_info_list).replace("u\'", "\'"))
            return False
        if fc_hba_port_content.find("hdisk") == -1:
            hba_card_info_list.append({"HBATimeout": "not involved"})
            getStringRet(self.context, str(hba_card_info_list).replace("u\'", "\'"))
            return False
        return True

    def handle_cmd_result(self, hba_port, result_key, cmd_exec_result):
        """
        回文处理
        :param hba_port: hba端口
        :param result_key: 命令描述
        :param cmd_exec_result: 执行回文
        :return: 处理结果
        """
        self.display.put(result_key, cmd_exec_result)
        cmd_exec_result_changed = cmd_exec_result.replace("+", "")
        find_result = re.findall(r'dyntrk\s+yes[.\s].+fc_err_recov\s+fast_fail', cmd_exec_result_changed, re.S)
        return {"Host": {"PortID": hba_port}, "Timeout": "changed" if find_result else "unchanged"}

    def handle_hba_info(self, fc_hba_port_list):
        """
        处理HBA数据
        :param fc_hba_port_list: fc端口数据
        """
        hba_card_info_list = []
        fun_err_msg = ""
        hba_port_list = self.get_hba_port_list(fc_hba_port_list)
        for hba_port in list(set(hba_port_list)):
            result_key = "cmd_display_hba_card_tmout_cfg_" + hba_port
            exec_cmd = "lsattr -El " + hba_port
            cmd_exec_result = self.exec_cmd(exec_cmd)
            hba_card_info_dict = self.handle_cmd_result(hba_port, result_key, cmd_exec_result)
            hba_card_info_list.append(hba_card_info_dict)
            fun_err_msg += check_result(cmd_exec_result, exec_cmd, self.language)
        self.display.put("err_msg", fun_err_msg + self.display.get("err_msg", ""))
        getStringRet(self.context, str(hba_card_info_list).replace("u\'", "\'"))
