#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2020. All rights reserved.
import json
import traceback
from common.contentParse import get_return_from_txt


def execute(context):
    global logger
    logger = context.get("Logger")
    result = get_powerha_cluster_result(context)
    ret_map = context.get("map")
    ret_map.put("data", json.dumps(result))
    return context


def get_powerha_cluster_result(context):
    """
    获取数据结果
    :param context:上下文
    :return: 数据结果
    """
    result = {}
    result["host_ip"] = context.get("ip")
    try:
        power_version = get_power_version(context)
        power_name, power_disk, node_names, groups = get_power_info(context)
        group_vg_dict = get_group_vg(context)
        heartbeat_disk = get_heartbeat_disk(context)
    except Exception:
        logger.error("parse error: %s" % str(traceback.format_exc()))
        power_name, power_disk, heartbeat_disk, node_names, groups = "", "", "", [], []
    result["cluster_name"] = power_name
    result["cluster_version"] = power_version
    result["repository_disk"] = power_disk
    result["heartbeat_disk"] = heartbeat_disk
    result["nodes"] = get_power_nodes(node_names, groups, group_vg_dict)
    return result


def get_heartbeat_disk(context):
    """
    获取心跳盘
    :param context: context
    :return: 心跳盘
    """
    heartbeat_disk = []
    network_lvs = get_network_lv_list(context)
    vg_lvs_dict = get_vg_items_dict(context, "lsvg -o|lsvg -li", "LV NAME")
    vg_pvs_dict = get_vg_items_dict(context, "lsvg -o|lsvg -pi", "PV_NAME")
    for vg_name, lvs in vg_lvs_dict.items():
        for lv in lvs:
            if lv in network_lvs:
                heartbeat_disk.extend(vg_pvs_dict.get(vg_name, []))
                break
    return ", ".join(heartbeat_disk)


def get_network_lv_list(context):
    """
    获取pv列表
    :param context: context
    :return: pv列表
    """
    context["command"] = "/usr/es/sbin/cluster/utilities/cltopinfo"
    network_lv_info = get_return_from_txt(context)
    begin_flag = False
    network_lvs = []
    for line in network_lv_info:
        if line.startswith("NODE "):
            begin_flag = True
            continue
        if line.startswith("Resource Group "):
            break
        if not line.strip() or not begin_flag or line.strip().startswith("Network "):
            continue
        network_lvs.append(line.strip().split()[-1].split("/")[-1])
    return network_lvs


def get_vg_items_dict(context, cmd, key_word):
    """
    获取vg与lvs/pvs的映射
    :param context: context
    :param cmd: 命令
    :param key_word: 关键字
    :return: vg与lvs/pvs的映射
    """
    context["command"] = cmd
    vg_items_info = get_return_from_txt(context)
    vg_items_dict = {}
    vg = ""
    items = []
    item_flag = False
    for line in vg_items_info:
        if line.endswith(":"):
            if vg and items:
                vg_items_dict[vg] = items
            vg = line.rstrip(":")
            items = []
            item_flag = False
            continue
        if line.startswith(key_word):
            item_flag = True
            continue
        if item_flag and line.strip():
            items.append(line.split()[0])
    if vg and items:
        vg_items_dict[vg] = items
    return vg_items_dict


def get_power_version(context):
    """
    获取power ha的版本
    :param context: context
    :return: 版本
    """
    version = ""
    context["command"] = "lslpp -l|grep cluster"
    version_info = get_return_from_txt(context)
    for line in version_info:
        if "cluster.es.server.rte" in line and len(line.split()) > 2:
            version = line.split()[1]
    return version


def get_power_info(context):
    """
    获取power ha的集群名字、仓库盘、节点列表、groups
    :param context: context
    :return: 版本
    """
    power_name = ""
    power_disk = ""
    node_names = []
    groups = []
    context["command"] = "/usr/es/sbin/cluster/utilities/cltopinfo"
    cltop_info = get_return_from_txt(context)
    for line in cltop_info:
        if "Cluster Name" in line and ":" in line:
            power_name = line.split(":")[-1]
            continue
        if "Repository Disk" in line and ":" in line:
            power_disk = line.split(":")[-1]
            continue
        if "NODE" in line \
                and len(line.split()) > 1 and line.strip().endswith(":"):
            node_names.append(line.strip().split()[-1][:-1])
        if "Resource Group" in line and len(line.split()) > 2:
            groups.append(line.split()[-1])
    return power_name, power_disk, node_names, groups


def get_value_without_key(line, key_item):
    """
    key-value之间没有分隔符，去掉key值，保留value
    :param line: 行数据
    :param key_item: key值
    :return: value值
    """
    return line[len(key_item):].strip()


def get_group_vg(context):
    """
    获取group与vg对应的字典
    :param context: context
    :return: group-vg
    """
    group_list = get_group_vg_list(context)
    group_vg = {}
    volume_groups = "Volume Groups"
    volume_groups_concurrent = "Concurrent Volume Groups"
    resource_group_name = "Resource Group Name"
    for one_group in group_list:
        group = ""
        vg_list = []
        for line in one_group:
            if line.startswith(resource_group_name):
                group = get_value_without_key(line, resource_group_name)
                continue
            if line.startswith(volume_groups):
                vg_list.append(get_value_without_key(line, volume_groups))
                continue
            if line.startswith(volume_groups_concurrent):
                vg_list.append(get_value_without_key(line, volume_groups_concurrent))
                continue
        if group and vg_list:
            group_vg[group] = "\n".join(vg_list)
    return group_vg


def get_group_vg_list(context):
    """
    对group_vg 回显进行分割，形成独立的列表
    :param context: context
    :return: group_vg列表
    """
    context["command"] = "/usr/es/sbin/cluster/utilities/clshowres"
    group_vg_info = get_return_from_txt(context)
    group_lists = []
    one_group = []
    flag = False
    for index, line in enumerate(group_vg_info):
        if line.startswith("Resource Group Name"):
            flag = True
        if line.startswith("Resource Group Name") and one_group:
            group_lists.append(one_group)
            one_group = []
        if flag:
            one_group.append(line)
        if flag and index == len(group_vg_info) - 1:
            group_lists.append(one_group)
    return group_lists


def get_power_nodes(node_names, groups, group_vg_dict):
    """
    获取powerHa的节点信息
    :param node_names: node_names
    :param groups: groups
    :param group_vg_dict: group_vg_dict
    :return: 节点信息
    """
    nodes = []
    if not node_names or not groups:
        return get_empty_power_ha()
    for node in node_names:
        for group in groups:
            one_node = {}
            one_node["node_name"] = node
            one_node["node_group"] = group
            one_node["node_vg"] = group_vg_dict.get(group, "")
            nodes.append(one_node)
    return nodes


def get_empty_power_ha():
    """
    生成空数据
    :return: 空数据
    """
    nodes = []
    one_node = {}
    one_node["node_name"] = ""
    one_node["node_group"] = ""
    one_node["node_vg"] = ""
    nodes.append(one_node)
    return nodes
