# --*-- coding:utf-8 --*--
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2020. All rights reserved.
import json
import traceback

from common.contentParse import get_return_from_txt
from common.util import is_digit


def execute(context):
    global logger
    logger = context.get("Logger")
    result = get_result(context)
    ret_map = context.get("map")
    ret_map.put("data", json.dumps(result))
    return context


def get_result(context):
    """
    获取vg相关数据
    :param context:上下文
    :return: vg数据
    """
    result = {}
    result["host_ip"] = context.get("ip")
    vg_info_list = []
    try:
        vg_list = get_vg_info(context)
        # 没有vg数据，返回一个空格式
        if not vg_list:
            one_vg_na = get_null_result(context)
            vg_info_list.append(one_vg_na)
            result["vg_info_list"] = vg_info_list
            return result
        fs_list = get_filesystem_info(context)
        vg_major_number = get_vg_major_number(context)
        # 循环vg列表，组装数据
        for vg in vg_list:
            vg_info_list.append(get_one_vg_info(vg, fs_list, vg_major_number))
    except Exception:
        logger.error("parse error: %s" % str(traceback.format_exc()))
        vg_info_list.append(get_null_result())
    result["vg_info_list"] = vg_info_list
    return result


def get_one_vg_info(vg, fs_list, vg_major_number):
    """
    组装一个vg数据
    :param vg: vg信息
    :param fs_list: 文件系统数据
    :param vg_major_number: major_number数据
    :return: 一个vg数据
    """
    one_vg = {}
    lvs = []
    vg_name = vg.get("vg_name", "")
    one_vg["vg_name"] = vg_name
    one_vg["vg_state"] = vg.get("vg_state", "")
    one_vg["vg_size"] = vg.get("vg_size", "")
    one_vg["pvs"] = vg.get("pvs", [])
    one_vg["vg_major_number"] = vg_major_number.get(vg_name, "")
    lv_list = vg.get("lvs", [])
    # 组装lvm+file
    for lv in lv_list:
        one_lv = {}
        one_lv["lv_name"] = lv.get("lv_name", "")
        one_lv["lv_state"] = lv.get("lv_state", "")
        one_lv["lv_size"] = lv.get("lv_size", "")
        one_lv = get_filesystem(one_lv, fs_list)
        lvs.append(one_lv)
    one_vg["lvs"] = lvs
    return one_vg


def get_vg_major_number(context):
    """
     获取vg的major number，如IPL_rootvg（最后一列/分割最后一个字段） 10（第五列，去掉右边的逗号）
        crw-rw----    1 root     system       10,  0 Mar 12 18:31 /dev/IPL_rootvg
        crw-------    1 root     system       10,  0 Jul 16 16:16 /dev/__vg10
        crw-------    1 root     system       37,  0 Apr 03 14:25 /dev/__vg37
    :param context: context
    :return: vg的major number
    """
    context["command"] = "ls -al /dev/*"
    major_info = get_return_from_txt(context)
    vg_major = {}
    for line in major_info:
        items = line.split()
        if len(items) < 6:
            continue
        major = items[4].strip(",").strip()
        vg = items[-1].split("/")[-1].strip()
        vg_major[vg] = major
    return vg_major


def get_filesystem(one_lv, fs_list):
    """
    组装filesystem必须包含有 lv_name
    :param one_lv: one_lv
    :param fs_list: fs_list
    :return: one_lv
    """
    flag = False
    for fs in fs_list:
        filesystem = fs.get("filesystem", "")
        temp_filesystem = filesystem
        if "/" in temp_filesystem:
            filesystem = temp_filesystem.split("/")[-1]
        if one_lv.get("lv_name", "") in filesystem:
            one_lv["filesystem"] = fs.get("filesystem", "")
            one_lv["fs_total_size"] = fs.get("fs_total_size", "")
            one_lv["fs_used_size"] = fs.get("fs_used_size", "")
            one_lv["fs_mount_point"] = fs.get("fs_mount_point", "")
            flag = True
            break
    if not flag:
        one_lv["filesystem"] = ""
        one_lv["fs_total_size"] = ""
        one_lv["fs_used_size"] = ""
        one_lv["fs_mount_point"] = ""
    return one_lv


def get_vg_name_size(context):
    """
    获取vg及其size
    :param context: context
    :return: vg及其size
    """
    context["command"] = "lsvg"
    vg_names = get_return_from_txt(context)
    vg_name_state = get_vg_name_state(context)
    vgs = []
    for name in vg_names:
        context["command"] = "lsvg " + name
        vg_info = get_return_from_txt(context)
        one_vg = {}
        flag = False
        for line in vg_info:
            if "PP SIZE:" in line:
                pp_size = line.split("PP SIZE:")[-1].split()[0]
            if "TOTAL PPs:" in line:
                pp_total = line.split("TOTAL PPs:")[-1].split()[0]
                flag = True
        if not flag:
            continue
        one_vg["vg_name"] = name
        one_vg["vg_state"] = vg_name_state.get(name, "")
        one_vg["pp_size"] = pp_size
        one_vg["pp_total"] = pp_total
        vgs.append(one_vg)
    return vgs


def get_vg_name_state(context):
    """
    获取vg_name与state的对应
    :param context: 上下文
    :return: vg_name_state
    """
    context["command"] = "lspv"
    vg_state_info = get_return_from_txt(context)
    vg_name_state = {}
    for line in vg_state_info:
        if len(line.split()) == 4:
            vg_name_state[line.split()[2]] = line.split()[-1]
    return vg_name_state


def get_vg_list_with_lv(context):
    """
    获取vg与lv的对应
    :param context: 上下文
    :return: vg_list_with_lv
    """
    context["command"] = "lsvg -o|lsvg -li"
    vg_info = get_return_from_txt(context)
    flag = False
    vg_lv_list = []
    one_vg = []
    for index, line in enumerate(vg_info):
        if line.endswith(":"):
            flag = True
        if ":" in line and one_vg:
            vg_lv_list.append(one_vg)
            one_vg = []
        if flag and line.strip():
            one_vg.append(line)
        if flag and index == len(vg_info) - 1:
            vg_lv_list.append(one_vg)
    return vg_lv_list


def get_lv_name_size(context):
    """
    获取lv及其size
    :param context: context
    :return: lv及其size
    """
    vg_list = get_vg_list_with_lv(context)
    lvs = []
    vg_name = ""
    for one_vg in vg_list:
        lv_flag = False
        for line in one_vg:
            if line.endswith(":"):
                vg_name = line.split(":")[0]
                lv_flag = False
            if "LV NAME" in line:
                lv_flag = True
                continue
            if not lv_flag or not vg_name:
                continue
            items = line.split()
            if len(items) > 5:
                one_lv = {"vg_name": vg_name, "lv_name": items[0], "lv_pps": items[3], "lv_state": items[5]}
                lvs.append(one_lv)
    return lvs


def get_vg_list_with_pv(context):
    """
    获取vg与pv的对应
    :param context: 上下文
    :return: vg_list_with_pv
    """
    context["command"] = "lsvg -o|lsvg -pi"
    vg_info = get_return_from_txt(context)
    flag = False
    vg_pv_list = []
    one_vg = []
    for index, line in enumerate(vg_info):
        if line.endswith(":"):
            flag = True
        if ":" in line and one_vg:
            vg_pv_list.append(one_vg)
            one_vg = []
        if flag and line.strip():
            one_vg.append(line)
        if flag and index == len(vg_info) - 1:
            vg_pv_list.append(one_vg)
    return vg_pv_list


def get_pv_name_size(context):
    """
    获取pv及其size
    :param context: context
    :return: pv及其size
    """
    vg_list = get_vg_list_with_pv(context)
    pvs = []
    vg_name = ""
    for one_vg in vg_list:
        pv_flag = False
        for line in one_vg:
            if line.endswith(":"):
                vg_name = line.split(":")[0]
                pv_flag = False
            if "PV_NAME" in line:
                pv_flag = True
                continue
            if not pv_flag or not vg_name:
                continue
            items = line.split()
            if len(items) > 2:
                one_pv = {"vg_name": vg_name, "pv_name": items[0], "pv_pps": items[2]}
                pvs.append(one_pv)
    return pvs


def get_size_by_ppsize(pp_size, pps):
    """
    计算容量size，单位MiB pp_size单位只有mib
    :param pp_size:
    :param pps:
    :return:
    """
    float_value = 0.0
    if not is_digit(pp_size) or not is_digit(pps):
        return float_value
    size = round(float(pp_size.strip()) * float(pps.strip()) / 1024, 2)
    return str(size) + " GiB"


def get_id_dict(context):
    """
    获取pv-pvid数据
    :param context:  context
    :return: vg 数据
    """
    context["command"] = "lspv"
    pv_ids_info = get_return_from_txt(context)
    pv_ids = {}
    for line in pv_ids_info:
        items = line.split()
        if len(items) == 4:
            pv_ids[items[0]] = items[1]
    return pv_ids


def get_vg_info(context):
    """
    获取vg 的相关数据
    :param context:  context
    :return: vg 数据
    """
    vg_list_info = get_vg_name_size(context)
    lv_list_info = get_lv_name_size(context)
    pv_list_info = get_pv_name_size(context)
    pv_id_dic = get_id_dict(context)
    if not vg_list_info:
        return []
    vg_list = []
    for vg in vg_list_info:
        one_vg = {}
        vg_name = vg.get("vg_name", "")
        vg_pp_size = vg.get("pp_size", "")
        vg_pp_total = vg.get("pp_total", "")
        one_vg["vg_name"] = vg_name
        one_vg["vg_state"] = vg.get("vg_state", "")
        one_vg["vg_size"] = get_size_by_ppsize(vg_pp_size, vg_pp_total)
        lvs = []
        for lv in lv_list_info:
            if vg_name == lv.get("vg_name", ""):
                one_lv = {}
                one_lv["lv_name"] = lv.get("lv_name", "")
                one_lv["lv_state"] = lv.get("lv_state", "")
                lv_pps = lv.get("lv_pps", "")
                one_lv["lv_size"] = get_size_by_ppsize(vg_pp_size, lv_pps)
                lvs.append(one_lv)
        one_vg["lvs"] = lvs
        pvs = []
        for pv in pv_list_info:
            if vg_name == pv.get("vg_name", ""):
                pv_pps = pv.get("pv_pps", "")
                pv_size = get_size_by_ppsize(vg_pp_size, pv_pps)
                pv_tmp = {}
                pv_name = pv.get("pv_name", "")
                pv_tmp["pv_name"] = pv_name
                pv_tmp["pv_pvid"] = pv_id_dic.get(pv_name, "")
                pv_tmp["pv_size"] = pv_size
                pvs.append(pv_tmp)
        if not pvs:
            pvs.append({"pv_name": "", "pv_size": "", "pv_pvid": ""})
        one_vg["pvs"] = pvs
        vg_list.append(one_vg)

    return vg_list


def get_filesystem_info(context):
    """
    获取filesystem 数据
    :param context: 上下文
    :return: filesystem数据
    """
    context["command"] = "df -g"
    file_system_info = get_return_from_txt(context)

    file_info = []
    all_info = []
    for line in file_system_info[1:]:
        all_info.extend(line.split())
    line_num = len(all_info) / 7
    for i in range(0, line_num):
        one_file = {}
        one_file["filesystem"] = all_info[7 * i]
        total_size = all_info[7 * i + 1]
        free_size = all_info[7 * i + 2]
        one_file["fs_total_size"] = total_size + " g"
        if is_digit(total_size) and is_digit(free_size):
            use_size = float(total_size) - float(free_size)
            size = "" if use_size == float("0") else str(use_size) + " g"
            one_file["fs_used_size"] = size
        else:
            one_file["fs_used_size"] = ""
        one_file["fs_mount_point"] = all_info[7 * i + 6]
        file_info.append(one_file)
    return file_info


def get_null_result(context):
    """
    空数据格式
    :param context: 上下文
    :return: 空数据
    """
    lvs = []
    one_vg_na = {}
    one_vg_na["host_ip"] = context.get("ip")
    one_vg_na["vg_name"] = ""
    one_vg_na["vg_state"] = ""
    one_vg_na["vg_size"] = ""
    one_vg_na["vg_major_number"] = ""
    one_vg_na["pvs"] = [{"pv_name": "", "pv_size": "", "pv_pvid": ""}]
    one_lv = {}
    one_lv["lv_name"] = ""
    one_lv["lv_state"] = ""
    one_lv["lv_size"] = ""
    one_lv["filesystem"] = ""
    one_lv["fs_total_size"] = ""
    one_lv["fs_used_size"] = ""
    one_lv["fs_mount_point"] = ""
    lvs.append(one_lv)
    one_vg_na["lvs"] = lvs
    return one_vg_na
