# --*-- coding:utf-8 --*--
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2020. All rights reserved.

import json
import traceback
import re

from common.migration_constants import HOST_TYPE_AIX
from common.contentParse import get_return_from_txt
from hosts.common.migra_summary_util import get_host_name, get_multi_type_version
from hosts.common.migra_summary_util import get_cluster_group
from hosts.common.migra_summary_util import get_empty_overview_result
from hosts.AIX.migration.aix_os import get_type_version
from hosts.common.migration_parse_utils import get_host_ip


def execute(context):
    global logger
    logger = context.get("Logger")
    result = get_result(context)
    ret_map = context.get("map")
    ret_map.put("data", json.dumps(result))
    return context


def get_result(context):
    """
    获取数据
    :param context: 上下文
    :return: 数据
    """
    result = {}
    try:
        multi_type, multi_version = get_multi_type_version_aix(context)
        cluster_name = get_cluster_name(context)
        db_name, db_version = get_db_name_version(context)
        hba_list, hba_type = get_hba_wwn_list_type(context)
        result["host_ip"] = context.get("ip")
        result["host_name"] = get_host_name(context)
        result["os_version"] = get_os_version(context)
        result["hba_type"] = hba_type
        result["hba_wwn_list"] = hba_list
        result["multi_type"] = multi_type
        result["multi_version"] = multi_version
        result["cluster_name"] = cluster_name
        result["hba_detail"] = parse_hba_detail(context)
        if cluster_name:
            result["cluster_version"] = db_version
        else:
            result["cluster_version"] = ""
        result["cluster_ips"] = get_cluster_group(context)
        result["lvm_type"] = get_lvm_type(context)
        result["db_name"] = db_name
        result["db_version"] = db_version
        result["disk_use"] = get_disk_use(context)
        result["real_ip"] = get_host_ip(context)
        result["multi_path_status"] = ""
    except Exception:
        logger.error("parse error: %s" % str(traceback.format_exc()))
        result = get_empty_overview_result(context)
    return result


def parse_hba_detail(context):
    context["command"] = "lscfg -vpl fcs*"
    hba_detail_lines = get_return_from_txt(context)
    hba_detail_lines = [line for line in hba_detail_lines if line.strip()]
    hba_cards = parse_hba_card_infos(hba_detail_lines)
    hba_details = []
    for hba_info in hba_cards:
        one_hba_infos = [
            "ID=" + hba_info.get("wwn", ""),
            "Vendor=" + hba_info.get('vendor', ""),
            "Type=" + hba_info.get("type", ""),
            "Speed=" + hba_info.get("speed", ""),
            "Location=" + hba_info.get("location", "")
        ]
        hba_details.append(";".join(one_hba_infos))
    return "\n".join(hba_details)


def parse_hba_card_infos(hba_detail_lines):
    one_hba = {}
    hba_cards = []
    for line in hba_detail_lines:
        if line.startswith("fcs"):
            one_hba = {}
            hba_cards.append(one_hba)
            items = line.split()
            items = [item for item in items if item.endswith('Gb')]
            one_hba["speed"] = items[0] if items else ""
        if '..' not in line:
            continue
        if 'Manufacturer' in line:
            one_hba["vendor"] = parse_hba_info_value(line)
        if 'Customer Card ID Number' in line:
            one_hba['type'] = parse_hba_info_value(line)
        if 'Network Address' in line:
            one_hba['wwn'] = parse_hba_info_value(line)
        if 'Hardware Location Code' in line:
            one_hba['location'] = parse_hba_info_value(line)
    return hba_cards


def parse_hba_info_value(line):
    return re.split("\\.{2,}", line)[1].strip()


def get_disk_use(context):
    """
    获取 磁盘绑定形式
    :param context: 上下文
    :return:磁盘绑定形式
    """
    # 该版本先不做，留空
    return ""


def get_db_name_version(context):
    """
    获取db的name、version
    :param context: 上下文
    :return: db的name、version
    """
    context["command"] = "sqlplus / as sysdba"
    version_info = get_return_from_txt(context)
    for line in version_info:
        if "Release" in line:
            if len(line.split("Release")) > 1:
                version = line.split("Release")[1].split()[0]
                return "oracle", version
    return "", ""


def get_lvm_type(context):
    """
    获取 lvm 的类型
    :param context: 上下文
    :return: lvm 的类型
    """
    lvm_type = []
    context["command"] = "lsvg"
    lvm_info = get_return_from_txt(context)
    for line in lvm_info:
        if line and not "rootvg" == line.lower().strip():
            lvm_type.append("LVM")
            break

    context["command"] = "hastatus -sum"
    lvm_info = get_return_from_txt(context)
    for line in lvm_info:
        if "-- SYSTEM STATE" in line:
            lvm_type.append("VxVM")
            break
    if not lvm_type:
        return ""
    return ", ".join(lvm_type)


def get_os_version(context):
    """
    获取类型+版本
    :param context:上下文
    :return: 类型+版本
    """
    os_type, os_version = get_type_version(context)
    return os_type + " " + os_version


def get_cluster_name(context):
    """
    获取集群的名字、版本
    :param context: 上下文
    :return: 名字、版本
    """
    context["command"] = "select value from v$parameter " \
                         "where name='cluster_database';"
    cluster_info = get_return_from_txt(context)
    flag = False

    for line in cluster_info:
        if "----------" in line:
            flag = True
            continue
        if flag:
            if "TRUE" in line:
                return "Oracle RAC"
    return ""


def get_hba_wwn_list_type(context):
    """
    获取hba_wwn、type
    :param context: 上下文
    :return: hba_wwn
    """
    context["command"] = "lscfg -vpl fcs*"
    hba_info = get_return_from_txt(context)
    hba_list = []
    hba_type = ""
    for line in hba_info:
        if "Network Address" in line:
            hba_list.append(line.split(".")[-1])
        if "Customer Card ID Number" in line and not hba_type:
            hba_type = line.split(".")[-1]
    return "\n".join(hba_list), hba_type


def get_multi_type_version_aix(context):
    """
    获取多路径的类型、版本
    :param context: 上下文
    :return: 类型、版本
    """
    multi_type, multi_version = get_multi_type_version(context, HOST_TYPE_AIX)
    if multi_type:
        return multi_type, multi_version
    context["command"] = "lsdev -Cc disk"
    mpio_multi = get_return_from_txt(context)
    for line in mpio_multi:
        if "mpio" in line.lower():
            return "MPIO", "NA"
    return "", ""
