# --*-- coding:utf-8 --*--
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2020. All rights reserved.
import json
import traceback

from common.migration_constants import HOST_TYPE_HPUX
from common.contentParse import get_return_from_txt
from hosts.common.migra_summary_util import get_host_name, get_multi_type_version
from hosts.common.migra_summary_util import get_cluster_group
from hosts.common.migra_summary_util import get_nmp_disk_list
from hosts.common.migra_summary_util import get_empty_overview_result
from hosts.HPUX.migration.hpux_os import get_type_version
from hosts.common.migration_parse_utils import get_host_ip
from hosts.parse_util import get_key_values_from_content_lines


def execute(context):
    result = get_result(context)
    ret_map = context.get("map")
    ret_map.put("data", json.dumps(result))
    return context


def get_result(context):
    """
    获取数据
    :param context: 上下文
    :return: 数据
    """
    logger = context.get("Logger")
    try:
        result = {}
        multi_type, multi_version = get_multi_type_version_hp_ux(context)
        cluster_name = get_cluster_name(context)
        db_name, db_version = get_db_name_version(context)
        hba_list, hba_type = get_hba_wwn_list_type(context)
        result["host_ip"] = context.get("ip")
        result["host_name"] = get_host_name(context)
        result["os_version"] = get_os_version(context)
        result["hba_type"] = hba_type
        result["hba_wwn_list"] = hba_list
        result["multi_type"] = multi_type
        result["multi_version"] = multi_version
        result["cluster_name"] = cluster_name
        result["hba_detail"] = get_hba_detail(context)
        if cluster_name:
            result["cluster_version"] = db_version
        else:
            result["cluster_version"] = ""
        result["cluster_ips"] = get_cluster_group(context)
        result["lvm_type"] = ""
        result["db_name"] = db_name
        result["db_version"] = db_version
        result["disk_use"] = ""
        result["real_ip"] = get_host_ip(context)
        result["multi_path_status"] = ""
        return result
    except Exception:
        logger.error("parse error: %s" % str(traceback.format_exc()))
        return get_empty_overview_result(context)


def get_hba_detail(context):
    context["command"] = "ioscan -funC fc"
    disk_lines = get_return_from_txt(context)
    disk_values = [line for line in disk_lines if line.strip()]
    hba_detail = []
    for disk_value in disk_values:
        context["command"] = "fcmsutil %s" % disk_value
        hba_port = get_return_from_txt(context)
        start_bid = "Vendor ID is"
        keys_tuple = ('Vendor ID is', 'Link Speed', 'N Port Port World Wide Name', 'Hardware Path is', "TYPE")
        delimiter = " ="
        hba_detail_info = get_key_values_from_content_lines(hba_port, start_bid, keys_tuple, delimiter)
        for info in hba_detail_info:
            hba_detail.append(build_single_hba_info(info))
    return "\n".join(hba_detail).strip()


def build_single_hba_info(info):
    single_hba_detail = [
        "ID=" + info.get('N Port Port World Wide Name', ''),
        "Vendor=" + info.get('Vendor ID is', ''),
        "Type=" + info.get('TYPE', ''),
        "Speed=" + info.get('Link Speed', ''),
        "Location=" + info.get('Hardware Path is', '')
    ]
    return ";".join(single_hba_detail).strip()


def get_db_name_version(context):
    """
    获取db的name、version
    :param context: 上下文
    :return: db的name、version
    """
    context["command"] = "sqlplus / as sysdba"
    version_info = get_return_from_txt(context)
    for line in version_info:
        if "Release" in line:
            if len(line.split("Release")) > 1:
                version = line.split("Release")[1].split()[0]
                return "oracle", version
    return "", ""


def get_os_version(context):
    """
    获取类型+版本
    :param context:上下文
    :return: 类型+版本
    """
    os_type, os_version = get_type_version(context)
    return os_type + " " + os_version


def get_cluster_name(context):
    """
    获取集群的名字、版本
    :param context: 上下文
    :return: 名字、版本
    """
    context["command"] = "select value from v$parameter " \
                         "where name='cluster_database';"
    cluster_info = get_return_from_txt(context)
    flag = False

    for line in cluster_info:
        if "----------" in line:
            flag = True
            continue
        if flag:
            if "TRUE" in line:
                return "Oracle RAC"
    return ""


def get_hba_type_devlist(context):
    """
    获取hba_type和dev_list
    :param context: context
    :return: hba_type和dev_list
    """
    context["command"] = "ioscan -funC fc"
    hba_info = get_return_from_txt(context)
    dev_flag = False
    dev_list = []
    hba_type = ""
    for line in hba_info:
        if "Fibre Channel" in line:
            if not hba_type:
                hba_type = line.split("Fibre Channel")[0].strip().split("  ")[
                    -1].strip()
            dev_flag = True
            continue
        if dev_flag:
            dev_list.append(line.strip())
            dev_flag = False
            continue
    return hba_type, dev_list


def get_hba_wwn_list(dev_list, context):
    """
    获取hp的fc wwn
    :param dev_list: dev_list
    :param context: context
    :return: hp的fc wwn
    """
    hba_wwn_list = []
    for dev in dev_list:
        context["command"] = "fcmsutil %s" % dev
        hba_wwn_info = get_return_from_txt(context)
        for line in hba_wwn_info:
            if "N_Port Port World Wide Name" in line:
                hba_wwn_list.append(line.split("=")[-1].strip())
                break
    return hba_wwn_list


def get_hba_wwn_list_type(context):
    """
    获取hba_wwn、type
    :param context: 上下文
    :return: hba_wwn
    """
    hba_type, dev_list = get_hba_type_devlist(context)
    hba_wwn_list = get_hba_wwn_list(dev_list, context)
    return "\n".join(hba_wwn_list), hba_type


def get_multi_type_version_hp_ux(context):
    """
    获取多路径的类型、版本
    :param context: 上下文
    :return: 类型、版本
    """
    multi_type, multi_version = get_multi_type_version(context, HOST_TYPE_HPUX)
    if multi_type:
        return multi_type, multi_version
    nmp_disk_list = get_nmp_disk_list(context)
    if nmp_disk_list:
        return "NMP", "NA"
    return "", ""
