# --*-- coding:utf-8 --*--
import re
import traceback
from common.contentParse import get_return_from_txt
from common.migration_constants import HOST_TYPE_LINUX, HOST_TYPE_AIX
from common.util import chang_unit2_gb
from hosts.common.migration_parse_utils import safe_get_split


def get_udev_result(context):
    """
    获取udev 的数据结果
    :param context: 上下文
    :return: udev数据结果
    """
    result = []
    real_udev = get_right_udev(context)
    # asmdisk_name
    oracle_disk_paths = get_oracle_diskpath(context)
    if not real_udev or not oracle_disk_paths:
        return result
    key_words = ["KERNEL", "RESULT", "OWNER", "GROUP"]
    for line in real_udev:
        flag = True
        for key_word in key_words:
            if key_word not in line:
                flag = False
                break
        # 这一行必须包含有所有关键字，并且oracle_diskpath以NAME字段结尾
        if flag and verify_name(line, oracle_disk_paths):
            udev_one = {}
            udev_one["udev_kernel"] = \
                line.split("KERNEL")[1].split(",")[0][3:-1]
            udev_one["result"] = line.split("RESULT")[1].split(",")[0][4:-1]
            udev_one["asmdisk_name"] = get_asm_disk_name(line)
            udev_one["udev_owner"] = line.split("OWNER")[1].split(",")[0][2:-1]
            udev_one["udev_group"] = line.split("GROUP")[1].split(",")[0][2:-1]
            result.append(udev_one)
    return result


def get_asm_disk_name(line):
    """
    获取asm磁盘路径
    :param line: 数据行
    :return: asm磁盘路径
    """
    key_word = ""
    if "NAME" in line:
        key_word = "NAME"
    if "SYMLINK+" in line:
        key_word = "SYMLINK+"
    if key_word:
        return line.split(key_word)[1].split(",")[0][2:-1]
    return ""


def get_udev_list(context):
    """
    获取命令匹配的 结果集
    :param context:  上下文
    :return: 匹配的 结果集
    """
    command = "cmd_display_oracle_udev_raw_rule"
    context["command"] = command
    udev_info = get_return_from_txt(context)

    result_list = []
    temp_list = []
    includ_command = "cat /etc/udev/rules.d/"
    end_command = ".rules"

    for num, line in enumerate(udev_info):
        if includ_command in line \
                and line.strip().lower().endswith(end_command):
            if temp_list:
                result_list.append(temp_list)
                temp_list = []
        temp_list.append(line)
        if num == len(udev_info) - 1:
            result_list.append(temp_list)
    return result_list


def get_right_udev(context):
    """
    获取正确的udev ，需要匹配"KERNEL", "RESULT", "OWNER", "GROUP"
    :param context: 上下文
    :return: udev 回显对象
    """
    udev_list = get_udev_list(context)
    key_words = ["KERNEL", "RESULT", "OWNER", "GROUP"]
    for one_udev in udev_list:
        for line in one_udev:
            flag = True
            for key_word in key_words:
                if key_word not in line:
                    flag = False
                    break
            if flag:
                return one_udev
    return []


def verify_name(line, oracle_disk_paths):
    """
    验证disk_name 存在于oracle path路径
    :param oracle_disk_paths: oracle_disk_paths
    :param line: 回显行
    :return: 是否存在
    """
    for oracle_path in oracle_disk_paths:
        if oracle_path.endswith(get_asm_disk_name(line)):
            return True
    return False


def get_oracle_diskpath(context):
    """
    获取 oracel中diskpath
    :param context:上下文
    :return: 数据结果
    """
    result = []
    disk_info = get_oracle_disk_info(context)
    for line in disk_info:
        result.append(line.split()[-4])
    return result


def get_oracle_disk_info(context):
    """
    获取磁盘列表
    :param context:上下文
    :return: 磁盘列表
    """
    context["command"] = "select group_number,disk_number,name,mount_status," \
                         "header_status,path,state,total_mb,free_mb,sector_size from v$asm_disk;"
    command_result = get_return_from_txt(context)
    flag = False
    info_list = []
    second_flag = False
    for line in command_result:
        if "FREE_MB" in line:
            second_flag = True
            continue
        if "-------" not in line and second_flag:
            flag = True
        if flag:
            info_list.append(line)

    all_right_line = []
    temp_line = ""
    # 处理换行的场景；以及close状态，当前命令总字段10个，存在数据缺失一个字段值场景
    for line in info_list:
        if not line.strip():
            continue
        line_split = line.split()
        if len(line_split) == 10 or (len(line_split) == 9 and line_split[-7] == "CLOSED"):
            all_right_line.append(line)
            continue
        if 4 < len(line_split) <= 9:
            temp_line = line
            continue
        two_line_item_count = len(temp_line.split()) + len(line_split)
        if two_line_item_count == 10 or (two_line_item_count == 9 and "CLOSED" in temp_line):
            all_right_line.append(temp_line + " " + line)
            temp_line = ""
            continue
    return all_right_line


def get_oracle_version(context):
    """
    获取oracle 数据库 版本
    :param context:上下文
    :return:数据库版本
    """
    version = ""
    context["command"] = "sqlplus / as sysdba"
    version_info = get_return_from_txt(context)
    for line in version_info:
        items = line.split("Release")
        if len(items) > 1:
            version = items[1].split()[0]
    return version


def get_mpio_multi(context):
    """
    获取mpio多路径
    :param context: 上下文
    :return: mpio多路径
    """
    logger = context.get("Logger")
    try:
        result = {}
        result["multipath_type"] = "MPIO"
        result["version"] = "NA"
        disks = get_mpio_disks(context)
        if not disks:
            return False, result
        disk_wwn_dict = get_disk_wwn_mpio(context)
        disk_vp = get_mpio_disk_vp(context)
        disk_list = []
        for line in disks:
            one_disk = {}
            disk_name = line.split()[0]
            one_disk["disk_name"] = disk_name
            one_disk["vendor"] = disk_vp.get(disk_name, {}).get("vendor", "")
            one_disk["product"] = disk_vp.get(disk_name, {}).get("product", "")
            one_disk["lun_wwn"] = disk_wwn_dict.get(disk_name, "")
            one_disk["lun_size"] = get_size_by_disk_name(disk_name, context)
            one_disk["reserve_policy"] = \
                get_policy_by_disk_name(disk_name, context)
            disk_list.append(one_disk)
        result["disk_list"] = disk_list
        return True, result
    except Exception:
        logger.error("parse error: %s" % str(traceback.format_exc()))
        return False, result


def get_mpio_disk_vp(context):
    """
    获取aix主机自带多路径mpio，disk对应的vendor、product
    :param context: 上下文
    :return: disk对应的vendor、product
    """
    disk_vp = {}
    context["command"] = "lscfg -vl hdisk*"
    disk_vp_info = get_return_from_txt(context)
    disk_lists = []
    one_disk = []
    flag = False
    for index, line in enumerate(disk_vp_info):
        if "hdisk" in line:
            flag = True
        if "hdisk" in line and one_disk:
            disk_lists.append(one_disk)
            one_disk = []
        if flag:
            one_disk.append(line)
        if flag and index == len(disk_vp_info) - 1:
            disk_lists.append(one_disk)
    for disk in disk_lists:
        disk_name = ""
        disk_vendor = ""
        disk_product = ""
        for line in disk:
            if line.strip().startswith("hdisk"):
                disk_name = line.split()[0]
            if line.strip().startswith("Manufacturer"):
                disk_vendor = line.split(".")[-1]
            if line.strip().startswith("Machine Type and Model"):
                disk_product = line.split(".")[-1]
        if disk_name and disk_vendor and disk_product:
            vp = {}
            vp["vendor"] = disk_vendor
            vp["product"] = disk_product
            disk_vp[disk_name] = vp
    return disk_vp


def is_aix_native_multi_path(line):
    """
    是否是自带多路径管理--MPIO
    AIX主机处理HP 3para的特殊格式：以3PAR InServ Virtual Volume结尾，认为是MPIO管理
    :param line: 数据行
    :return: 是否是自带多路径管理--MPIO
    """
    line = line.strip().lower()
    return "MPIO".lower() in line or line.endswith("3PAR InServ Virtual Volume".lower())


def get_mpio_disks(context):
    """
    获取mpio的disks
    :param context: 上下文
    :return: disks 回显
    """
    context["command"] = "lsdev -Cc disk"
    disk_info = get_return_from_txt(context)
    disks = []
    for line in disk_info:
        if is_aix_native_multi_path(line):
            disks.append(line)
    return disks


def get_disk_wwn_mpio(context):
    """
    获取 disk与wwn对应的字典
    :param context: 上下文
    :return: 字典
    """
    result = {}
    context["command"] = "lspv -u"
    disk_wwn_info = get_return_from_txt(context)
    for line in disk_wwn_info:
        if len(line) < 80:
            continue
        result[line[0:16].strip()] = line[80:min(len(line), 148)].strip()
    return result


def get_ultra_multi(context, host_type):
    """
    获取ultra多路径
    :param host_type: 主机类型
    :param context: 上下文
    :return: ultra多路径
    """
    logger = context.get("Logger")
    try:
        result = {}
        multipath_type = "UltraPath"
        result["multipath_type"] = multipath_type
        version = get_ultra_version(context, host_type)
        result["version"] = version

        # 不存在多路径
        if not version:
            return False, result
        header_line, disks = get_ultra_disks(context, host_type)
        # 存在多路径，但是没有数据
        if version and not disks:
            result["disk_list"] = get_null_disk_list()
            return True, result
        disk_list = get_ultra_disk_list(header_line, disks, context, host_type)
        result["disk_list"] = disk_list
        return True, result
    except Exception:
        logger.error("parse error: %s" % str(traceback.format_exc()))
        return False, result


def get_dmp_multi(context):
    """
    获取dmp多路径
    :param context: 上下文
    :return: dmp多路径
    """
    logger = context.get("Logger")
    result = {}
    version = get_dmp_version(context)
    # 版本获取不到,则表示不存在dmp多路径
    if not version:
        return False, result
    try:
        result["multipath_type"] = "DMP"
        result["version"] = version
        disks = get_dmp_disk(context)
        # 存在多路径, 但是没有disk数据, 结果设置空数据
        if not disks:
            result["disk_list"] = get_null_disk_list()
            return True, result
        result["disk_list"] = disks
        return True, result
    except Exception:
        logger.error("parse error: %s" % str(traceback.format_exc()))
        return False, result


def get_ultra_disks(context, host_type):
    """
    获取ultra的disks
    :param host_type: host_type
    :param context: 上下文
    :return: disks 回显
    """
    disk_info = []
    if host_type == "linux":
        context["command"] = "upadmin show vlun type=all"
        disk_info = get_return_from_txt(context)
        # 老版本环境命令兼容
        if "Vlun ID" not in "".join(disk_info):
            context["command"] = "upadmin show vlun"
            disk_info = get_return_from_txt(context)
    elif host_type == "aix":
        context["command"] = "upadm show vlun type=all"
        disk_info = get_return_from_txt(context)
        # 老版本环境命令兼容
        if "Vlun ID" not in "".join(disk_info):
            context["command"] = "upadm show vlun"
            disk_info = get_return_from_txt(context)
    elif host_type == "solaris":
        context["command"] = "upadm show vlun type=all"
        disk_info = get_return_from_txt(context)
    header_line, disks = "", []
    flag = False
    for line in disk_info:
        if "Vlun ID" in line and not flag:
            flag = True
            header_line = line
            continue
        if flag and "-----" in line:
            break
        if flag:
            disks.append(line)
    return header_line, disks


def check_old_linux_ultra(header_line):
    """
    判断是否为Linux主机的老版本ultra，会缺少第二个Disk字段，那么获取数据时，需要将后面的索引-1
    老版本 Vlun ID         Name   Lun WWN    Status Capacity Ctrl(Own/Work)    Array Name
    新版本 Vlun ID  Disk   Name   Lun WWN    Status  Capacity  Ctrl(Own/Work)  Array Name
    :param header_line: 数据表头
    :return: 索引操作值
    """
    items = re.split("\\s{2,}", header_line)
    if len(items) > 2 and items[1] == "Disk":
        return 0
    return -1


def get_ultra_disk_list(header_line, disks, context, host_type):
    """
    获取ultra多路径disk列表
    :param header_line: 数据表头
    :param disks: 磁盘数据回文行
    :param context: context
    :param host_type: 主机类型
    :return: ultra多路径disk列表
    """
    result = []
    id_vp = get_ultra_id_vp(disks, host_type, context)
    index_handle = check_old_linux_ultra(header_line) if host_type == "linux" else -1
    for line in disks:
        one_disk = {}
        items = line.split()
        if host_type == "linux":
            disk_name = safe_get_split(items, 1)
            one_disk["lun_wwn"] = safe_get_split(items, 3 + index_handle)
            one_disk["lun_size"] = round(chang_unit2_gb(safe_get_split(items, 5 + index_handle)), 2)
        elif host_type == "aix":
            disk_name = safe_get_split(items, 2)
            one_disk["lun_wwn"] = safe_get_split(items, 4)
            one_disk["lun_size"] = round(chang_unit2_gb(safe_get_split(items, 7)), 2)
        elif host_type == "solaris":
            disk_name = safe_get_split(items, 2)
            one_disk["lun_wwn"] = safe_get_split(items, 3)
            one_disk["lun_size"] = round(chang_unit2_gb(safe_get_split(items, 5)), 2)
        else:
            disk_name = ""
        one_disk["disk_name"] = disk_name
        lun_id = items[0]
        one_disk["vendor"] = id_vp.get(lun_id, {}).get("vendor", "")
        one_disk["product"] = id_vp.get(lun_id, {}).get("product", "")
        if "aix" == host_type:
            one_disk["reserve_policy"] = \
                get_policy_by_disk_name(disk_name, context)
        result.append(one_disk)
    return result


def get_dmp_disk(context):
    """
    获取dmp多路径disk列表
    :param context: context
    :return: dmp多路径disk列表
    """
    result = []
    context["command"] = "vxdmpadm list dmpnode all"
    disk_info = get_return_from_txt(context)
    if not disk_info:
        return result

    disk_list = []
    all_disk_list = []
    for line in disk_info:
        if "not found" in line or "No such file or directory" in line:
            return []
        if line.strip():
            disk_list.append(line)
            continue
        if disk_list:
            all_disk_list.append(disk_list)
        disk_list = []

    for disk_list in all_disk_list:
        one_disk = {}
        for one_disk_info in disk_list:
            one_disk_list = one_disk_info.split("=")
            keyword = one_disk_list[0].strip().lower()
            value = one_disk_list[1].strip()
            if "dmpdev" == keyword:
                one_disk["disk_name"] = value
                continue
            if "vid" == keyword:
                one_disk["vendor"] = value
                continue
            if "pid" == keyword:
                one_disk["product"] = value
                continue
            if "lun-sno" == keyword:
                one_disk["lun_wwn"] = value
                continue
            if "scsi3_vpd" == keyword and "-" != value:
                one_disk["lun_wwn"] = value
                continue
            one_disk["lun_size"] = ""
        result.append(one_disk)
    return result


def get_ultra_id_vp(disks, host_type, context):
    """
     获取ultra多路径的id 与verdor、product对应关系
    :param host_type: host_type
    :param context: context
    :param disks: 回显
    :return: id 与verdor、product对应关系
    """
    id_dic = {}
    for line in disks:
        vlun_id = line.split()[0]
        vp = get_ultra_vp_byid(vlun_id, host_type, context)
        id_dic[vlun_id] = vp
    return id_dic


def get_ultra_vp_byid(vlun_id, host_type, context):
    """
    通过id获取verdor、product
    :param host_type: host_type
    :param context: context
    :param vlun_id: vlun_id
    :return: verdor、product
    """
    result = {}
    if host_type == "linux":
        context["command"] = "upadmin show vlun id=%s type=all" % vlun_id
        vp_info = get_return_from_txt(context)
        if "Product Name" not in "".join(vp_info):
            context["command"] = "upadmin show vlun id=%s" % vlun_id
    elif host_type == "aix":
        context["command"] = "upadm show vlun id=%s" % vlun_id
    elif host_type == "solaris":
        context["command"] = "upadm show vlun id=%s type=all" % vlun_id
    vp_info = get_return_from_txt(context)
    for line in vp_info:
        if "Product Name" in line:
            result["product"] = line.split(":")[1].strip()
        if "Vendor Name" in line:
            result["vendor"] = line.split(":")[1].strip()
    return result


def get_ultra_version(context, host_type):
    """
    获取ultra 多路径版本
    :param host_type: host_type
    :param context: 上下文
    :return: 版本
    """
    if host_type == HOST_TYPE_LINUX:
        context["command"] = "upadmin show version"
    else:
        context["command"] = "upadm show version"
    ultra_multi = get_return_from_txt(context)
    for line in ultra_multi:
        if "software version" in line.lower():
            return line.split(":")[1].strip()
    return ""


def get_dmp_version(context):
    """
    获取solaris dmp多路径版本
    :param context: 上下文
    :return: dmp多路径版本
    """
    context["command"] = "pkginfo -l VRTSvxvm"
    dmp_multi = get_return_from_txt(context)
    for line in dmp_multi:
        if "VERSION" == line.split(":")[0].strip().upper():
            return line.split(":")[1].strip()
    return ""


def get_hdlm_multi(context, host_type):
    """
    获取hdlm多路径
    :param host_type:  host_type
    :param context: 上下文
    :return: hdlm多路径
    """
    logger = context.get("Logger")
    try:
        result = {}
        result["multipath_type"] = "HDLM"
        version = get_hdlm_version(context, host_type)
        result["version"] = version
        if not version:
            return False, result
        disks = get_hdlm_disks(context, host_type)
        if version and not disks:
            result["disk_list"] = get_null_disk_list()
            return True, result
        disk_list = get_hdlm_disk_list(disks, context, host_type)
        result["disk_list"] = disk_list
        return True, result
    except Exception:
        logger.error("parse error: %s" % str(traceback.format_exc()))
        return False, result


def get_hdlm_disk_list(disks, context, host_type):
    """
    获取  多路径的磁盘信息
    :param host_type: host_type
    :param context: context
    :param disks: 上下文
    :return: disk_list
    """
    disk_list = []
    if HOST_TYPE_LINUX == host_type:
        name_size = get_disk_size_linux(context)
    indicates = []
    for disk in disks:
        one_disk = {}
        ilu = disk.split()[5]
        hdev_name = disk.split()[-1]
        indicate = ilu + hdev_name
        # 每个LUN是多条链路，列iLU与HDevName组合为唯一标识符，
        # 所以列iLU与HDevName组合是存在重复项的，过滤掉重复项后再分析
        if indicate in indicates:
            continue
        disk_name = disk.split()[-1]
        one_disk["disk_name"] = disk_name
        if HOST_TYPE_LINUX == host_type:
            one_disk["lun_size"] = get_size_by_name(disk_name, name_size)
        elif HOST_TYPE_AIX == host_type:
            one_disk["lun_size"] = get_size_by_disk_name(disk_name, context)
            one_disk["reserve_policy"] = \
                get_policy_by_disk_name(disk_name, context)
        one_disk["lun_wwn"] = ilu
        one_disk["vendor"] = disk.split()[2]
        one_disk["product"] = disk.split()[3][1:]
        indicates.append(indicate)
        disk_list.append(one_disk)
    return disk_list


def get_hdlm_disks(context, host_type):
    """
    组装成disk 列表
    :param host_type: host_type
    :param context: 上下文
    :return: disk 列表
    """
    context["command"] = "dlnkmgr view -path"
    disk_info = get_return_from_txt(context)
    disks = get_hdlm_disks_from_cli(disk_info)
    if disks:
        return disks
    # 日立多路径，环境变量需要人为设置，防止没有设置环境变量
    if host_type == "linux":
        context["command"] = "/opt/DynamicLinkManager/bin/dlnkmgr view -path"
    elif host_type == "aix":
        context["command"] = "/usr/DynamicLinkManager/bin/dlnkmgr view -path"
    disk_info = get_return_from_txt(context)
    disks = get_hdlm_disks_from_cli(disk_info)
    return disks


def get_hdlm_disks_from_cli(disk_info):
    """
    解析日立多路径回文
    :param disk_info: 回文
    :return:
    """
    flag = False
    disks = []
    for line in disk_info:
        if "PathID" in line:
            flag = True
            continue
        if "The HDLM command completed normally" in line:
            break
        if flag:
            disks.append(line)
    return disks


def get_hdlm_version(context, host_type):
    """
    获取 hdlm 的版本
    :param host_type: host_type
    :param context: 上下文
    :return: hdlm 版本
    """
    version = ""
    context["command"] = "dlnkmgr view -sys"
    version_info = get_return_from_txt(context)
    for line in version_info:
        if "HDLM Version" in line:
            return line.split(":")[1].strip()
    # 日立多路径，环境变量需要人为设置，防止没有设置环境变量
    if host_type == "linux":
        context["command"] = "/opt/DynamicLinkManager/bin/dlnkmgr view -sys"
    elif host_type == "aix":
        context["command"] = "/usr/DynamicLinkManager/bin/dlnkmgr view -sys"
    version_info = get_return_from_txt(context)
    for line in version_info:
        if "HDLM Version" in line:
            return line.split(":")[1].strip()
    return version


def get_power_multi(context, host_type):
    """
    获取power多路径
    :param host_type: host_type
    :param context: 上下文
    :return: power多路径
    """
    logger = context.get("Logger")
    result = {}
    try:
        result["multipath_type"] = "PowerPath"
        version = get_powerpath_version(context)
        result["version"] = version
        if not version:
            return False, result
        disks = get_power_disks(context)
        if version and not disks:
            result["disk_list"] = get_null_disk_list()
            return True, result
        disk_list = get_power_disk_list(disks, context, host_type)
        result["disk_list"] = disk_list
        return True, result
    except Exception:
        logger.error("parse error: %s" % str(traceback.format_exc()))
        return False, result


def get_power_disk_list(disks, context, host_type):
    """
    获取 power 多路径的磁盘信息
    :param host_type: host_type
    :param context: 上下文
    :param disks: 磁盘信息
    :return: disk_list
    """
    disk_size_dict = get_disk_size_linux(context) if "linux" in host_type else {}
    disk_list = []
    for disk in disks:
        one_disk = {}
        prduct_flag = False
        for line in disk:
            if "Pseudo name" in line:
                disk_name = line.split("=")[1]
                one_disk["disk_name"] = disk_name
                if "linux" == host_type:
                    one_disk["lun_size"] = \
                        get_size_by_name(disk_name, disk_size_dict)
                elif "aix" == host_type:
                    one_disk["lun_size"] = \
                        get_size_by_disk_name(disk_name, context)
                    one_disk["reserve_policy"] = \
                        get_policy_by_disk_name(disk_name, context)
                elif "hpux" == host_type:
                    one_disk["lun_size"] = \
                        get_size_by_disk_name_hpux(disk_name, context)
                elif "solaris" == host_type:
                    one_disk["lun_size"] = \
                        get_size_by_disk_name_solaris_power(disk, context)
                one_disk["vendor"] = "EMC"
                prduct_flag = True
                continue
            if prduct_flag:
                one_disk["product"] = line.split()[0]
                prduct_flag = False
            if "Logical device ID" in line:
                one_disk["lun_wwn"] = line.split("=")[1].split()[0]
            if "Device WWN=" in line:
                one_disk["lun_wwn"] = line.split("=")[1].split()[0]
        disk_list.append(one_disk)
    return disk_list


def get_power_disks(context):
    """
    组装成disk 列表
    :param context: 上下文
    :return: disk 列表
    """
    context["command"] = "powermt display dev=all"
    disk_lines = get_return_from_txt(context)
    disk_lists = []
    one_disk = []
    flag = False
    for index, line in enumerate(disk_lines):
        if "Pseudo name" in line:
            flag = True
        if "Pseudo name" in line and one_disk:
            disk_lists.append(one_disk)
            one_disk = []
        if flag:
            one_disk.append(line)
        if index == len(disk_lines) - 1 and flag:
            disk_lists.append(one_disk)
    return disk_lists


def get_powerpath_version(context):
    """
    获取 powerpath 的版本
    :param context: 上下文
    :return: powerpath版本
    """
    version = ""
    context["command"] = "powermt version"
    version_info = get_return_from_txt(context)
    for line in version_info:
        if "Version" in line:
            version = line.split("Version")[1]
    return version


def get_disk_name_size_linux(line, logger):
    """
    获取 disk_name、disk_size(Linux主机)
    :param line: 数据行
    :param logger: 日志打印器
    :return: 字典
    """
    try:
        disk_name = line.split()[1][:-1]
        disk_size = line.split(",")[1].strip().split()[0] + " Bytes"
        return disk_name, round(chang_unit2_gb(disk_size), 2)
    except Exception:
        logger.error("parse disk size from fdisk error: %s" % str(traceback.format_exc()))
        return "", ""


def get_disk_size_linux(context):
    """
    获取 disk与size对应的字典(Linux主机)
    :param context: 上下文
    :return: 字典
    """
    context["command"] = "fdisk -l"
    disk_infos = get_return_from_txt(context)
    name_size = {}
    logger = context.get("Logger")
    for line in disk_infos:
        if "Disk" in line and "bytes" in line:
            disk_name, disk_size = get_disk_name_size_linux(line, logger)
            name_size[disk_name] = disk_size
    return name_size


def get_size_by_name(disk_name, name_size):
    """
    通过disk_name 获取容量
    :param disk_name: disk_name
    :param name_size: name_size
    :return: 容量
    """
    disk_names = name_size.keys()
    for name in disk_names:
        if name == "/dev/" + disk_name:
            return name_size.get(name, "")
    return ""


def get_null_disk_list():
    """
    生成空disk数据
    :return: 空数据
    """
    disk_list = []
    one_disk = {}
    one_disk["disk_name"] = ""
    one_disk["lun_size"] = ""
    one_disk["lun_wwn"] = ""
    one_disk["vendor"] = ""
    one_disk["product"] = ""
    one_disk["reserve_policy"] = ""
    disk_list.append(one_disk)
    return disk_list


def get_size_by_disk_name(disk_name, context):
    """
    aix主机通过disk_name 获取size大小
    :param context: context
    :param disk_name: disk_name
    :return: size大小
    """
    try:
        context["command"] = "bootinfo -s " + disk_name
        size = get_return_from_txt(context)[0] + "MiB"
        return round(chang_unit2_gb(size), 2) if size != 'MiB' else 'NA'
    except Exception:
        logger = context.get("Logger")
        logger.error("parse error: %s" % str(traceback.format_exc()))
        return ""


def get_size_by_disk_name_hpux(disk_name, context):
    """
    hpux主机通过disk_name 获取size大小
    :param context: context
    :param disk_name: disk_name
    :return: size大小
    """
    context["command"] = "diskinfo /dev/rdisk/" + disk_name
    size_info = get_return_from_txt(context)
    size = ""
    for line in size_info:
        if "size" in line and ":" in line:
            size_str = line.split(":")[-1]
            size = size_str.strip().split()[0] + change_unit2stand(
                size_str.strip().split()[-1])
            break
    return str(round(chang_unit2_gb(size), 2))


def get_size_by_disk_name_solaris_power(disk, context):
    """
    solaris主机通过disk_name 获取size大小(power多路径)
    :param context: 上下文
    :param disk: 回文
    :return: size大小
    """
    for line in disk:
        if len(line.split()) > 3 and line.split()[2].startswith("c"):
            context["command"] = "luxadm display /dev/rdsk/" + line.split()[2]
            size_info = get_return_from_txt(context)
            for size_line in size_info:
                if "unformatted capacity" in size_line.lower() \
                        and ":" in size_line:
                    size_str = size_line.split(":")[-1]. \
                        strip().lower().replace("mbytes", "mb")
                    return chang_unit2_gb(size_str)
    return "0"


def get_policy_by_disk_name(disk_name, context):
    """
    aix主机通过disk_name 获取reserve policy
    :param context: context
    :param disk_name: disk_name
    :return: size大小
    """
    context["command"] = "lsattr -El " + disk_name
    policy_info = get_return_from_txt(context)
    for line in policy_info:
        if "reserve_policy" in line:
            return line.split()[1]
    return ""


def get_disk_rule(context, host_type):
    """
    磁盘绑定方式
    :param host_type: host_type
    :param context: 上下文
    :return: 绑定方式
    """
    if "aix" == host_type:
        return ""
    context["command"] = "oracleasm listdisks |xargs oracleasm querydisk -p"
    asmlib_info = get_return_from_txt(context)
    for line in asmlib_info:
        if "command not found" in line:
            break
        if "LABEL" in line and "TYPE" in line:
            return "ASMlib"

    udev_list = get_udev_result(context)
    if udev_list:
        return "udev"
    return ""


def get_oracle_cluster_result(context, host_type):
    """
    获取数据结果
    :param host_type: host_type
    :param context:上下文
    :return: 数据结果
    """
    result = {}
    logger = context.get("Logger")
    result["host_ip"] = context.get("ip")
    try:
        oracle_version = get_oracle_version(context)
        asm_diskstring_value = get_asm_diskstring_value(context)
        result["oracle_version"] = oracle_version
        result["asm_diskstring_value"] = asm_diskstring_value
        if "linux" == host_type:
            oracleasm_scanorder, oracleasm_scanexclude = \
                get_oracleasm_result(context)
            result["oracleasm_scanorder"] = oracleasm_scanorder
            result["oracleasm_scanexclude"] = oracleasm_scanexclude
        disk_info = get_oracle_disk_info(context)
        result["disk_rule"] = ""
        if not oracle_version or not disk_info:
            disk_list = get_null_obj_oracle_cluster()
            result["disk_list"] = disk_list
            return result
        result["disk_rule"] = get_disk_rule(context, host_type)

        result = get_oracle_disk_list(result, disk_info, host_type, context)
    except Exception:
        logger.error("parse error: %s" % str(traceback.format_exc()))
        result["disk_list"] = get_null_obj_oracle_cluster()
    return result


def get_pure_oracle_cluster_result(context):
    """
    获取纯oracle数据结果
    :param context:上下文
    :return: 数据结果
    """
    logger = context.get("Logger")
    result = {}
    host_ip = context.get("ip")
    oracle_version = get_oracle_version(context)
    asm_diskstring_value = get_asm_diskstring_value(context)
    result["host_ip"] = host_ip
    result["oracle_version"] = oracle_version
    result["asm_diskstring_value"] = asm_diskstring_value
    try:
        disk_info = get_oracle_disk_info(context)
        if not oracle_version or not disk_info:
            result["disk_list"] = get_null_obj_oracle_cluster()
            return result
        result = get_oracle_disk_list(result, disk_info, "", context)
    except Exception:
        logger.error("parse error: %s" % str(traceback.format_exc()))
        result["disk_list"] = get_null_obj_oracle_cluster()
    return result


def get_oracle_disk_list(result, disk_info, host_type, context):
    """
    获取oracle集群的disk_list
    :param host_type: host_type
    :param result: 结果集
    :param disk_info: diskInfo
    :param context: 上下文
    :return: 结果集
    """
    # 排除掉字段不为9个的行
    disk_list = []
    udev_list = []
    path_wwn = {}
    if result.get("disk_rule", "") == "udev":
        udev_list = get_udev_result(context)
    elif result.get("disk_rule", "") == "ASMlib":
        path_wwn = get_path_wwn(disk_info, host_type, context)
    for line in disk_info:
        one_disk = {"asmdisk_name": ""}
        if len(line.split()) == 10:
            one_disk["asmdisk_name"] = line.split()[2]
        one_disk["lun_wwn"] = ""
        one_disk["user"] = ""
        one_disk["group"] = ""
        one_disk["asmdisk_path"] = line.split()[-5]
        one_disk["asmdisk_state"] = line.split()[-4]
        one_disk["asmdisk_total_size"] = line.split()[-3]
        one_disk["asmdisk_free_size"] = line.split()[-2]
        one_disk["asmdisk_sector_size"] = line.split()[-1]
        if result.get("disk_rule", "") == "udev":
            one_disk = get_oracle_disk_udev_info(one_disk, udev_list)
        elif result.get("disk_rule", "") == "ASMlib":
            user, group = get_user_group_asmlib(context)
            one_disk["user"] = user
            one_disk["group"] = group
            one_disk["lun_wwn"] = \
                path_wwn.get(one_disk.get("asmdisk_path", ""), "")
        elif not result.get("disk_rule", "") and host_type == "aix":
            one_disk = get_aix_oracle_disk_info(one_disk, context)
        disk_list.append(one_disk)
    result["disk_list"] = disk_list
    return result


def get_aix_oracle_disk_info(one_disk, context):
    """
    组装aix的oracle——disk数据
    :param context: context
    :param one_disk: disk
    :return: one_disk
    """
    flag = False
    disk_path = one_disk.get("asmdisk_path", "")
    if disk_path.startswith("/dev/rhdisk") \
            or disk_path.startswith("/dev/rhdiskpower"):
        flag, one_disk = get_aix_disk_with_multi(one_disk, context)
    if not flag:
        one_disk["disk_name"] = ""
        one_disk["permission"] = ""
        one_disk["major"] = ""
        one_disk["minor"] = ""
    return one_disk


def get_aix_disk_with_multi(one_disk, context):
    """
    aix主机有多路径时，获取permiss、user、group等数据的方法
    :param one_disk: one_disk
    :param context: 上下文
    :return: one_disk
    """
    context["command"] = "ls -al /dev/*"
    dev_info = get_return_from_txt(context)
    for line in dev_info:
        if one_disk.get("asmdisk_path", "").endswith(line.split()[-1]):
            one_disk["disk_name"] = line.split()[-1]
            one_disk["permission"] = line.split()[0]
            one_disk["user"] = line.split()[2]
            one_disk["group"] = line.split()[3]
            one_disk["major"] = line.split()[4][:-1]
            one_disk["minor"] = line.split()[5]
            return True, one_disk
    return False, one_disk


def get_path_wwn(disk_info, host_type, context):
    """
    asmlib绑定时，获取path-wwn 字典
    :param host_type: host_type
    :param disk_info:disk_info
    :param context: 上下文
    :return: path-wwn 字典
    """
    context["command"] = "oracleasm listdisks |xargs oracleasm querydisk -p"
    asm_disk_info = get_return_from_txt(context)
    path_wwn = {}
    disk_list = get_multi_oracle(host_type, context)
    for line in disk_info:
        path = line.split()[-4]
        path_end = path.split("/")[-1]
        lun_wwn = ""
        for disk_line in asm_disk_info:
            if "LABEL" in disk_line and path_end in disk_line:
                multi_disk = disk_line.split()[0][:-1].split("/")[-1]
                lun_wwn = get_asmlib_wwn(multi_disk, disk_list)
        path_wwn[path] = lun_wwn
    return path_wwn


def get_asmlib_wwn(multi_disk, disk_list):
    """
    asmlib绑定，获取wwn
    :param multi_disk: 多路径中oracle磁盘数据
    :param disk_list: context
    :return: wwn
    """
    disk_name = re.split("\\d+$", multi_disk)[0]
    for disk in disk_list:
        if disk_name.strip() == disk.get("disk_name", "").strip():
            return disk.get("lun_wwn", "")
    return ""


def get_multi_oracle(host_type, context):
    """
    获取多路径信息
    :param host_type: 主机类型
    :param context: 上下文
    :return: 多路径信息
    """
    power_flag, power_multi = get_power_multi(context, host_type)
    ultra_flag, ultra_multi = get_ultra_multi(context, host_type)
    hdlm_flag, hdlm_multi = get_hdlm_multi(context, host_type)
    dm_flag, dm_multi = False, {}
    if host_type == "linux":
        dm_flag, dm_multi = get_dm_multi(context)
    if hdlm_flag:
        return hdlm_multi.get("disk_list", "")
    elif power_flag:
        return power_multi.get("disk_list", "")
    elif ultra_flag:
        return ultra_multi.get("disk_list", "")
    elif dm_flag:
        return dm_multi.get("disk_list", "")


def get_user_group_asmlib(context):
    """
    asmlib绑定时，获取user/group
    :param context: 上下文
    :return: user/group
    """
    context["command"] = "cat /etc/sysconfig/oracleasm"
    oracleasm_info = get_return_from_txt(context)
    user = ""
    group = ""
    for line in oracleasm_info:
        if not line.startswith("#") and "ORACLEASM_UID" in line:
            user = line.split("=")[1]
        if not line.startswith("#") and "ORACLEASM_GID" in line:
            group = line.split("=")[1]
    return user, group


def get_oracle_disk_udev_info(one_disk, udev_list):
    """
    如果是udev绑定，补充disk信息
    :param udev_list: udev_list
    :param one_disk: one_disk
    :return: one_disk
    """
    for udev in udev_list:
        if one_disk.get("asmdisk_path", "").endswith(
                udev.get("asmdisk_name", "")):
            one_disk["lun_wwn"] = udev.get("result", "")
            one_disk["user"] = udev.get("udev_owner", "")
            one_disk["group"] = udev.get("udev_group", "")
            return one_disk
    return one_disk


def get_asm_diskstring_value(context):
    """
    获取发现磁盘路径
    :param context:上下文
    :return:发现磁盘路径
    """
    context["command"] = "show parameter asm_diskstring"
    command_result = get_return_from_txt(context)
    value = ""
    flag = False
    for line in command_result:
        if "-------" in line:
            flag = True
            continue
        if flag and len(line.split()) > 2:
            value = line.split()[2]
            break
    return value


def get_oracleasm_result(context):
    """
    获取 oracleasm 数据结果
    :param context: 上下文
    :return: oracleasm 数据结果
    """
    command = "cat /etc/sysconfig/oracleasm"
    context["command"] = command
    oracleasm_scanorder = ""
    oracleasm_scanexclude = ""
    oracleasm_info = get_return_from_txt(context)
    for line in oracleasm_info:
        if "ORACLEASM_SCANORDER" in line and "=" in line:
            oracleasm_scanorder = line.split("=")[1].strip('"')
        if "ORACLEASM_SCANEXCLUDE" in line and "=" in line:
            oracleasm_scanexclude = line.split("=")[1].strip('"')
    return oracleasm_scanorder, oracleasm_scanexclude


def get_null_obj_oracle_cluster():
    """
    生成空数据
    :return: 空数据
    """
    disk_list = []
    one_disk = {}
    one_disk["asmdisk_name"] = ""
    one_disk["asmdisk_path"] = ""
    one_disk["asmdisk_state"] = ""
    one_disk["asmdisk_total_size"] = ""
    one_disk["asmdisk_free_size"] = ""
    one_disk["disk_rule"] = ""
    one_disk["disk_name"] = ""
    one_disk["lun_wwn"] = ""
    one_disk["user"] = ""
    one_disk["group"] = ""
    one_disk["asm_diskstring_value"] = ""
    one_disk["major"] = ""
    one_disk["minor"] = ""
    one_disk["permission"] = ""
    disk_list.append(one_disk)
    return disk_list


def get_dm_multi(context):
    """
    获取dm 多路径
    :param context:上下文
    :return: dm 多路径
    """
    logger = context.get("Logger")
    result = {}
    try:
        multipath_type = "DM-Multipath"
        result["multipath_type"] = multipath_type
        result["version"] = "NA"
        disks = get_dm_disks(context)
        if not disks:
            return False, result

        disk_list = get_dm_disk_list(context, disks)
        result["disk_list"] = disk_list
        return True, result
    except Exception:
        logger.error("parse error: %s" % str(traceback.format_exc()))
        return False, result


def get_dm_disks(context):
    """
    获取dm的disks
    :param context: 上下文
    :return: disks 回显
    """
    disk_lists = []
    one_disk = []
    context["command"] = "multipath -ll|cat"
    disk_info = get_return_from_txt(context)
    # 将全部回显分割为单个的disk 进行存储
    flag = False
    for index, line in enumerate(disk_info):
        if "dm-" in line:
            flag = True
        if "dm-" in line and one_disk:
            disk_lists.append(one_disk)
            one_disk = []
        if flag:
            one_disk.append(line)
        if flag and index == len(disk_info) - 1:
            disk_lists.append(one_disk)
    return disk_lists


def get_dm_disk_list(context, disks):
    """
    获取dm disk_list
    :param context: 上下文
    :param disks: 回显
    :return: disk_list
    """
    result_list = []
    for disk_lines in disks:
        disk, dm_name = {}, ""
        for line in disk_lines:
            if "dm-" in line:
                # 这里需要进行容错：
                # （1）第一个字段不存在，disk_name 就要使用 dm- 这个字段
                # （2）逗号后可能空格
                before_line = line.split("dm-")[0]
                if len(before_line.split()) == 2:
                    before_product = line.split(",")[0]
                    disk["product"] = get_split_item(line.split(","), 1)
                    before_product_list = before_product.strip().split()
                    disk["disk_name"] = before_product_list[0]
                    disk["lun_wwn"] = deal_dm_lun_wwn(before_product_list[1])
                    disk["vendor"] = get_split_item(before_product_list, 3)
                    dm_name = before_product_list[2]
                else:
                    before_product = line.split(",")[0]
                    disk["product"] = line.split(",")[1].strip()
                    before_product_list = before_product.strip().split()
                    disk["lun_wwn"] = deal_dm_lun_wwn(before_product_list[0])
                    disk["disk_name"] = before_product_list[1]
                    disk["vendor"] = before_product_list[-1]
                    dm_name = disk["disk_name"]
            if "size=" in line:
                size = line.split()[0].split("size=")[1]
                if "]" in size:
                    size = size.split("]")[0]
                disk["lun_size"] = chang_unit2_gb(size)
        disk["max_sectors_kb"] = get_disk_sector(context, dm_name)
        result_list.append(disk)
    return result_list


def get_disk_sector(context, dm_name):
    """
    获取dm盘的扇区大小
    :param context: 上下文
    :param dm_name: dm盘名字
    :return: 扇区大小
    """
    context["command"] = "cat /sys/block/%s/queue/max_sectors_kb" % dm_name
    disk_info = get_return_from_txt(context)
    for line in disk_info:
        line = line.strip()
        if line.isdigit():
            return line
    return ""


def deal_dm_lun_wwn(lun_wwn):
    """
    处理linux主机的lun_wwn数据
    :param lun_wwn: lun_wwn
    :return: 处理后数据
    """
    wwn = lun_wwn.strip().rstrip(")").lstrip("(")
    if wwn.startswith("3") or wwn.startswith("2"):
        return wwn[1:]
    return wwn


def get_hp_nmp_multi(context):
    """
    获取hpux nmp 多路径
    :param context:上下文
    :return: dm 多路径
    """
    logger = context.get("Logger")
    try:
        result = {}
        multipath_type = "NMP"
        result["multipath_type"] = multipath_type
        result["version"] = "NA"
        disk_list = get_nmp_disk_list(context)
        if not disk_list:
            return False, result
        result["disk_list"] = disk_list
        return True, result
    except Exception:
        logger.error("parse error: %s" % str(traceback.format_exc()))
        return False, result


def get_nmp_disk_list(context):
    """
    获取hpux  nmp disk_list
    :param context: context
    :return: disk_list
    """
    result_list = []
    context["command"] = "scsimgr -p get_attr all_lun -a device_file -a wwid"
    disk_info = get_return_from_txt(context)
    for line in disk_info:
        if "/dev/rdisk/" in line:
            disk = {}
            splits = line.split(":")
            disk["disk_name"] = splits[0]
            disk["lun_wwn"] = splits[-1][2:]
            vendor, product, size = get_hp_nmp_vps(splits[0], context)
            disk["vendor"] = vendor
            disk["product"] = product
            disk["lun_size"] = size
            result_list.append(disk)
    return result_list


def get_hp_nmp_vps(disk_name_path, context):
    """
    hpux主机通过disk_name 获取vendor,product,size
    :param context: context
    :param disk_name_path: disk_name_path
    :return: vendor,product,size
    """
    context["command"] = "diskinfo " + disk_name_path
    vps_info = get_return_from_txt(context)
    vendor = ""
    product = ""
    size = ""
    for line in vps_info:
        if "vendor" in line and ":" in line:
            vendor = line.split(":")[-1].strip()
            continue
        if "product id" in line and ":" in line:
            product = line.split(":")[-1].strip()
            continue
        if "size" in line and ":" in line:
            size = line.split(":")[-1].strip().split()[0] + " Kb"
            continue
    return vendor, product, round(chang_unit2_gb(size), 2)


def get_vm_nmp_multi(context):
    """
    获取VMware nmp 多路径
    :param context:上下文
    :return: dm 多路径
    """
    logger = context.get("Logger")
    try:
        result = {}
        multipath_type = "NMP"
        result["multipath_type"] = multipath_type
        result["version"] = "NA"
        disk_list = get_vm_multi_disk_list(context, "NMP")
        if not disk_list:
            return False, result
        result["disk_list"] = disk_list
        return True, result
    except Exception:
        logger.error("parse error: %s" % str(traceback.format_exc()))
        return False, result


def get_vm_ultra_multi(context):
    """
    获取VMware UltraPath 多路径
    :param context:上下文
    :return: UltraPath 多路径
    """
    logger = context.get("Logger")
    try:
        result = {}
        multipath_type = "UltraPath"
        result["multipath_type"] = multipath_type
        version = get_vm_ultra_version(context)
        if not version:
            return False, result
        result["version"] = version
        disk_list = get_vm_multi_disk_list(context, "MP_PLUGIN_ULTRAPATH")
        if not disk_list:
            disk_list = get_empty_disk_list()
        result["disk_list"] = disk_list
        return True, result
    except Exception:
        logger.error("parse error: %s" % str(traceback.format_exc()))
        return False, result


def get_vm_other_multis(context):
    """
    获取VMware UltraPath 多路径
    :param context:上下文
    :return: UltraPath 多路径
    """
    logger = context.get("Logger")
    try:

        disk_lists = get_vm_multi_disk_list(context, "other")
        if not disk_lists:
            return False, []
        multipath_types = get_multi_path_types(disk_lists)
        results = get_multi_path(multipath_types, disk_lists)
        if results:
            return True, results
        return False, results
    except Exception:
        logger.error("parse error: %s" % str(traceback.format_exc()))
        return False, []


def get_multi_path(multipath_types, disk_lists):
    """
    获取多路径
    :param multipath_types: 多路径类型
    :param disk_lists: 磁盘
    :return: 多路径
    """
    results = []
    for m_type in multipath_types:
        result = {}
        disk_list = []
        result["multipath_type"] = m_type
        result["version"] = "NA"
        for disk in disk_lists:
            if m_type.strip() == disk.get("multipath_plugin", "").strip():
                disk_list.append(disk)
        if disk_list:
            result["disk_list"] = disk_list
            results.append(result)
    return results


def get_multi_path_types(disk_lists):
    """
    获取多路径类型
    :param disk_lists: 磁盘
    :return: 多路径类型
    """
    multipath_types = []
    for disk in disk_lists:
        multipath_plugin = disk.get("multipath_plugin", "")
        if multipath_plugin and multipath_plugin not in multipath_types:
            multipath_types.append(multipath_plugin)
    return multipath_types


def get_vm_ultra_version(context):
    """
    获取vm ultra的版本
    :param context: context
    :return: vm ultra的版本
    """
    context["command"] = "esxcli upadm show version"
    ultra_version_info = get_return_from_txt(context)
    version = ""
    for line in ultra_version_info:
        if "Software Version" in line:
            version = line.split(":")[-1].strip()
    return version


def get_vm_multi_disk_list(context, multi_type):
    """
    获取VMware multi disk_list
    :param context: context
    :return: disk_list
    """
    result_list = []
    lun_list = get_vm_lun_list(context)
    multi_lun_list = get_nmp_lun_list(lun_list, multi_type)
    host_lun_ids = get_nmp_lun_host_lun_ids(context)
    lun_dic_list = get_vm_lun_dic_list(multi_lun_list)
    lun_plugin_list = get_vm_lun_plugin_list(context)
    lun_datastore_list = get_vm_lun_datastore_list(context)
    lun_array_path_list = get_vm_lun_array_path_list(context)
    vm_list = get_vm_host_info(context)
    for lun in lun_dic_list:
        disk = get_one_empty_vm_disk()
        disk["lun_wwn"] = lun.get("lun_wwn", "")
        disk["host_lun_id"] = host_lun_ids.get(disk["lun_wwn"], "")
        disk["lun_size"] = lun.get("lun_size", "")
        disk["vendor"] = lun.get("vendor", "")
        disk["product"] = lun.get("product", "")
        disk["vaai_status"] = lun.get("vaai_status", "")
        disk["boot_device"] = lun.get("boot_device", "")
        disk["multipath_plugin"] = lun.get("multipath_plugin", "")
        disk["other_uid"] = lun.get("other_uid", "")
        disk["reservation_key"] = get_reservation_key(context, lun.get("devfs_path", ""))
        for lun_plugin in lun_plugin_list:
            if disk["lun_wwn"] == lun_plugin.get("lun_wwn", ""):
                disk["vaai_plugin_name"] = \
                    lun_plugin.get("vaai_plugin_name", "")
                break
        for datastore in lun_datastore_list:
            if disk["lun_wwn"] == datastore.get("lun_wwn", ""):
                disk["datastore_name"] = datastore.get("datastore_name", "")
                disk["datastore_type"] = datastore.get("datastore_type", "")
                break
        for array_path in lun_array_path_list:
            if disk["lun_wwn"] == array_path.get("lun_wwn", ""):
                disk["path_policy"] = array_path.get("path_policy", "")
                disk["storage_array_type"] = \
                    array_path.get("storage_array_type", "")
                break
        set_disk_rdm(disk, vm_list)
        result_list.append(disk)
    return result_list


def get_reservation_key(context, path):
    """
    获取预留数据
    :param context: context
    :param path: 磁盘路径
    :return: 数据
    """
    context["command"] = "vmkfstools -L readresv %s" % path
    data_info = get_return_from_txt(context)
    key_word = "Reservation Key :"
    for line in data_info:
        if line.startswith(key_word):
            return line.replace(key_word, "").strip()
    return ""


def get_nmp_lun_host_lun_ids(context):
    context["command"] = "esxcli storage core path list"
    data_lines = get_return_from_txt(context)
    lun_wwn = ''
    host_lun_ids = {}
    for line in data_lines:
        if line.startswith('Device:') and 'naa.' in line:
            lun_wwn = line.split("naa.")[-1].strip()
        if line.startswith('LUN:') and lun_wwn:
            host_lun_ids[lun_wwn] = line.split(':')[-1].strip()
    return host_lun_ids


def get_nmp_lun_list(lun_list, multi_type):
    """
    过滤VMware的multi_type lun
    :param lun_list: lun_list
    :return: multi_type lun
    """
    multi_type_lun_list = []
    for lun in lun_list:
        multi_type_flag = False
        for line in lun:
            if line.strip().startswith("Multipath Plugin") \
                    and multi_type != "other" \
                    and line.strip().endswith(multi_type):
                multi_type_flag = True
                break
            elif line.strip().startswith("Multipath Plugin") \
                    and multi_type == "other" \
                    and not line.strip().endswith("NMP") \
                    and not line.strip().endswith("MP_PLUGIN_ULTRAPATH"):
                multi_type_flag = True
                break
        if multi_type_flag:
            multi_type_lun_list.append(lun)
    return multi_type_lun_list


def set_disk_rdm(disk, vm_list):
    """
    判断是否为RDM
    :param disk: disk
    :param vm_list: 虚拟机
    :return: pass
    """
    for vm in vm_list:
        controller_disk_list = vm.get("controller_disk_list", [])
        for controller_disk in controller_disk_list:
            if disk.get("other_uid", "").strip() and \
                    disk.get("other_uid", "") == \
                    controller_disk.get("raw_lun_id", ""):
                disk["datastore_type"] = "RDM"
                disk["compatibility_mode"] = \
                    controller_disk.get("compatibility_mode", "")
                return


def get_vm_lun_array_path_list(context):
    """
    获取lun与Storage Array Type、Path Selection Policy映射
    :param context: context
    :return: lun与Storage Array Type、Path Selection Policy映射
    """
    lun_array_path_list = []
    context["command"] = "esxcli storage nmp device list"
    lun_array_path = get_return_from_txt(context)
    one_lun_array_path = {}
    for line in lun_array_path:
        if line.strip().lower().startswith("naa."):
            one_lun_array_path["lun_wwn"] = line.split(".")[-1].strip()
            continue
        if line.strip().lower().startswith("storage array type:"):
            one_lun_array_path["storage_array_type"] = \
                line.split(":")[-1].strip()
            continue
        if line.strip().lower().startswith("path selection policy:"):
            one_lun_array_path["path_policy"] = line.split(":")[-1].strip()
            lun_array_path_list.append(one_lun_array_path)
            one_lun_array_path = {}
            continue
    return lun_array_path_list


def get_vm_lun_datastore_list(context):
    """
    获取lun与datastore的映射
    :param context: context
    :return: lun与datastore的映射
    """
    datastore_list = get_datastore_list(context)
    lun_datastore_list = []
    context["command"] = "esxcli storage vmfs extent list"
    lun_datastore = get_return_from_txt(context)
    flag = False
    for line in lun_datastore:
        items = re.split('\\s{2,}', line)
        if "---" in line and len(items) == 5:
            flag = True
            continue
        if flag and len(items) == 5:
            one_lun_datastore = {}
            lun_wwn = items[-2].split(".")[-1].strip()
            datastore_name = items[0].strip()
            one_lun_datastore["lun_wwn"] = lun_wwn
            one_lun_datastore["datastore_name"] = datastore_name
            for datastore in datastore_list:
                if datastore_name == datastore.get("datastore_name"):
                    one_lun_datastore["datastore_type"] = \
                        datastore.get("datastore_type")
                    break
            lun_datastore_list.append(one_lun_datastore)
    return lun_datastore_list


def get_datastore_list(context):
    """
    获取datastore的映射
    :param context: context
    :return: datastore的映射
    """
    datastore_list = []
    context["command"] = "esxcli storage filesystem list"
    datastore = get_return_from_txt(context)
    flag = False
    for line in datastore:
        items = re.split('\\s{2,}', line)
        if "---" in line and len(items) == 7:
            flag = True
            continue
        if flag and len(items) == 7:
            one_datastore = {}
            datastore_name = items[1]
            datastore_type = items[-3]
            one_datastore["datastore_name"] = datastore_name
            one_datastore["datastore_type"] = datastore_type
            datastore_list.append(one_datastore)
    return datastore_list


def get_vm_lun_plugin_list(context):
    """
    获取lun与plugin name的映射
    :param context: context
    :return: lun与plugin name的映射
    """
    lun_plugin_list = []
    context["command"] = "esxcli storage core device vaai status get"
    lun_plugin = get_return_from_txt(context)
    one_lun_plugin = {}
    for line in lun_plugin:
        if line.strip().lower().startswith("naa."):
            one_lun_plugin["lun_wwn"] = line.split(".")[-1].strip()
        if line.strip().lower().startswith("vaai plugin name"):
            one_lun_plugin["vaai_plugin_name"] = line.split(":")[-1].strip()
            lun_plugin_list.append(one_lun_plugin)
            one_lun_plugin = {}
    return lun_plugin_list


def parse_vm_lun_dict(lun):
    """
    解析vm的lun数据(第一行是lun wwn，不是标准的分隔符分割的)
    :param lun: 回文
    :return: lun_dic
    """
    lun_dict = {}
    separator = ":"
    for line in lun:
        if line.strip().startswith("naa.") or line.strip().startswith("eui."):
            lun_dict["lun_wwn"] = line.split(".")[-1].strip()
            continue
        items = line.split(separator)
        if len(items) > 1:
            lun_dict[items[0].strip()] = (separator.join(items[1:])).strip()
    return lun_dict


def get_vm_lun_dic_list(lun_list):
    """
    解析为lun_dic_list
    :param lun_list: lun_list
    :return: lun_dic_list
    """
    lun_dic_list = []
    for lun in lun_list:
        disk = {}
        lun_dict = parse_vm_lun_dict(lun)
        disk["lun_wwn"] = lun_dict.get("lun_wwn", "")
        disk["lun_size"] = round(chang_unit2_gb(lun_dict.get("Size", "") + " Mib"), 2)
        disk["vendor"] = lun_dict.get("Vendor", "")
        disk["product"] = lun_dict.get("Model", "")
        disk["vaai_status"] = lun_dict.get("VAAI Status", "")
        disk["boot_device"] = lun_dict.get("Is Boot Device", "")
        disk["other_uid"] = lun_dict.get("Other UIDs", "")
        disk["multipath_plugin"] = lun_dict.get("Multipath Plugin", "")
        disk["devfs_path"] = lun_dict.get("Devfs Path", "")
        lun_dic_list.append(disk)
    return lun_dic_list


def get_vm_lun_list(context):
    """
    获取VMware  Lun list
    :param context: context
    :return: Lun list
    """
    lun_lists = []
    one_lun = []
    context["command"] = "esxcli storage core device list"
    lun_info = get_return_from_txt(context)
    # 将全部回显分割为单个的disk 进行存储
    flag = False
    for index, line in enumerate(lun_info):
        if line.strip().lower().startswith("naa.") \
                or line.strip().lower().startswith("eui."):
            flag = True
        if not line.strip() and one_lun:
            lun_lists.append(one_lun)
            one_lun = []
            flag = False
            continue
        if flag:
            one_lun.append(line)
        if flag and index == len(lun_info) - 1:
            lun_lists.append(one_lun)
    return lun_lists


def get_one_empty_vm_disk():
    """
    生成一个空的disk
    :return: empty_vm_disk
    """
    disk = {}
    disk["datastore_name"] = "NA"
    disk["datastore_type"] = "NA"
    disk["compatibility_mode"] = "NA"
    disk["lun_wwn"] = ""
    disk["lun_size"] = ""
    disk["vaai_status"] = ""
    disk["vaai_plugin_name"] = ""
    disk["storage_array_type"] = ""
    disk["path_policy"] = ""
    disk["boot_device"] = ""
    disk["vendor"] = ""
    disk["product"] = ""
    disk["other_uid"] = ""
    return disk


def get_stms_multi(context):
    """
    获取solaris stms 多路径
    :param context:上下文
    :return: stms 多路径
    """
    result = {}
    try:
        multipath_type = "STMS"
        result["multipath_type"] = multipath_type
        result["version"] = "NA"
        disk_list = get_stms_disk_list(context)
        if not disk_list:
            return False, result
        result["disk_list"] = disk_list
        return True, result
    except Exception:
        context.get("Logger").error("parse error: %s" % str(traceback.format_exc()))
        return False, result


def get_stms_disk_list(context):
    """
    获取stms disk_list
    :param context: context
    :return: disk_list
    """
    stms_disks = get_stms_disks(context)
    result_list = []
    for disk_line in stms_disks:
        disk = {}
        disk_name = disk_line.split("/")[-1]
        lun_wwn = ""
        if len(disk_name) > 7:
            lun_wwn = disk_name[3:-4]
        vendor, product, size = get_stms_vps(disk_line, context)
        disk["disk_name"] = disk_name
        disk["lun_wwn"] = lun_wwn
        disk["vendor"] = vendor
        disk["product"] = product
        disk["lun_size"] = size
        disk["is_alua"] = get_stms_alua(context, disk_line)
        result_list.append(disk)
    return result_list


def get_stms_alua(context, disk_path):
    """
    获取Solaris STMS磁盘是否启用ALUA
    :param context: 上下文
    :param disk_path: 磁盘路径
    :return: yes/no
    """
    command = "mpathadm show lu " + disk_path
    context["command"] = command
    disk_detail = get_return_from_txt(context)
    for line in disk_detail:
        if "Asymmetric:" in line:
            return line[line.index(":") + 1:].strip()
    return ""


def get_stms_vps(disk_name_path, context):
    """
    solaris主机通过disk_name 获取vendor,product,size
    :param context: context
    :param disk_name_path: disk_name_path
    :return: vendor,product,size
    """
    context["command"] = "luxadm display " + disk_name_path
    vps_info = get_return_from_txt(context)
    vendor = ""
    product = ""
    size = ""
    for line in vps_info:
        if "Vendor" in line and ":" in line:
            vendor = line.split(":")[-1].strip()
            continue
        if "Product ID" in line and ":" in line:
            product = line.split(":")[-1].strip()
            continue
        if "Unformatted capacity" in line and ":" in line:
            size_unit = line.split(":")[-1]
            size_temp = size_unit.split()[0]
            unit = size_unit.split()[-1]
            size = size_temp + change_unit2stand(unit)
            continue
    return vendor, product, round(chang_unit2_gb(size), 2)


def get_stms_disks(context):
    """
    获取stms disks
    :param context: context
    :return: stms disks
    """
    stms_disk_infos = get_stms_disk_infos(context)
    stms_disks = []

    for one_disk_info in stms_disk_infos:
        disk_name = ""
        flag = False
        for line in one_disk_info:
            if line.strip().startswith("/dev/rdsk/"):
                disk_name = line.strip()
                continue
            if line.strip().startswith("Total Path Count") and ":" in line:
                flag = line.split(":")[-1].strip() != "1"
        if flag and disk_name:
            stms_disks.append(disk_name)
    return stms_disks


def get_stms_disk_infos(context):
    """
    获取stms disk infos
    :param context: context
    :return: stms disk infos
    """
    disk_lists = []
    one_disk = []
    context["command"] = "mpathadm list lu"
    disk_info = get_return_from_txt(context)
    flag = False
    for index, line in enumerate(disk_info):
        if line.strip().startswith("/"):
            flag = True
        if line.strip().startswith("/") and one_disk:
            disk_lists.append(one_disk)
            one_disk = []
        if flag:
            one_disk.append(line)
        if flag and index == len(disk_info) - 1:
            disk_lists.append(one_disk)
    return disk_lists


def get_host_name(context):
    """
    获取主机名
    :param context: 上下文
    :return: 主机名
    """
    context["command"] = "uname -a"
    name_info = get_return_from_txt(context)
    name_lines = filter(lambda name_item: name_item.strip(), name_info)
    if not name_lines:
        return ""
    return get_split_item(re.split("\\s+", name_lines[0]), 1)


def get_cluster_group(context):
    """
    获取集群的ip列表
    :param context: 上下文
    :return: ip列表
    """
    context["command"] = "olsnodes -s"
    node_name_info = get_return_from_txt(context)
    node_names = []
    for line in node_name_info:
        if "command not found" in line:
            return []
        if line.strip():
            node_names.append(line.split()[0])

    context["command"] = "cat /etc/hosts"
    hosts_info = get_return_from_txt(context)
    node_ips = []
    for node_name in node_names:
        for line in hosts_info:
            if node_name in line.split():
                node_ips.append(line.split()[0])
                break
    return node_ips


def get_vm_host_info(context):
    """
    获取VMware主机 虚拟机名称，file和OS类型
    :param context: context
    :return: 虚拟机名称，file和OS类型
    """
    vm_list = []
    context["command"] = "vim-cmd vmsvc/getallvms"
    vm_info = get_return_from_txt(context)
    flag = False
    for line in vm_info:
        if "vmid" in line.lower() and "guest os" in line.lower() and \
                "version" in line.lower():
            flag = True
            continue
        if flag and line.strip():
            vm_host = {}
            split_line = re.split("\\s\\s+", line)
            if len(split_line) < 4 or not split_line[2].startswith("["):
                continue
            vm_id = split_line[0]
            vm_host["vm_id"] = vm_id
            vm_host["vm_state"] = get_vm_state(vm_id, context)
            vm_host["vm_name"] = split_line[1]
            vm_host["vm_file"] = split_line[2]
            vm_host["vm_os_type"] = split_line[3]
            controller_list, disk_list = get_controllers_disks(vm_id, context)
            controller_disk_list = \
                get_controller_disk_list(controller_list, disk_list)
            vm_host["controller_disk_list"] = controller_disk_list
            vm_list.append(vm_host)
    if not vm_list:
        return get_empty_vm_host()
    return vm_list


def get_controller_disk_list(controller_list, disk_list):
    """
    组装controller_disk
    :param controller_list: controller_list
    :param disk_list: disk_list
    :return: controller_disk_list
    """
    controller_disk_list = []
    for disk in disk_list:
        one_controller_disk = {}
        one_controller_disk["disk_id"] = disk.get("disk_id", "")
        one_controller_disk["disk_label"] = disk.get("disk_label", "")
        one_controller_disk["disk_size"] = disk.get("disk_size", "")
        one_controller_disk["disk_path"] = disk.get("disk_path", "")
        one_controller_disk["unit_number"] = disk.get("unit_number", "")
        raw_lun_id = disk.get("raw_lun_id", "")
        if raw_lun_id:
            one_controller_disk["raw"] = "TRUE"
            one_controller_disk["compatibility_mode"] = \
                disk.get("compatibility_mode", "")
            one_controller_disk["raw_lun_id"] = raw_lun_id
        else:
            one_controller_disk["raw"] = "FALSE"
            one_controller_disk["compatibility_mode"] = "NA"
            one_controller_disk["raw_lun_id"] = "NA"

        flag = False
        for controller in controller_list:
            disk_ids = controller.get("disk_ids", [])
            if one_controller_disk["disk_id"] in disk_ids:
                one_controller_disk["controller_id"] = \
                    controller.get("controller_id", "")
                one_controller_disk["controller_type"] = \
                    controller.get("controller_type", "")
                one_controller_disk["controller_lable"] = \
                    controller.get("controller_lable", "")
                one_controller_disk["controller_share_bus"] = \
                    controller.get("controller_share_bus", "")
                flag = True
        if not flag:
            one_controller_disk["controller_id"] = ""
            one_controller_disk["controller_lable"] = ""
            one_controller_disk["controller_type"] = ""
            one_controller_disk["controller_share_bus"] = ""
        controller_disk_list.append(one_controller_disk)
    return controller_disk_list


def get_controllers_disks(vm_id, context):
    """
    获取一个虚拟机下的SCSI controllers和虚拟磁盘
    :param vm_id: vm_id
    :param context: context
    :return: SCSI controllers和虚拟磁盘
    """
    context["command"] = "vim-cmd vmsvc/device.getdevices " + vm_id
    vm_info = get_return_from_txt(context)
    controller_key_words = ["vim.vm.device.ParaVirtualSCSIController", "vim.vm.device.VirtualLsiLogicSASController",
                            "vim.vm.device.VirtualLsiLogicController", "vim.vm.device.VirtualBusLogicController"]
    disk_key_word = "vim.vm.device.VirtualDisk"
    scsi_controller_info_list = get_vm_info_by_key_words(controller_key_words, vm_info)
    disk_info_list = get_vm_info_by_key_word(disk_key_word, vm_info)
    controller_list = get_vm_controller_list(scsi_controller_info_list)
    disk_list = get_vm_disk_list(disk_info_list)
    return controller_list, disk_list


def get_vm_disk_list(disk_info_list):
    """
    解析一个vmid下的disks
    :param disk_info_list: disk_info_list
    :return: disks
    """
    disk_list = []
    for disk_info in disk_info_list:
        one_disk = {}
        for line in disk_info:
            if line.lower().strip().startswith("key ="):
                one_disk["disk_id"] = line.split("=")[1].strip()[:-1]
                continue
            if line.lower().strip().startswith("label ="):
                one_disk["disk_label"] = line.split("\"")[1]
                continue
            if line.lower().strip().startswith("summary ="):
                size = line.split("\"")[1].replace(",", "")
                one_disk["disk_size"] = round(chang_unit2_gb(size), 2)
                continue
            if line.lower().strip().startswith("filename ="):
                one_disk["disk_path"] = line.split("\"")[1]
                continue
            if line.lower().strip().startswith("unitnumber ="):
                one_disk["unit_number"] = line.split("=")[1].strip()[:-1]
                continue
            if line.lower().strip().startswith("devicename ="):
                one_disk["raw_lun_id"] = line.split("\"")[1]
                continue
            if line.lower().strip().startswith("compatibilitymode ="):
                one_disk["compatibility_mode"] = line.split("\"")[1]
                continue
        disk_list.append(one_disk)
    return disk_list


def get_vm_controller_list(scsi_controller_info_list):
    """
    解析一个vmid下的controllers
    :param scsi_controller_info_list: scsi_controller_info_list
    :return: 一个vmid下的controllers
    """
    controller_list = []
    for controller in scsi_controller_info_list:
        one_controller = {}
        disk_ids = []
        device_flag = False
        for line in controller:
            if line.lower().strip().startswith("key"):
                one_controller["controller_id"] = \
                    line.split("=")[1].strip()[:-1]
                continue
            if line.lower().strip().startswith("summary"):
                one_controller["controller_type"] = line.split("\"")[1]
                continue
            if line.lower().strip().startswith("sharedbus"):
                one_controller["controller_share_bus"] = line.split("\"")[1]
                continue
            if line.lower().strip().startswith("label ="):
                one_controller["controller_lable"] = line.split("\"")[1]
                continue
            if line.lower().strip().startswith("device = (int)"):
                device_flag = True
                continue
            if device_flag and "]," in line:
                device_flag = False
            if device_flag:
                if "," in line:
                    disk_ids.append(line.strip()[:-1])
                else:
                    disk_ids.append(line.strip())
        one_controller["disk_ids"] = disk_ids
        controller_list.append(one_controller)
    return controller_list


def get_vm_info_by_key_word(key_word, vm_info):
    """
    vm主机根据指定标志，返回回文list
    :param key_word: key_word
    :param vm_info: vm_info
    :return: 回文list
    """
    info_list = []
    one_info = []
    flag = False
    for index, line in enumerate(vm_info):
        if line.strip().startswith("(") and key_word not in line:
            flag = False
        if line.strip().startswith("(") and key_word in line:
            flag = True
        if line.strip().startswith("(") and one_info:
            info_list.append(one_info)
            one_info = []
        if flag:
            one_info.append(line)
        if flag and index == len(vm_info) - 1 and one_info:
            info_list.append(one_info)
    return info_list


def get_vm_info_by_key_words(key_words, vm_info):
    """
    vm主机根据 多个指定标志，返回回文list
    :param key_words: key_word列表
    :param vm_info: vm_info
    :return: 回文list
    """
    data = []
    for key_word in key_words:
        data.extend(get_vm_info_by_key_word(key_word, vm_info))
    return data


def get_vm_state(vmid, context):
    """
    获取vm虚拟机的状态
    :param vmid: vmid
    :param context: context
    :return: vm虚拟机的状态
    """
    context["command"] = "vim-cmd vmsvc/get.summary " + vmid
    state_info = get_return_from_txt(context)
    for line in state_info:
        if "powerstate" in line.lower() and "=" in line:
            return line.split("\"")[1]
    return ""


def get_empty_vm_host():
    """
    生成一个空的vnhost
    :return: 空的vnhost
    """
    vm_list = []
    vm_host = {}
    vm_host["vm_id"] = ""
    vm_host["vm_state"] = ""
    vm_host["vm_name"] = ""
    vm_host["vm_file"] = ""
    vm_host["vm_os_type"] = ""
    controller_disk_list = []
    controller_disk = {}
    controller_disk["disk_id"] = ""
    controller_disk["disk_label"] = ""
    controller_disk["disk_size"] = ""
    controller_disk["disk_path"] = ""
    controller_disk["unit_number"] = ""
    controller_disk["raw"] = ""
    controller_disk["compatibility_mode"] = ""
    controller_disk["raw_lun_id"] = ""
    controller_disk["controller_id"] = ""
    controller_disk["controller_lable"] = ""
    controller_disk["controller_type"] = ""
    controller_disk["controller_share_bus"] = ""
    controller_disk_list.append(controller_disk)
    vm_host["controller_disk_list"] = controller_disk_list
    vm_list.append(vm_host)
    return vm_list


def change_unit2stand(unit):
    """
    单位标准化为bytes、Kb、Mb、Gb、Tb
    :param unit: unit
    :return: unit
    """
    unit_up = unit.upper()
    if "KB" in unit_up:
        return "Kb"
    elif "MB" in unit_up:
        return "Mb"
    elif "GB" in unit_up:
        return "Gb"
    elif "TB" in unit_up:
        return "Tb"
    elif "BYTE" in unit_up:
        return "bytes"
    return ""


def get_split_item(split_items, index):
    """
    安全获取split分割的字符
    :param split_items: split_items
    :param index: index
    :return: item
    """
    if index + 1 > len(split_items):
        return ""
    else:
        return split_items[index]


def get_ret_list(lines, symbol):
    """
    通过标志进行分割
    :param lines: lines
    :param symbol: symbol
    :return: list
    """
    flag = False
    lists = []
    one = []
    for index, line in enumerate(lines):
        if symbol in line:
            flag = True
        if symbol in line and one:
            lists.append(one)
            one = []
        if flag:
            one.append(line)
        if flag and index == len(lines) - 1:
            lists.append(one)
    return lists


def get_empty_disk_list():
    """
    生成vmware多路径 空的disk_list数据
    :return: 空数据
    """
    disk_list = []
    one_disk = {}
    one_disk["datastore_name"] = ""
    one_disk["datastore_type"] = ""
    one_disk["compatibility_mode"] = ""
    one_disk["lun_size"] = ""
    one_disk["lun_wwn"] = ""
    one_disk["vaai_status"] = ""
    one_disk["vaai_plugin_name"] = ""
    one_disk["storage_array_type"] = ""
    one_disk["path_policy"] = ""
    one_disk["boot_device"] = ""
    one_disk["vendor"] = ""
    one_disk["product"] = ""
    disk_list.append(one_disk)
    return disk_list


def get_empty_overview_result(context):
    """
    空数据
    :param context: 上下文
    :return: 数据
    """
    result = {}
    result["host_ip"] = context.get("ip")
    result["host_name"] = ""
    result["os_version"] = ""
    result["hba_type"] = ""
    result["hba_wwn_list"] = ""
    result["multi_type"] = ""
    result["multi_version"] = ""
    result["cluster_name"] = ""
    result["cluster_version"] = ""
    result["cluster_ips"] = [""]
    result["lvm_type"] = ""
    result["db_name"] = ""
    result["db_version"] = ""
    result["disk_use"] = ""
    result["real_ip"] = ""
    result["multi_path_status"] = ""
    return result


def get_multi_type_version(context, host_type):
    """
    获取主机的多路径类型、版本
    :param context: 上下文
    :param host_type: 主机类型
    :return: 数据
    """
    ultra_version = get_ultra_version(context, host_type)
    if ultra_version:
        return "UltraPath", ultra_version
    power_version = get_powerpath_version(context)
    if power_version:
        return "PowerPath", power_version
    hdlm_version = get_hdlm_version(context, host_type)
    if hdlm_version:
        return "HDLM", ultra_version
    return "", ""
