#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.
import traceback
from common.contentParse import get_return_from_txt
from hosts.common.migration_parse_utils import safe_get_split
from hosts.common.migration_parse_utils import get_horizontal_cli_ret_by_head


def get_result(context):
    """
    获取gpfs集群数据
    :param context: 上下文
    :return: 数据
    """
    logger = context.get("Logger")
    try:
        result = {}
        result["host_ip"] = context.get("ip")
        cluster_name, cluster_version, cluster_tiebreaker = get_cluster_info(context)
        if not cluster_name:
            return get_empty_result(context)
        result["cluster_name"] = cluster_name
        result["cluster_version"] = cluster_version
        result["cluster_tiebreaker"] = cluster_tiebreaker
        result["cluster_repository"] = get_cluster_repository(context)
        result["node_disks"] = get_disk_list(context)
        return result
    except Exception:
        logger.error("parse error: %s" % str(traceback.format_exc()))
        return get_empty_result(context)


def get_cluster_info(context):
    """
    获取集群信息
    :param context: 上下文
    :return: 数据
    """
    cluster_name, cluster_version, cluster_tiebreaker = "", "", ""
    context["command"] = "mmlsconfig"
    cluster_info = get_return_from_txt(context)
    for line in cluster_info:
        if line.strip().startswith("clusterName") and not cluster_name:
            cluster_name = safe_get_split(line.split(), 1).strip()
        if line.strip().startswith("minReleaseLevel") and not cluster_version:
            cluster_version = safe_get_split(line.split(), 1).strip()
        if line.strip().startswith("tiebreakerDisks") and not cluster_tiebreaker:
            cluster_tiebreaker = safe_get_split(line.split(), 1).strip()
    return cluster_name, cluster_version, cluster_tiebreaker


def get_cluster_repository(context):
    """
    获取集群repository信息
    :param context: 上下文
    :return: 数据
    """
    context["command"] = "mmlscluster"
    cluster_info = get_return_from_txt(context)
    for line in cluster_info:
        if line.strip().startswith("Repository type:"):
            return safe_get_split(line.split(":"), 1).strip()
    return ""


def get_node_state(context):
    """
    获取节点/状态信息
    :param context: 上下文
    :return: 数据
    """
    context["command"] = "mmgetstate -a"
    node_info = get_return_from_txt(context, True)
    node_info_dics = get_horizontal_cli_ret_by_head("".join(node_info))
    nodes = {}
    for node in node_info_dics:
        nodes[node.get("Node name", "")] = node.get("GPFS state", "")
    return nodes


def get_node_designation(context):
    """
    获取节点/designation信息
    :param context: 上下文
    :return: 数据
    """
    context["command"] = "mmlscluster"
    flag = False
    node_info = get_return_from_txt(context)
    nodes = {}
    for line in node_info:
        if "Node" in line and "Daemon node name" in line and "Designation" in line:
            flag = True
            continue
        if flag:
            items = line.split()
            nodes[safe_get_split(items, 1)] = safe_get_split(items, 4)
    return nodes


def get_file_system(context):
    """
    获取磁盘信息
    :param context: 上下文
    :return: 数据
    """
    context["command"] = "mmlsnsd -L"
    disk_info = get_return_from_txt(context, True)
    disk_info_dics = get_horizontal_cli_ret_by_head("".join(disk_info))
    disks = {}
    for node in disk_info_dics:
        disk = {}
        disk["file_system"] = node.get("File system", "")
        disk["nsd"] = node.get("NSD servers", "")
        disks[node.get("Disk name", "")] = disk
    return disks


def get_disk_list(context):
    """
    获取集群节点信息
    :param context: 上下文
    :return: 数据
    """
    disks = []
    node_state = get_node_state(context)
    node_designation = get_node_designation(context)
    disk_file_system_nsd = get_file_system(context)

    context["command"] = "mmlsnsd -m"
    node_info = get_return_from_txt(context, True)
    node_info_dics = get_horizontal_cli_ret_by_head("".join(node_info))
    for node in node_info_dics:
        disk = {}
        node_name = node.get("Node name", "")
        disk_name = node.get("Disk name", "")
        disk["node_name"] = node_name
        disk["node_state"] = node_state.get(node_name, "")
        disk["node_designation"] = node_designation.get(node_name, "")
        disk["disk_name"] = disk_name
        disk["multi_disk_name"] = node.get("Device", "")
        disk["file_system"] = disk_file_system_nsd.get(disk_name, {}).get("file_system", "")
        disk["nsd"] = disk_file_system_nsd.get(disk_name, {}).get("nsd", "")
        disks.append(disk)
    return disks


def get_empty_result(context):
    """
    空数据
    :param context: 上下文
    :return: 数据
    """
    result = {}
    result["host_ip"] = context.get("ip")
    result["cluster_name"] = ""
    result["cluster_version"] = ""
    result["cluster_tiebreaker"] = ""
    result["cluster_repository"] = ""
    disk = {}
    disk["node_name"] = ""
    disk["node_state"] = ""
    disk["node_designation"] = ""
    disk["disk_name"] = ""
    disk["multi_disk_name"] = ""
    disk["file_system"] = ""
    disk["nsd"] = ""
    result["node_disks"] = [disk]
    return result
