# --*-- coding:utf-8 --*--
# Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.
import re
import traceback
from common.contentParse import get_return_from_txt


def get_os_type(context):
    """
    匿名数据获取操作系统类型
    :param context: 上下文
    :return: 类型
    """
    context["command"] = "wmic OS get Caption /value"
    op_info_list = get_return_from_txt(context)
    for line in op_info_list:
        if "caption" in line.lower() and "=" in line:
            return line.split("=")[-1]
    return ""


def get_version(context):
    """
    匿名数据获取操作系统版本
    :param context: 上下文
    :return: 版本
    """
    context["command"] = "wmic OS get Version /value"
    op_info_list = get_return_from_txt(context)
    for line in op_info_list:
        if "version" in line.lower() and "=" in line:
            return line.split("=")[-1]
    return ""


def get_anonymous_os_type_version(context):
    """
    匿名数据获取操作系统类型、版本（systeminfo涉及ip地址）
    :param context: 上下文
    :return: 类型、版本
    """
    logger = context.get("Logger")
    try:
        return get_os_type(context), get_version(context)
    except Exception:
        logger.error("migration error: %s" % str(traceback.format_exc()))
        return "", ""


def get_os_type_version(context):
    """
    获取操作系统类型、版本
    :param context: 上下文
    :return: 类型、版本
    """
    logger = context.get("Logger")
    try:
        context["command"] = "systeminfo"
        op_sys_info_list = get_return_from_txt(context)
        os_type = ""
        os_version = ""
        for line in op_sys_info_list:
            line = line.strip()
            if line.startswith("OS Name:") or line.startswith("OS 名称:"):
                os_type = line.split(":")[1].strip()
                continue
            if line.startswith("OS Version:") or line.startswith("OS 版本:"):
                os_version = line.split(":")[1].strip()
                continue
        if not os_type or not os_version:
            return get_anonymous_os_type_version(context)
        return os_type, os_version
    except Exception as e:
        logger.error("migration error: " + "\n".join(op_sys_info_list) + str(e) + str(traceback.format_exc()))
        return "", ""


def get_title_main_line(cli_ret_list, line_index):
    """
    电子标签表头换行，只取第一行
    :param cli_ret_list: 回文
    :param line_index: ---的行索引
    :return: 表头行
    """
    if cli_ret_list[line_index - 1].startswith(' ') and line_index > 1:
        return cli_ret_list[line_index - 2]
    return cli_ret_list[line_index - 1]


def get_horizontal_cli_ret(cli_ret):
    """
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合
    @param cli_ret: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    """
    try:
        headline = ""
        i = 0
        cli_ret_list = cli_ret.encode("utf8").splitlines()
        for line in cli_ret_list:
            reg_headline = re.compile("^\s*-+(\s+-+)*\s*$")
            match_headline = reg_headline.search(line)
            if match_headline:
                headline = match_headline.group()
                break
            i += 1
        if headline == "" or i == 0 or i >= len(cli_ret_list) - 1:
            return []
        title = get_title_main_line(cli_ret_list, i)
        field_words = cli_ret_list[(i + 1):]
        reg_split = re.compile("\s*-+\s*")
        tuple_idxs = []
        start_pos = 0
        while (start_pos <= len(headline)):
            match = reg_split.search(headline, start_pos)
            if match:
                end_pos = match.end()
                tuple_idxs.append((start_pos, end_pos))
                start_pos = end_pos
            else:
                break
        keys = []
        for item in tuple_idxs:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key.decode("utf8"))
        required_line_len = tuple_idxs[-1][0]
        dict_list = []
        for line in field_words:
            # 标题换行的场景
            if re.search("^-+(\s+-+)*\s*$", line):
                continue
            if len(line.strip()) == 0:
                continue
            if len(line) <= required_line_len:
                continue
            vals = []
            for item in tuple_idxs:
                vals.append(safe_cut_and_decode(line, item[0], item[1]))
            dict_list.append(dict(zip(keys, vals)))
        return dict_list
    except Exception as e:
        return []


def safe_cut_and_decode(line, start_idx, end_idx):
    """
    UTF-8编码下，ASCII码为1位，中文等字符占3位，在Windows CMD界面显示中占2个字符宽度
    Windows场景，原文是GB2312编码，中文占2位，转为UTF-8编码后变为3位，编码后拆分时需要调整下标位置
    start_idx 和 end_idx 是按照GB2312编码的下标位置，需同步调整为UTF-8下标位置
    """
    prefix = safe_cut(line, start_idx)
    core_sub = safe_cut(line, end_idx)
    return core_sub[len(prefix):].strip()


def safe_cut(line, length):
    new_len = length
    sub, diff = safe_decode(line[0:new_len])
    if len(sub) == length:
        return sub
    # 存在中文等占用3位的字符，解码后的子串长度加上中文字符数，应得原拆分长度
    big_char_count = (new_len - diff - len(sub)) / 2
    # 第一个条件判断是否已达到预期拆分位置，第二个条件防止进入死循环
    while len(sub) + big_char_count < length and new_len < length + big_char_count:
        new_len = length + big_char_count
        sub, diff = safe_decode(line[0:new_len])
        big_char_count = (new_len - diff - len(sub)) / 2
    return sub


def safe_decode(item):
    """
    UTF-8安全解码，解码失败可能下标将一个中文字符（3位）拆开，发现异常减少末尾位数重新解码
    """
    try:
        return item.decode("utf8"), 0
    except Exception:
        try:
            return item[0:-1].decode("utf8"), 1
        except Exception:
            return item[0:-2].decode("utf8"), 2


def check_not_found_command(line):
    """
    检查是否命令不存在
    :param line: 数据行
    :return: true-命令不存在
    """
    return "is not recognized" in line or "不是内部" in line


def get_vertical_result(content_lines, separator):
    """
    获取水平解析得到字典key-value数据
    :param content_lines: 数据行
    :param separator: 分隔符
    :return: true-命令不存在
    """
    result_dict = {}
    for line in content_lines:
        items = line.split(separator)
        if not items:
            continue
        result_dict[items[0].strip()] = separator.join(items[1:]).strip()
    return result_dict


def get_windows_physical_disks(context):
    """
    获取windows的物理磁盘
    :param context: 上下文
    :return: 数据
    """
    disks = []
    context["command"] = "powershell \"Get-PhysicalDisk | select *\""
    data_info = get_return_from_txt(context)
    one_disk = []
    start_flag = False
    for line in data_info:
        if line.strip().startswith("ClassName"):
            start_flag = True
            if one_disk:
                disks.append(one_disk)
                one_disk = []
        if start_flag:
            one_disk.append(line)
    if one_disk:
        disks.append(one_disk)
    return disks


def get_id_letter_data(context):
    """
    获取 id与盘符的映射
    :param context: 上下文
    :return: 映射数据
    """
    context["command"] = "powershell Get-Partition"
    data_info = get_return_from_txt(context)
    id_letter_data = {}
    for line in data_info:
        drive_letter = get_drive_letter(line)
        device_id = get_device_id(context, drive_letter) if drive_letter else ""
        if drive_letter and device_id:
            drive_letter_tmp = id_letter_data.get(device_id, "")
            id_letter_data[device_id] = drive_letter if not drive_letter_tmp else drive_letter_tmp + "\n" + drive_letter
    return id_letter_data


def get_device_id(context, drive_letter):
    """
    获取 盘符对应的id值
    :param context: 上下文
    :param drive_letter: 盘符
    :return: 盘符id值
    """
    context["command"] = "powershell \"Get-Partition -DriveLetter %s | select *\"" % drive_letter
    data_info = get_return_from_txt(context)
    for line in data_info:
        if line.startswith("DiskNumber") and ":" in line:
            return line.split(":")[1].strip()
    return ""


def get_drive_letter(line):
    """
    获取盘符
    :param line: 数据行
    :return: 盘符
    """
    items = line.split()
    if len(items) > 2:
        drive_letter = items[1]
        if len(drive_letter) == 1 and drive_letter.isalpha():
            return drive_letter.strip()
    return ""


def get_vxdmp_multi_type_version(context):
    """
    获取windows主机的vxdmp多路径类型、版本
    :param context: 上下文
    :return: 数据
    """
    context["command"] = "vxassist version"
    version_info = get_return_from_txt(context)
    if not "".join(version_info).strip():
        return "", ""
    for line in version_info:
        if check_not_found_command(line):
            return "", ""
    return "VXDMP", "".join(version_info).strip()
