# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.
from com.huawei.ism.tool.deploy.entity import DeployHbaInfo
from common.cliUtil import cmd_execute
from common.base_deploy_collect import BaseDeployCollect


def execute(context):
    """
    开局兼容性信息收集入口
    :param context: 传入上下文
    """
    host_info = LinuxHostInfo(context)
    host_info.get_info()


class LinuxHostInfo(BaseDeployCollect):
    def set_server_model(self):
        """
        设置服务器型号
        :return:
        """
        cmd = "dmidecode -s system-manufacturer|cat"
        dmi_flag, cli_ret_dmi = cmd_execute(
            self.cli_ret_list, self.cli, cmd, self.logger
        )

        cmd = "dmidecode -s system-product-name|cat"
        code_flag, cli_ret_code = cmd_execute(
            self.cli_ret_list, self.cli, cmd, self.logger
        )
        if not dmi_flag or not code_flag:
            self.host_info.setServerModel("error")
        else:
            server_model = "{} {}".format(
                self.get_last_line(cli_ret_dmi),
                self.get_last_line(cli_ret_code),
            )
            self.host_info.setServerModel(server_model)

    def set_bus_adapter(self):
        """
        设置HBA卡信息
        :return:
        """
        # hba卡信息汇总
        hba_info_list = list()
        # Emulex卡集合
        emulex_hba_list = list()
        # Qlogic卡集合
        qlogic_hba_list = list()
        # roce协议集合
        roce_hba_list = list()
        # 获取Emulex 卡信息
        self.get_hba_card(
            emulex_hba_list,
            "find /sys/class/scsi_host/host*/ | grep modelname | "
            "awk -F '/modelname' '{print $1}'|cat",
        )
        self.build_emulex_hba_info(emulex_hba_list, hba_info_list)
        # 获取Qlogic 卡信息
        self.get_hba_card(
            qlogic_hba_list,
            "find /sys/class/scsi_host/host*/ | grep model_name | awk"
            " -F '/model_name' '{print $1}'|cat",
        )
        self.build_qlogic_hba_info(qlogic_hba_list, hba_info_list)
        # roce协议集合
        self.get_hba_card(
            roce_hba_list,
            "lspci |grep Mellanox | awk '{print $1}'|cat",
        )

        self.build_roce_hba_info(roce_hba_list, hba_info_list)
        self.host_info.setHbaInfoList(hba_info_list)

    def build_roce_hba_info(self, roce_hba_list, hba_info_list):
        """
        构建roce协议接口卡信息
        :param roce_hba_list: roce协议接口卡名称
        :param hba_info_list: 所有接口卡对象列表
        :return:
        """

        for hba_card in roce_hba_list:
            hba_info = DeployHbaInfo()
            rece_driver_list = self.set_rece_interface_driver()
            self.set_roce_bus_adapter(hba_card, hba_info, rece_driver_list)

            hba_info_list.append(hba_info)

    def set_roce_bus_adapter(self, hba_card, hba_info, rece_driver_list):
        bus_adapter_list = list()
        cmd = "lspci |grep Mellanox|cat"
        _, cli_ret = cmd_execute(
            self.cli_ret_list, self.cli, cmd, self.logger
        )
        for line in cli_ret.splitlines():
            if hba_card in line:
                bus_adapter_list.append(line)
                break
        cmd = "lspci -vv -s {} | grep 'Part number'|cat".format(hba_card)
        _, cli_ret = cmd_execute(
            self.cli_ret_list, self.cli, cmd, self.logger
        )

        bus_adapter_list.append(self.get_cli_ret(cli_ret, cmd))

        hba_info.setHostBusAdapter("\n".join(bus_adapter_list))
        for rece_driver in rece_driver_list:
            if hba_card in rece_driver:
                hba_info.setHbaDriverVersion(rece_driver)
                hba_info.setHbaFirmwareVersion(rece_driver)
                break

    def set_rece_interface_driver(self):
        cmd = "ifconfig|grep flags| awk -F ': flags' '{print $1}'|cat"
        rece_mark_list = list()
        rece_driver_list = list()
        self.get_hba_card(rece_mark_list, cmd)
        for mark in rece_mark_list:
            cmd = "ethtool -i {}|cat".format(mark)
            _, cli_ret = cmd_execute(
                self.cli_ret_list, self.cli, cmd, self.logger
            )
            rece_driver_list.append(self.get_cli_ret(cli_ret, cmd))
        return rece_driver_list

    def set_multi_path_info(self):
        """
        设置多路径信息
        注：检查以下3种多路径：UltraPath、DM-Multipath、VxVM (DMP) 7.2.0
        命令：可能出现多种多路径模式共存，查看以下3种多路径，有返回值，
        代表有对应的多路径，
        upadm show version（--有版本(关键字Software Version)回显就是
        UltraPath）--返回字符串UltraPath
        rpm -qa |grep -i multipath--有版本(关键字device-mapper-multipath)
        回显就是DM--返回字符串DM-Multipath
        rpm -qa | grep -i VRTSaslapm--有版本(关键字VRTSaslapm)回显就是DMP，
        需要返回版本信息VxVM (DMP) 7.2.0
        :return:
        """
        multi_path_list = list()
        cmd = "upadmin show version|cat"
        flag, cli_ret = cmd_execute(
            self.cli_ret_list, self.cli, cmd, self.logger
        )
        if flag and "Software Version" in cli_ret:
            multi_path_list.append(self.ULTRA_PATH)

        cmd = "rpm -qa |grep -i multipath|cat"
        flag, cli_ret = cmd_execute(
            self.cli_ret_list, self.cli, cmd, self.logger
        )

        if flag and "device-mapper-multipath" in cli_ret:
            multi_path_list.append(self.DM_MULTI_PATH)

        cmd = "rpm -qa | grep -i VRTSaslapm|cat"
        flag, cli_ret = cmd_execute(
            self.cli_ret_list, self.cli, cmd, self.logger
        )

        if (
            flag
            and "VRTSaslapm-" in cli_ret
        ):
            multi_path_list.append(self.DMP_PATH)

        self.host_info.setMultipathingSoftware(";".join(multi_path_list))

    def set_operating_system(self):
        """
        设置操作系统信息
        :return:
        """
        operating_system_list = list()
        cmd = "cat /etc/*-release|cat"
        flag, cli_ret = cmd_execute(
            self.cli_ret_list, self.cli, cmd, self.logger
        )
        operating_system_list.append(self.get_cli_ret(cli_ret, cmd))
        cmd = "uname -ra|cat"
        flag, cli_ret = cmd_execute(
            self.cli_ret_list, self.cli, cmd, self.logger
        )
        operating_system_list.append(self.get_last_line(cli_ret))

        self.host_info.setOperatingSystem("\n".join(operating_system_list))

    def build_emulex_hba_info(self, emulex_hba_info_list, hba_info_list):
        """
        设置每个HBA的详细信息
        hostbusadapter 为主机总线适配器
        hbadriverversion 未HBA卡驱动版本
        hbafirmwareversion 为HBA卡固件版本
        :param emulex_hba_info_list:
        :param hba_info_list:
        :return:
        """
        for hba_card in emulex_hba_info_list:
            hba_info = DeployHbaInfo()
            self.set_interface_bus_adapter(hba_card, hba_info)

            self.set_emulex_interface_driver(hba_card, hba_info)

            self.set_emulex_interface_firmware(hba_card, hba_info)

            hba_info_list.append(hba_info)

    def build_qlogic_hba_info(self, qlogic_hba_info_list, hba_info_list):
        """
        设置每个qlogic HBA的详细信息
        hostbusadapter 为主机总线适配器
        hbadriverversion 未HBA卡驱动版本
        hbafirmwareversion 为HBA卡固件版本
        :param qlogic_hba_info_list:
        :param hba_info_list:
        :return:
        """
        for hba_card in qlogic_hba_info_list:
            hba_info = DeployHbaInfo()
            self.set_interface_bus_adapter(hba_card, hba_info)

            self.set_qlogic_interface_driver(hba_card, hba_info)

            self.set_qlogic_interface_firmware(hba_card, hba_info)

            hba_info_list.append(hba_info)

    def set_emulex_interface_firmware(self, hba_card, hba_info):
        """
        设置hba卡固件总线适配器
        :param hba_card:
        :param hba_info:
        :return:
        """
        cmd = "cat {}/fwrev|cat".format(hba_card)
        _, cli_ret = cmd_execute(self.cli_ret_list, self.cli, cmd, self.logger)
        hba_info.setHbaFirmwareVersion(self.get_last_line(cli_ret))

    def set_qlogic_interface_firmware(self, hba_card, hba_info):
        """
        设置hba卡固件总线适配器
        :param hba_card:
        :param hba_info:
        :return:
        """
        cmd = "cat {}/fw_version|cat".format(hba_card)
        _, cli_ret = cmd_execute(self.cli_ret_list, self.cli, cmd, self.logger)
        hba_info.setHbaFirmwareVersion(self.get_last_line(cli_ret))

    def set_qlogic_interface_driver(self, hba_card, hba_info):
        """
        设置HBA卡驱动版本
        :param hba_card:
        :param hba_info:
        :return:
        """
        cmd = "cat {}/driver_version|cat".format(hba_card)
        _, cli_ret = cmd_execute(self.cli_ret_list, self.cli, cmd, self.logger)
        hba_info.setHbaDriverVersion(self.get_last_line(cli_ret))

    def set_emulex_interface_driver(self, hba_card, hba_info):
        """
        设置HBA卡驱动版本
        :param hba_card:
        :param hba_info:
        :return:
        """
        cmd = "cat {}/lpfc_drvr_version|cat".format(hba_card)
        _, cli_ret = cmd_execute(self.cli_ret_list, self.cli, cmd, self.logger)
        hba_info.setHbaDriverVersion(self.get_last_line(cli_ret))

    def set_interface_bus_adapter(self, hba_card, hba_info):
        """
        设置HBA卡固件版本
        :param hba_card:
        :param hba_info:
        :return:
        """
        cmd = "cat {}/model*name|cat".format(hba_card)
        mode_name_flag, cli_ret_modename = cmd_execute(
            self.cli_ret_list, self.cli, cmd, self.logger
        )

        cmd = "cat {}/model*desc|cat".format(hba_card)
        flag, cli_ret_code = cmd_execute(
            self.cli_ret_list, self.cli, cmd, self.logger
        )
        if not flag or not mode_name_flag:
            hba_info.setHostBusAdapter("error")
        else:
            server_model = "{} {}".format(
                self.get_last_line(cli_ret_modename),
                self.get_last_line(cli_ret_code),
            )
            hba_info.setHostBusAdapter(server_model)
