# -*- coding: UTF-8 -*-
import re

from common.util import updateItemProgress
from common.contentParse import setCmdRet4Eval

TARGET_VERSION = "2.2.13"
QUERY_COMMAND = "zgrep 'Get plugin info' /var/log/huawei/huawei-csi-node*" \
                " | awk '{print $9}' | sort | uniq | cut -c 1-6 | cat"
# 在UpgradeEvaluation中使用的key值，两边需要同步修改
CSI_VERSION_TOO_EARLY = "linux.csi.version.too.early"
GET_CSI_VERSION_FAILED = "linux.get.csi.version.failed"
RUN_COMMAND_FAILED = "linux.run.command.failed"


def execute(context):
    checker = CSIVersionChecker(context)
    ret = checker.do_check()
    checker.write_result_info(ret)
    return ret, checker.cmd_display, checker.err_key


class CSIVersionChecker:
    def __init__(self, context):
        self.context = context
        self.ssh = None
        self.cmd_display = None
        self.logger = None
        self.command = QUERY_COMMAND
        self.target_version = TARGET_VERSION
        self.output = ""
        self.success_str = ""
        self.failed_str = ""
        self.err_key = ""
        self.init_param()

    def init_param(self):
        # 初始化工作
        self.ssh = self.context.get("SSH")
        self.cmd_display = self.context.get("ret_map")
        self.logger = self.context.get("Logger")

        # 下面的字符串有固定格式，否则InfoGrab里识别不了任务是否成功
        if self.context.get("lang") == "en":
            self.success_str = "{}:\texecute success\r\n"
            self.failed_str = "{}:\texecute failed\r\n"
        else:
            self.success_str = u"{}:\t执行成功\r\n"
            self.failed_str = u"{}:\t执行失败\r\n"

    def do_check(self):
        # 1. 获取版本字符串
        self.update_progress(10)
        if not self.get_csi_versions_by_ssh():
            return False

        # 2. 如果该主机不涉及CSI，则返回校验成功
        self.update_progress(40)
        if self.is_csi_not_involved():
            self.logger.info("The CSI is not installed on the host.")
            return True

        # 3. 校验最新的CSI版本
        self.update_progress(70)
        ret = self.check_csi_version()

        self.update_progress(99)
        return ret

    def check_csi_version(self):
        csi_version_list = self.get_csi_version_list()

        if not csi_version_list:
            self.append_error(
                "Get csi version failed.\nCommand: {0}.\nOutput: {1}.".format(
                    self.command, self.output),
                err_key=GET_CSI_VERSION_FAILED)
            return False

        for csi_version in csi_version_list:
            if self.is_version_valid(csi_version):
                return True

        self.append_error("CSI version is too early, minimum version "
                          "must be {0}.".format(self.target_version),
                          err_key=CSI_VERSION_TOO_EARLY)
        return False

    def get_csi_version_list(self):
        csi_version_list = []
        for line in self.output.split("\n"):
            line = line.strip()
            if not line:
                continue
            match_result = re.match(r"\d+\.\d+\.\d+", line)
            if not match_result:
                self.logger.info("SSH output line ignore:{}.".format(line))
                continue
            csi_version_list.append(match_result.group())

        return csi_version_list

    def is_csi_not_involved(self):
        return "No such file or directory" in self.output

    def update_progress(self, percent):
        updateItemProgress(self.context, percent)

    def is_version_valid(self, csi_version):
        subversion_list = csi_version.split(".")
        tgt_subversion_list = self.target_version.split(".")
        for index, subversion in enumerate(subversion_list):
            if int(subversion) < int(tgt_subversion_list[index]):
                return False

        return True

    def get_csi_versions_by_ssh(self):
        try:
            self.output = self.ssh.execCmdHasLog(self.command)
        except Exception as e:
            self.append_error(
                "Exception occurred.\nCommand: {0}.\nExceptions: {1}.".format(
                    self.command, "\n".join(e.args)),
                err_key=RUN_COMMAND_FAILED)
            self.set_cmd_status(False, self.command)
            return False

        if (not self.output or "TOOLKIT_EXE_CMD_FAILED" in self.output or
                "TOOLKIT_SEND_CMD_TIME_OUT" in self.output):
            self.append_error(
                "Run command failed.\nCommand is: {0}".format(self.command),
                err_key=RUN_COMMAND_FAILED)
            self.set_cmd_status(False, self.command)
            return False

        self.set_cmd_status(True, self.command)
        return True

    def set_cmd_status(self, ret, msg):
        if ret:
            self.cmd_display.put("err_msg", self.success_str.format(msg))
        else:
            self.cmd_display.put("err_msg", self.failed_str.format(msg))

    def append_error(self, msg, err_key=""):
        if len(err_key) > 0:
            self.err_key = err_key
        self.logger.error(msg)

    def write_result_info(self, check_result):
        json_object = {"check_result": str(check_result),
                       "err_msg": self.err_key,
                       "target_version": TARGET_VERSION}
        try:
            unicode_output = unicode(self.output)
        except Exception as e:
            unicode_output = str(self.output)
            self.logger.error(
                "Encode output error.\nOutput: {0}.Exception: {1}.".format(
                    unicode_output, str(e)))

        json_object["ssh_output"] = unicode_output.encode('utf-8')

        setCmdRet4Eval(self.cmd_display, str(json_object), "csi_version_grab")
