#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2019. All rights reserved.

"""
@version: SmartKit V200R006C00
@time: 2019/10/22
@file: linux_lun_wwn_info.py
@function:
@modify:
"""

import sys
import os
import re
import traceback

current_absolute_path = os.path.dirname(os.path.abspath(__file__))
path = os.path.join(current_absolute_path, "..\\..")
sys.path.append(path)
from common.contentParse import create_result_info
from common import constants
from common.util import log, HostResultInfo, HostLunInfo, \
    qryUltrapathVersion, getUltrapathIntVer, getArrayIdsManagedByUltrapath, \
    getTargetPortNumByMultipath, qryMultipathdStatus, \
    create_lun_info, get_host_wwn_info

LAUNCHER = "'cat /sys/class/fc_host/host*/port*name ' or " \
           "'cat /etc/iscsi/initiatorname.iscsi'"


def get_result_info(context):
    """
    :param context:
    :return:
    """
    host_result_info = HostResultInfo()
    host_lun_info = HostLunInfo()

    cli = context.get("SSH")

    host_lun_info.set_host_id(cli.getHost())
    cli_rets = ''

    try:
        # from launcher get initiator info
        _, cli_rets, initiator_wwn = get_host_wwn_info(context)
        if initiator_wwn:
            host_lun_info.set_initiator(initiator_wwn)
            host_result_info.set_hosts(host_lun_info)
        else:
            log.info(context, "Cannot find host initiator info, set pass.")
            create_result_info(context, "True", cli_rets, "",
                               host_result_info)
            return

        is_ultrapath_qry_succ, ultrapath_version, cli_ret = \
            qryUltrapathVersion(context)
        is_multipathd_qry_succ, is_multipathd_running, pathd_cli_ret = \
            qryMultipathdStatus(
                context)
        cli_rets += cli_ret
        cli_rets += pathd_cli_ret

        if not is_ultrapath_qry_succ or not is_multipathd_qry_succ:
            log.error(context, 'Query ultrapath version failed.')
            fail_cmd = constants.QRY_ULTRAPATH_CMD if is_ultrapath_qry_succ \
                else constants.QRY_MULTIPATHD_STATUS
            create_result_info(context, "False", cli_rets, fail_cmd,
                               host_result_info)
            return

        # Ultrapath and multipath are not installed
        if not ultrapath_version and not is_multipathd_running:
            log.info(context, 'the host uninstall ultrapath and multipathd, '
                              'set the result pass.')
            create_result_info(context, "False", cli_rets, "",
                               host_result_info)
            return

        if ultrapath_version:
            is_qry_succ, fail_cmd, cli_ret, lun_wwn = \
                query_ultrapath_lun_wwn_info(context, ultrapath_version)
            cli_rets += cli_ret
            if is_qry_succ is None:
                create_result_info(context, "True", cli_rets, "",
                                   host_result_info)
                log.info(context, 'ultrapath pass, return.')
                return
            elif is_qry_succ:
                create_lun_info(host_lun_info, lun_wwn)
                host_result_info.set_hosts(host_lun_info)
                log.info(context, 'ultrapath need to check.')
            else:
                create_result_info(context, "False", cli_rets, fail_cmd,
                                   host_result_info)
                log.info(context, 'ultrapath not pass, return.')
                return

        if is_multipathd_running:
            is_qry_succ, _, cli_ret, disk_wwn_dict, host_lun_id_dict = \
                getTargetPortNumByMultipath(context)
            cli_rets += cli_ret
            if is_qry_succ:
                if host_lun_id_dict:
                    disk_wwn_deal = host_lun_id_dict
                else:
                    disk_wwn_deal = {value: key for key, value in
                                     disk_wwn_dict.items()}
                create_lun_info(host_lun_info, disk_wwn_deal)
                host_result_info.set_hosts(host_lun_info)
            else:
                fail_cmd = constants.QRY_DISK_PTH_MANAGED_BY_MULTIPATHD_CMD
                create_result_info(context, "False", cli_rets, fail_cmd,
                                   host_result_info)
        create_result_info(context, "nocheck", cli_rets, "",
                           host_result_info)
    except Exception as e:
        log.info(context, "get host result info exception:%s:%s" % (
            str(e), str(traceback.format_exc())))
        create_result_info(context, "False", cli_rets, '',
                           host_result_info)


def query_multipath_lun_wwn_info(context):
    """
    :param context:
    :return:
    """
    return getTargetPortNumByMultipath(context)


def query_ultrapath_lun_wwn_info(context, ultrapath_version):
    """
    :param context:
    :param ultrapath_version:
    :return:
    """
    tup_ver = getUltrapathIntVer(context, ultrapath_version)
    lower_tup_ver = getUltrapathIntVer(context,
                                       constants.LINUX_LOWER_ULTRAPATH_VER)
    middle_tup_ver = getUltrapathIntVer(context,
                                        constants.LINUX_MIDDLE_ULTRAPATH_VER)
    consistency_tup_ver = \
        getUltrapathIntVer(context, constants.LINUX_CONSISTENCY_ULTRAPATH_VER)

    if tup_ver <= lower_tup_ver:
        return query_lower_version_info(
            context)
    elif tup_ver <= middle_tup_ver:
        return query_higher_version_info(
            context, True)
    elif tup_ver < consistency_tup_ver:
        return query_higher_version_info(
            context, False)
    else:
        log.info(context, 'the ultrapath version greater or equal to '
                          '8.06.034, set the result pass.')
        return None, "", "", {}


def query_lower_version_info(context):
    """
    :param context:
    :return:
    """
    cli = context.get("SSH")
    all_lun_wwn_dict = {}
    is_qry_succ, array_id_list, cli_ret = \
        getArrayIdsManagedByUltrapath(context)
    if not is_qry_succ:
        log.error(context, 'Query array managed by ultrapath failed.')
        return False, constants.QRY_ARRAY_ID_CMD, cli_ret, {}

    for array_id in array_id_list:
        try:
            array_lun_ret = cli.execCmdWithNoCheckResult(
                'upadm show lun array=' + array_id,
                constants.HOST_CMD_TIMEOUT)
            cli_ret += array_lun_ret
            is_succ, lun_wwn_dict = parse_all_lun_wwn_one_array(context,
                                                                array_lun_ret)
            all_lun_wwn_dict = dict(all_lun_wwn_dict, **lun_wwn_dict)
            log.info(context, 'lun wwns are:' + str(all_lun_wwn_dict))

        except Exception as e:
            log.error(context,
                      'Query array IDs exception:' + str(e))
            return False, 'upadm show lun array=' + array_id, cli_ret, {}

    return True, '', cli_ret, all_lun_wwn_dict


def query_higher_version_info(context, is_high_version):
    """
    :param context:
    :param is_high_version:
    :return:
    """
    cli = context.get("SSH")
    lun_wwn = {}
    cli_ret = ''
    vlu_id_cmd = constants.QRY_VLUN_ID_CMD if is_high_version \
        else constants.High_VER_QRY_VLUN_ID_CMD
    try:
        cli_ret = cli.execCmdWithNoCheckResult(vlu_id_cmd,
                                               constants.HOST_CMD_TIMEOUT)

        is_data_start = False
        for line in cli_ret.splitlines():
            if not is_data_start and re.search('Vlun ID', line,
                                               re.I) and re.search('WWN', line,
                                                                   re.I):
                is_data_start = True
                continue

            fields = line.split()
            if is_data_start and len(fields) >= 3 and fields[0].isdigit():
                lun_wwn[fields[3].strip()] = fields[0].strip()
        return True, '', cli_ret, lun_wwn
    except Exception as e:
        log.error(context, 'Query VLUN IDs exception:' + str(e))
        return False, vlu_id_cmd, cli_ret, {}


def parse_all_lun_wwn_one_array(context, array_lun_ret):
    """
    :param context:
    :param array_lun_ret:
    :return:
    """
    lun_wwn_dict = {}
    try:
        for line in array_lun_ret.splitlines():
            line = line.strip().lower()
            # find a new LUN
            if 'lun #' in line and "wwn:" in line:
                lun_id = line.split('-')[0].split('lun #')[-1].strip()
                lun_wwn = line.split('wwn:')[-1].split()[0].strip()
                lun_wwn_dict[lun_wwn] = lun_id
                continue
            elif 'information of lun' in line and "wwn:" in line:
                lun_id = line.split('-')[0].split('information of lun')[
                    -1].strip()
                lun_wwn = line.split('wwn:')[-1].split()[0].strip()
                lun_wwn_dict[lun_wwn] = lun_id
                continue
    except Exception as e:
        log.error(context, 'Query VLUN wwn exception:' + str(e))
        return False, {}
    return True, lun_wwn_dict
