# -*- coding: UTF-8 -*-
import sys
import os
import re

path = os.path.dirname(os.path.abspath(__file__))
path = os.path.join(path, "..\\..")
sys.path.append(path)
from common import util
from common import constants
from com.huawei.ism.tool.infograb.context import EvalResultEnum

ITEMKEY = "LinuxOracleCheckASMlib"
DISPLAY_KEY = 'oracle_check_asmlib'
ORACLEASM_LIST_CMD = 'oracleasm listdisks |xargs oracleasm querydisk -p'
ORACLEASM_CONFIG_CMD = 'cat /etc/sysconfig/oracleasm'
ULTRAPATH_CMD = 'upadmin show version'
LINE_BREAK = "\n"
EXECUTE_ERROR = 'linux.oracle.check.asmlib.cmd.execute.error'
CHECK_ERROR = 'linux.oracle.check.asmlib.cmd.inner.error'
UNABLE_EVAL = "eval.host.multipath.buglist.failed"
MIX_ULTRAPATH_ERROR = 'eval.host.task.compatfail.' \
                      'hbatimeout.can.not.eval.huawie.self.enabled'
NO_PATH_ERROR = 'eval.message.solaris.multipath.buildin&utrlpath.noexists'
SCANORDER = 'ORACLEASM_SCANORDER=\"mpath dm\"'
SCANEXCLUDE = 'ORACLEASM_SCANEXCLUDE=\"sd\"'
NO_CHECK = 'No Check'
CMD_KEY_WORD = "asmlib_"

def execute(context):
    return CheckLinuxOracleASMlib(context).check()


class CheckLinuxOracleASMlib:

    def __init__(self, context):
        self.context = context
        self.cli = context.get("SSH")
        self.logger = context.get("Logger")
        self.language = context.get("lang")
        self.ret_dict = context.get("ret_map")
        self.all_cli_ret = []
        self.execute_cmd_dict = {}

    def check(self):
        try:
            util.updateItemProgress(self.context, constants.PROG5)
            # 检查是否使用自研多路径
            path_flag, path_error = self.check_only_use_ultrapath()
            # 自研
            if path_flag is True:
                util.updateItemProgress(self.context, constants.PROG90)
                self.build_result_for_infograb()
                return self.build_check_result(EvalResultEnum.PASSED)
            # 混用
            if path_flag == NO_CHECK:
                util.updateItemProgress(self.context, constants.PROG90)
                self.build_result_for_infograb()
                return self.build_check_result(EvalResultEnum.FAILED,
                                               path_error)
            # 检查oracleasm list参数
            oracleasm_list_cli = self.cli.execCmdHasLog(ORACLEASM_LIST_CMD)
            self.all_cli_ret.append(oracleasm_list_cli)
            self.bulid_infograb_single_result("oracleasm_list",
                                              oracleasm_list_cli)
            util.updateItemProgress(self.context, constants.PROG50)
            # 如果没有回显，或命令不支持则通过
            if not bool(oracleasm_list_cli) or \
                    self.check_cli_executed_fail(oracleasm_list_cli):
                util.updateItemProgress(self.context, constants.PROG90)
                self.build_result_for_infograb()
                return self.build_check_result(EvalResultEnum.PASSED)
            # 否则继续检查配置文件参数
            oracleasm_config_cli = self.cli.execCmdHasLog(
                ORACLEASM_CONFIG_CMD)
            self.all_cli_ret.append(oracleasm_config_cli)
            self.bulid_infograb_single_result("oracleasm_config",
                                              oracleasm_config_cli)
            if self.check_cli_executed_fail(
                    oracleasm_config_cli) is True:
                util.updateItemProgress(self.context, constants.PROG90)
                self.build_result_for_infograb()
                return self.build_check_result(EvalResultEnum.FAILED,
                                               UNABLE_EVAL,
                                               [ORACLEASM_CONFIG_CMD])
            self.build_result_for_infograb()
            return self.check_oracleasm_param(oracleasm_config_cli)
        except Exception as exception:
            self.logger.error(
                "LinuxOracleCheckASMlib error:" + str(exception))
            util.updateItemProgress(self.context, constants.PROG90)
            return self.build_check_result(EvalResultEnum.UNABLE_EVAL,
                                           CHECK_ERROR)

    def check_oracleasm_param(self, cli_ret):
        """
        检查配置文件参数
        :param cli_ret:
        :return:
        """
        if not bool(cli_ret):
            return self.build_check_result(EvalResultEnum.FAILED,
                                           EXECUTE_ERROR,
                                           [ORACLEASM_CONFIG_CMD])
        # 均存在，通过
        if SCANORDER in cli_ret and SCANEXCLUDE in cli_ret:
            return self.build_check_result(EvalResultEnum.PASSED)
        return self.build_check_result(EvalResultEnum.FAILED, EXECUTE_ERROR,
                                       [ORACLEASM_CONFIG_CMD])

    def check_only_use_ultrapath(self):
        """
        检查主机是否只使用了自研多路径
        :return: (T/F,error)
        """
        # 自研
        use_ultrapath = self.check_host_use_ultrapath()
        # 自带
        is_qry_succ, use_mutipath, pathd_cli_ret = util.qryMultipathdStatus(
            self.context)
        self.all_cli_ret.append(pathd_cli_ret)
        self.bulid_infograb_single_result("multipath", pathd_cli_ret)
        # 不允许混合使用
        if use_ultrapath and use_mutipath:
            return NO_CHECK, MIX_ULTRAPATH_ERROR
        # 只使用自研
        if use_ultrapath is True and use_mutipath is not True:
            return True, ''
        # 只使用自带
        if use_ultrapath is not True and use_mutipath is True:
            return False, ''
        # 都未使用(应该连接异常，无法添加主机一般不会出现)
        return NO_CHECK, NO_PATH_ERROR

    def check_host_use_ultrapath(self):
        """
        检查是否使用自研多路径
        :return: T,F
        """
        cli_ret = self.cli.execCmdHasLog(ULTRAPATH_CMD)
        self.all_cli_ret.append(cli_ret)
        self.bulid_infograb_single_result("ultrapath", cli_ret)
        ver_pattern = 'Software Version|UltraPath for Linux|' \
                      'Driver   Version'
        if re.search(ver_pattern, cli_ret, re.I):
            for line in cli_ret.splitlines():
                if re.search(ver_pattern, line, re.I):
                    return True
        return False

    def check_cli_executed_fail(self, cli_ret):
        """
        检查cli是否执行失败
        :param cli_ret: CLI
        :return: T,F
        """
        if not bool(cli_ret):
            return False
        if self.check_error_cli(cli_ret) is True:
            return True
        cli_lines = cli_ret.strip().splitlines()
        # 若回显去除头尾空格后只有一行（只有命令行）
        if len(cli_lines) < 2:
            return True
        return False

    def check_error_cli(self, cli_ret):
        """
        cli是否回显命令执行失败
        :param cli_ret: cli
        :return: T/F
        """
        if 'command not found' in cli_ret \
                or 'No such file or directory' in cli_ret \
                or 'not found' in cli_ret \
                or "TOOLKIT_SEND_CMD_TIME_OUT" in cli_ret \
                or "TOOLKIT_EXE_CMD_FAILED" in cli_ret:
            return True
        return False

    def build_check_result(self, result, err_msg_key='', param=None):
        """
        构造检查项执行结果
        :param result: 执行结果
        :param err_msg_key: 异常信息key
        :param param: 异常信息参数
        :return: 检查结果
        """
        item_eval_result = util.genEvalItemObj(ITEMKEY,
                                               result,
                                               '\n'.join(self.all_cli_ret),
                                               err_msg_key, param)
        self.ret_dict["evalResult"] = item_eval_result
        return self.ret_dict

    def bulid_infograb_single_result(self, cmd_key, cli_ret):
        """
        满足infograb展示规则的单条收集结果
        :param cmd_key:
        :param cli_ret:
        :return:
        """
        self.execute_cmd_dict[CMD_KEY_WORD + cmd_key] = cli_ret

    def build_result_for_infograb(self):
        """
        为infograb收集项构造结果
        :return:
        """
        message = ""
        if not bool(self.execute_cmd_dict):
            key = "cmd_display_" + ITEMKEY
            self.ret_dict.put(key, "")
            message = key + util.get_collect_cli_result(False,
                                                        self.language)
            self.ret_dict.put("err_msg", message)
            return
        for cmd_key, cli_ret in self.execute_cmd_dict.items():
            self.ret_dict.put("cmd_display_" + cmd_key, cli_ret)
            message += cmd_key + util.get_collect_cli_result(True,
                                                             self.language)
        self.ret_dict.put("err_msg", message)
