# coding=utf-8
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2023. All rights reserved.

from hosts.sudo_utils import get_sudo_cmd, LINUX
from common.cmd_execute import DES, CMD, BaseCmdExecute

db_cmds = [
    {DES: "cmd_display_database_oracle_cluster_name", CMD: "olsnodes -s"},
    {DES: "cmd_display_database_oracle_cluster_status", CMD: "crsctl stat res -t"},
    {DES: "cmd_display_database_oracle_cluster_vote", CMD: "crsctl query css votedisk"},
    {DES: "cmd_display_database_oracle_ocr_disk_path", CMD: "ocrcheck"},
]

to_sql_cmds = [
    {DES: "cmd_display_database_sqlplus", CMD: "sqlplus / as sysdba"},
    {DES: "cmd_display_database_set_linesize", CMD: "set linesize 200"},
    {DES: "cmd_display_database_set_pagesize", CMD: "set pagesize 300"},
    {DES: "cmd_display_database_set_colpath", CMD: "col path for a40"},
    {DES: "cmd_display_database_set_colname", CMD: "col name for a30"},
]

sql_info_cmds = [
    {DES: "cmd_display_sql_cluster_ora_cversion", CMD: "select * from v$version;"},
    {DES: "cmd_display_sql_cluster_database", CMD: "select value from v$parameter where name='cluster_database';"},
    {DES: "cmd_display_sql_cluster_ora_diskgroupinfo",
     CMD: "select group_number,name,state,total_mb,free_mb,sector_size from v$asm_diskgroup;"},
    {DES: "cmd_display_sql_cluster_ora_diskinfo",
     CMD: "select group_number,disk_number,name,mount_status,"
          "header_status,path,state,total_mb,free_mb,sector_size from v$asm_disk;"},
    {DES: "cmd_display_sql_cluster_ora_asmdiskpathinfo", CMD: "show parameter asm_diskstring"},
]


def execute(context):
    """
    :param context: 上下文
    :return: 以字典格式返回命令执行结果
    """
    cmd_executor = LinuxOracleRacInfo(context)
    cmd_executor.execute()
    return cmd_executor.display


class LinuxOracleRacInfo(BaseCmdExecute):
    def __init__(self, context):
        super(LinuxOracleRacInfo, self).__init__(context)

    def execute(self):
        db_users = self.context.get("oracle.rac.user", "grid,oracle").split(",")
        user = self.switch_to_database_user(db_users)
        if not user:
            self.log.error("cannot find oracle db users: " + ",".join(db_users))
            return

        self.execute_pure_cmd_list(commands=to_sql_cmds, cur_step=50, progress_width=10)
        self.execute_pure_cmd_list(commands=sql_info_cmds, cur_step=60, progress_width=30)

    def switch_to_database_user(self, db_users):
        if not db_users:
            db_users = ["grid", "oracle"]
        for user in db_users:
            cmd = get_sudo_cmd("su - " + user, LINUX, self.context)
            su_user_ret = self.cli.execCmdHasLogTimout(cmd, 20)
            self.check_result(su_user_ret, cmd)
            self.display.put("cmd_display_database_switch_user_" + user, su_user_ret)
            check_user_ret = self.cli.execCmdHasLogTimout("whoami", 20)
            if "not exist" in su_user_ret or user not in check_user_ret:
                continue
            # 执行数据库命令，判断数据库用户是否具备权限
            self.execute_pure_cmd_list(commands=db_cmds, cur_step=5, progress_width=45)
            if self.is_db_user_has_permission():
                return user
            # 数据库用户无权限，退出到外部用户
            self.cli.execCmdHasLogTimout("exit", 10)
        self.display.put("err_msg", self.display.get("err_msg", "") + self.err_msg)
        return ''

    def is_db_user_has_permission(self):
        for _, cmd_ret in self.display.items():
            if is_db_user_no_permission(cmd_ret):
                return False
        return True


def is_db_user_no_permission(db_cmd_ret):
    return 'not-found' in db_cmd_ret or 'not found' in db_cmd_ret or 'Permission denied' in db_cmd_ret
