# -*- coding: UTF-8 -*-
import sys
import os
import re
import json

path = os.path.dirname(os.path.abspath(__file__))
path = os.path.join(path, "..\\..")
sys.path.append(path)
from common import util
from common import contentParse
from common import constants
from common.exception import CmdExecuteException
from com.huawei.ism.tool.infograb.context import EvalResultEnum

ITEM_KEY = 'LunCapacityConsistencyCheck'
ULTRAPATH_CMD = 'upadmin show version'
SHOW_DISK_CAPACITY_CMD = 'lsblk -b'
SHOW_VLUN_ULTRAPATH_CMD = 'upadmin show vlun'
SHOW_VLUN_ULTRAPATH_ALL_CMD = 'upadmin show vlun type=all'
SHOW_VLUN_MULTIPATH_CMD = 'multipath -ll|cat'
MIX_ULTRAPATH_ERROR = 'eval.host.task.compatfail.' \
                      'hbatimeout.can.not.eval.huawie.self.enabled'
MULTIPATH_CMD = 'ps -e | grep multipathd |cat'
NO_PATH_ERROR = 'eval.host.multipath.reduntpath.self&buildin.notexists'
ERROR_KEY = 'lun.capacity.consistency.check.lun.execute.error'
CMD_KEY_WORD = "capacity_consitency_"


def execute(context):
    return LunCapacityConsistencyLinux(context).execute()


class LunCapacityConsistencyLinux:

    def __init__(self, context):
        self.context = context
        self.cli = context.get("SSH")
        self.logger = context.get("Logger")
        self.language = context.get("lang")
        self.all_cli_ret = []
        self.lun_wwn_capacity_dict = {}
        self.execute_error_keys = []
        self.fail_cmds = []
        self.execute_cmd_dict = {}

    def execute(self):
        util.updateItemProgress(self.context, constants.PROG5)
        try:
            self.check_path_type_and_show_lun()
        except CmdExecuteException as cmdException:
            self.logger.error(
                "linux lun capacacity consistency cmd error" +
                str(cmdException))
        except Exception as e:
            self.logger.error(
                "execute linux lun capacacity consistency error" + str(e))
            self.execute_error_keys.append(ERROR_KEY)
        # 组装数据
        self.build_result()
        self.build_result_for_infograb()
        util.updateItemProgress(self.context, constants.PROG90)
        return self.build_execute_result()

    def check_path_type_and_show_lun(self):
        # 自研
        use_ultrapath, ultrapath_version = self.check_host_use_ultrapath()
        # 自带
        is_qry_succ, use_mutipath, pathd_cli_ret = util.qryMultipathdStatus(
            self.context)
        self.bulid_infograb_single_result("multipath", pathd_cli_ret)
        self.all_cli_ret.append(pathd_cli_ret)
        if is_qry_succ is not True:
            self.fail_cmds.append(MULTIPATH_CMD)
            raise CmdExecuteException(MULTIPATH_CMD)
        # 不允许混合使用
        if use_ultrapath and use_mutipath:
            self.execute_error_keys.append(MIX_ULTRAPATH_ERROR)
            return False
        # 都不存在
        if not use_ultrapath and not use_mutipath:
            self.execute_error_keys.append(NO_PATH_ERROR)
            return False
        # 使用自研多路径
        if use_ultrapath:
            wwn_disk_dict = self.show_lun_wwn_by_ultrapath(
                ultrapath_version)
        # 使用自带多路径
        if use_mutipath:
            wwn_disk_dict = self.show_lun_wwn_by_multipath()
        # 获取盘符容量
        disk_capacity_dict = self.get_lun_wwn_capacity()
        # 解析
        self.analysis_lun_wwn_capacity(wwn_disk_dict, disk_capacity_dict,
                                       use_ultrapath)
        return True

    def show_lun_wwn_by_ultrapath(self, ultrapath_version):
        """
        获取自研多路径时的lun盘符-wwn信息
        example:
        upadmin show vlun
        ----------------------------------------------------
        Vlun ID   Disk  Name Lun WWN  Status Capacity Ctrl(Own/Work)
           0      sda    1    (32位)  Normal  3.00GB     0A/0A
        :return:
        """
        cmd = SHOW_VLUN_ULTRAPATH_ALL_CMD
        tup_ver = util.getUltrapathIntVer(self.context, ultrapath_version)
        middle_tup_ver = util.getUltrapathIntVer(self.context,
                                                 constants.
                                                 LINUX_MIDDLE_ULTRAPATH_VER)
        if tup_ver <= middle_tup_ver:
            cmd = SHOW_VLUN_ULTRAPATH_CMD
        cli_ret = self.cli.execCmdHasLog(cmd)
        self.bulid_infograb_single_result("ultrapath_vlun_info", cli_ret)
        self.all_cli_ret.append(cli_ret)
        self.check_cli_ret_error(cli_ret, cmd)
        cli_ret_list = cli_ret.splitlines()
        wwn_disk_dict = {}
        is_data_start = False
        for line in cli_ret_list:
            if not is_data_start and re.search('Vlun ID', line, re.I):
                is_data_start = True
                continue
            fields = line.split()
            if is_data_start and len(fields) >= 4:
                # 第二位Disk
                disk = fields[1].strip()
                # 第四位Lun WWN
                wwn = fields[3].strip()
                wwn_disk_dict[wwn] = [disk]
        return wwn_disk_dict

    def show_lun_wwn_by_multipath(self):
        """
        获取自带多路径时的lun盘符-wwn信息
        example:
        muiltipath -ll
        mpathd (123456781234567812345678123456781) dm-4 HUAWEI,XSG1
        :return:
        """
        cli_ret = self.cli.execCmdHasLog(SHOW_VLUN_MULTIPATH_CMD)
        self.bulid_infograb_single_result("multipath_vlun_info", cli_ret)
        self.all_cli_ret.append(cli_ret)
        self.check_cli_ret_error(cli_ret, SHOW_VLUN_MULTIPATH_CMD)
        wwn_disk_dict = {}
        for line in cli_ret.splitlines():
            # 获取华为盘符信息
            if "dm-" in line and 'HUAWEI' in line.upper():
                if self.is_wwn_line_contains_disk_alias(line):
                    wwn, disk_names = self.parse_wwn_info_with_disk_alias(line)
                else:
                    wwn, disk_names = self.parse_wwn_info_without_disk_alias(
                        line)
                wwn_disk_dict[wwn] = disk_names
        return wwn_disk_dict

    def is_wwn_line_contains_disk_alias(self, line):
        return "(" in line and ")" in line

    def parse_wwn_info_with_disk_alias(self, line):
        """
        有盘符别名的解析方式
        :param line: 解析的行
        :return: wwn, [盘符别名，盘符名]
        """
        # 截取括号中后32位内值为WWN
        patten = u"[(](.*?)[)]"
        result = re.findall(patten, line)
        wwn = result[0]
        length = len(wwn)
        if length > 32:
            wwn = wwn[length - 32:]
        disk_alias = line.split()[0]
        return wwn, [disk_alias, self.parse_disk_dm_name(line)]

    def parse_wwn_info_without_disk_alias(self, line):
        """
        没有盘符别名的解析方式
        :param line: 解析的行
        :return: wwn, [盘符名]
        """
        wwn = line.split()[0]
        length = len(wwn)
        if length > 32:
            wwn = wwn[length - 32:]
        return wwn, [self.parse_disk_dm_name(line)]

    def parse_disk_dm_name(self, line):
        dm_start_index = line.find("dm-")
        return line[dm_start_index:].split()[0]

    def get_lun_wwn_capacity(self):
        """
        获取lun盘符-容量
        example:
        lsblk -b
        NAME  MAJ:MIN  RM       SIZE RO MOUNTPOINT
        sda     8:0     0    1234567  0   /
        :return:
        """
        disk_capacity_dict = {}
        cli_ret = self.cli.execCmdHasLog(SHOW_DISK_CAPACITY_CMD)
        self.bulid_infograb_single_result("vlun_capacity", cli_ret)
        self.all_cli_ret.append(cli_ret)
        # 对齐方案，若该命令无法执行，则不继续检查
        if "command not found" in cli_ret \
                or "No such file or directory" in cli_ret:
            return disk_capacity_dict
        maj = 'MAJ:MIN'
        rm = 'RM'
        size = 'SIZE'
        for line in cli_ret.splitlines():
            if maj in line and rm in line and size in line:
                name_end = line.index(maj)
                size_start = line.index(rm) + len(rm)
                size_end = line.index(size) + len(size)
                break
        for line in cli_ret.splitlines():
            disk = line[0:name_end].strip()
            size = line[size_start:size_end].strip()
            disk_capacity_dict[disk] = size
        return disk_capacity_dict

    def analysis_lun_wwn_capacity(self, wwn_disk_dict, disk_capacity_dict,
                                  use_ultrapath):
        """
        解析
        :param wwn_disk_dict:
        :param disk_capacity_dict:
        :param use_ultrapath:
        :return:
        """
        if not bool(wwn_disk_dict) or not bool(disk_capacity_dict):
            return
        for wwn, disk_names in wwn_disk_dict.items():
            # 自研究需要完全匹配磁盘sda,sdb...
            disk_capacity = '0'
            if use_ultrapath:
                disk_capacity = disk_capacity_dict.get(disk_names[0], '0')
            else:
                # 自带磁盘中包含查出磁盘名dm-x均可表示
                for disk_name_cp, capacity in disk_capacity_dict.items():
                    if self.is_disk_names_in_disk_name_cp(
                            disk_names, disk_name_cp):
                        disk_capacity = capacity
                        break
            self.lun_wwn_capacity_dict[wwn] = disk_capacity

    def is_disk_names_in_disk_name_cp(self, disk_names, disk_name_cp):
        # disk_name_cp为--mpath (dm-1)场景
        disk_name_cps = disk_name_cp.replace("(", "").replace(")", "").split()
        for disk_name in disk_names:
            for name in disk_name_cps:
                index = name.find(disk_name)
                if index != -1 and name[index:] == disk_name:
                    return True
        return False

    def check_host_use_ultrapath(self):
        """
        检查是否使用自研多路径
        :return: T,F
        """
        cli_ret = self.cli.execCmdHasLog(ULTRAPATH_CMD)
        self.bulid_infograb_single_result("ultrapath_version", cli_ret)
        self.all_cli_ret.append(cli_ret)
        ver_pattern = u'Software Version|UltraPath ' \
                      u'for Linux|Driver\\s+Version'
        if re.search(ver_pattern, cli_ret, re.I):
            for line in cli_ret.splitlines():
                if re.search(ver_pattern, line, re.I):
                    return True, line.split(':')[1].strip()
        return False, ''

    def build_result(self):
        """
        组装查询结果
        :return:
        """
        ret_map = self.context.get("ret_map")
        luns_in_precise_mode = {"cli_rets": '\n'.join(self.all_cli_ret),
                                "execute_error": self.execute_error_keys,
                                "fail_cmds": ",".join(self.fail_cmds),
                                "lun_wwn_capacitys":
                                    self.lun_wwn_capacity_dict}
        self.logger.info(
            'linux_luns_in_precise_mode:' + str(luns_in_precise_mode))
        contentParse.setCmdRet4Eval(ret_map,
                                    json.dumps(luns_in_precise_mode),
                                    "HostLunWWNCapacsityInfo")

    def check_cli_ret_error(self, cli_ret, cmd):
        """
        检查异常回显
        :param cmd:
        :param cli_ret:
        :return:
        """
        if 'not found' in cli_ret or "command not found" in cli_ret\
                or "No such file or directory" in cli_ret \
                or "TOOLKIT_SEND_CMD_TIME_OUT" in cli_ret \
                or "TOOLKIT_EXE_CMD_FAILED" in cli_ret:
            self.fail_cmds.append(cmd)
            raise CmdExecuteException(cmd, cli_ret)

    def build_execute_result(self):
        """
        收集项结果
        :return:
        """
        ret_dict = self.context.get("ret_map")
        item_eval_result = util.genEvalItemObj(ITEM_KEY,
                                               EvalResultEnum.PASSED,
                                               '\n'.join(self.all_cli_ret),
                                               '')
        ret_dict['evalResult'] = item_eval_result
        return ret_dict

    def bulid_infograb_single_result(self, cmd_key, cli_ret):
        """
        满足infograb展示规则的单条收集结果
        :param cmd_key:
        :param cli_ret:
        :return:
        """
        self.execute_cmd_dict[CMD_KEY_WORD + cmd_key] = cli_ret

    def build_result_for_infograb(self):
        """
        为infograb收集项构造结果
        :return:
        """
        ret_dict = self.context.get("ret_map")
        message = ""
        if not bool(self.execute_cmd_dict):
            key = "cmd_display_" + ITEM_KEY
            ret_dict.put(key, "")
            message = key + util.get_collect_cli_result(False,
                                                        self.language)
            ret_dict.put("err_msg", message)
            return
        for cmd_key, cli_ret in self.execute_cmd_dict.items():
            ret_dict.put("cmd_display_" + cmd_key, cli_ret)
            message += cmd_key + util.get_collect_cli_result(True,
                                                             self.language)
        ret_dict.put("err_msg", message)
